from datetime import date

from sandbox.sandboxsdk import environments

from dateutil.relativedelta import relativedelta
from sandbox import sdk2
from sandbox.projects.cloud.billing.common.task import AutoBinaryTask

TABLE_PATH_PREFIX = "//home/cloud/billing"


class AggregateBalanceReports(AutoBinaryTask):
    """ Task to aggregate balance reports """

    class Parameters(AutoBinaryTask.Parameters):
        yt_cluster = sdk2.parameters.String(
            'Destination YT cluster',
            default='hahn',
            required=True
        )
        yql_token_name = sdk2.parameters.String(
            'YQL Token secret name',
            required=True
        )
        env = sdk2.parameters.StrictString(
            'Environment (preprod/prod)',
            regexp="^preprod|prod$",
            required=True
        )
        date_from = sdk2.parameters.StrictString(
            "Date YYYY-MM-DD, or 1st of previous month will be taken",
            regexp="^(\\d{4}-\\d{2}-\\d{2})?$",
            required=False,
        )
        date_to = sdk2.parameters.StrictString(
            "Date YYYY-MM-DD, or today will be taken",
            regexp="^(\\d{4}-\\d{2}-\\d{2})?$",
            required=False,
        )

    class Requirements(AutoBinaryTask.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('yql')
        )

    def on_enqueue(self):
        if not self.Parameters.date_from:
            date_from = (date.today() - relativedelta(months=1)).replace(day=1)
            self.Parameters.date_from = date_from.strftime("%Y-%m-%d")

        if not self.Parameters.date_to:
            self.Parameters.date_to = date.today().strftime("%Y-%m-%d")

    def on_execute(self):
        from yql.api.v1.client import YqlClient
        import yt.wrapper as ytw
        from cloud.billing.utils.scripts.aggregate_balance_reports import aggregate_balance_reports
        yql_token = sdk2.Vault.data(self.owner, self.Parameters.yql_token_name)
        yt_proxy = self.Parameters.yt_cluster
        yql_client = YqlClient(db=yt_proxy, token=yql_token)

        ytw.config['token'] = yql_token
        ytw.config['proxy']['url'] = yt_proxy

        aggregate_balance_reports(
            table_path_prefix=TABLE_PATH_PREFIX,
            date_from=self.Parameters.date_from,
            date_to=self.Parameters.date_to,
            env=self.Parameters.env,
            yql_client=yql_client,
        )
