from sandbox import sdk2
from sandbox.projects.cloud.billing.common.task import SmallTask
from sandbox.sandboxsdk import environments
from sandbox.projects.common import binary_task


class AssertNoNewRows(binary_task.LastBinaryTaskRelease, SmallTask):
    """ Task to check that there are no new rows in YT node """

    class Parameters(SmallTask.Parameters):
        juggler_host_for_result = sdk2.parameters.String(
            'Host Name in Juggler for monitoring result',
            default='',
            required=True
        )
        juggler_service_name_for_result = sdk2.parameters.String(
            'Service name in Juggler for monitoring result',
            default='',
            required=True
        )
        yt_cluster = sdk2.parameters.String(
            'Destination YT cluster',
            default='hahn',
            required=True
        )
        yql_token_name = sdk2.parameters.String(
            'YQL Token secret name',
            required=True
        )
        src_path = sdk2.parameters.String(
            'YT Path to source directory/table',
            required=True
        )
        is_directory = sdk2.parameters.Bool(
            'Pick if source is directory',
            required=True
        )
        time_field = sdk2.parameters.String(
            'Time_field. Field with timestamp to detect new rows'
        )
        window_size = sdk2.parameters.String(
            'Window_size (seconds). Find rows WHERE field_name >= now - window_size'
        )
        ext_params = binary_task.binary_release_parameters(stable=True)

    class Requirements(SmallTask.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('yql')
        )

    def on_execute(self):
        super(AssertNoNewRows, self).on_execute()

        from cloud.billing.utils.scripts.run_query import run_query
        from yql.api.v1.client import YqlClient
        import yt.wrapper as ytw

        yql_token = sdk2.Vault.data(self.owner, self.Parameters.yql_token_name)
        yt_proxy = self.Parameters.yt_cluster

        yql_client = YqlClient(db=yt_proxy, token=yql_token)

        ytw.config['token'] = yql_token
        ytw.config['proxy']['url'] = yt_proxy

        query = '$src_path = "{src_path}";\n' \
                '$time_field = "{time_field}";\n' \
                '$now = DateTime::ToSeconds(CurrentUtcTimestamp());\n' \
                '$window_size = {window_size};\n'
        if self.Parameters.is_directory:
            query += '$paths = (SELECT AGGREGATE_LIST(Path) FROM FOLDER($src_path));\n' \
                     '$rows = (SELECT * FROM EACH($paths));\n'
        else:
            query += '$rows = (SELECT * FROM $src_path);\n'

        query += 'SELECT * FROM $rows WHERE TableRow().$time_field > $now - $window_size order by TableRow().$time_field desc;'

        query = query.format(
            src_path=self.Parameters.src_path,
            time_field=self.Parameters.time_field,
            window_size=self.Parameters.window_size,
        )
        result, share_url = run_query(
            query=query,
            yql_client=yql_client,
            yt_wrapper=ytw
        )

        neok_description = "Query result is not empty!. {} new rows. Operation: {}".format(len(result), share_url)
        ok_description = "Query result is empty. OK. Operation: {}".format(share_url)
        if len(result) == 0:
            self.send_event_to_juggler('OK', self.Parameters.juggler_host_for_result,
                                       self.Parameters.juggler_service_name_for_result, ok_description)
        else:
            self.send_event_to_juggler('CRIT', self.Parameters.juggler_host_for_result,
                                       self.Parameters.juggler_service_name_for_result, neok_description)
