import logging

from sandbox import sdk2
from sandbox.common.types import task as types_task
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import errors

from sandbox.projects.cloud.billing.apply_query_to_table.source import ApplyQueryToTable
from sandbox.projects.cloud.billing.build_analytics_cube.source.params import CheckQueryResultIsEmptyParams
from sandbox.projects.cloud.billing.build_analytics_cube.source.utils import get_name
from sandbox.projects.cloud.billing.build_analytics_cube.source.utils import get_parameters_group_values
from sandbox.projects.cloud.billing.check_query_result_is_empty.source import CheckQueryResultIsEmpty
from sandbox.projects.cloud.billing.common.task import AutoBinaryTask
from sandbox.projects.cloud.billing.common.task import Task
from sandbox.projects.cloud.billing.yt_copy_table.source import YtCopyTable

CHECK_VALUE_SETS_ARE_SUBSET = 'value_sets_are_subset'
CHECK_VALUE_SETS_ARE_EQUAL = 'value_sets_are_equal'
CHECK_NULL_SEGMENT = 'null_segment'
CHECK_MANAGED_ACCOUNTS = 'managed_accounts'
CHECK_CURRENCY_QUOTES = 'currency_quotes'

CHECKS = (CHECK_VALUE_SETS_ARE_SUBSET, CHECK_VALUE_SETS_ARE_EQUAL,
          CHECK_NULL_SEGMENT, CHECK_MANAGED_ACCOUNTS, CHECK_CURRENCY_QUOTES)

LABEL_SUBSTS_BY_CHECK = {
    CHECK_VALUE_SETS_ARE_SUBSET: {"check_label": "Value sets are subset"},
    CHECK_VALUE_SETS_ARE_EQUAL: {"check_label": "Value sets are equal"},
    CHECK_NULL_SEGMENT: {"check_label": "Null segment"},
    CHECK_MANAGED_ACCOUNTS: {"check_label": "Managed accounts"},
    CHECK_CURRENCY_QUOTES: {"check_label": "Currency quotes"},
}

TASK_TYPE_BY_CHECK = {
    CHECK_VALUE_SETS_ARE_SUBSET: CheckQueryResultIsEmpty,
    CHECK_VALUE_SETS_ARE_EQUAL: CheckQueryResultIsEmpty,
    CHECK_NULL_SEGMENT: CheckQueryResultIsEmpty,
    CHECK_MANAGED_ACCOUNTS: CheckQueryResultIsEmpty,
    CHECK_CURRENCY_QUOTES: CheckQueryResultIsEmpty,
}


class BuildAnalyticsCube(AutoBinaryTask):
    TEMPORARY_CUBE_GENERATION_STEP = 'generate_temporary_cube'
    MAIN_COPY_STEP = 'main_copy_step'
    CHECKS_STEP = 'checks_step'

    class Parameters(AutoBinaryTask.Parameters):
        with sdk2.parameters.Group('Common parameters') as common_block:
            yt_cluster = sdk2.parameters.String(
                'Destination YT cluster',
                default='hahn',
                required=True
            )
            yql_token_name = sdk2.parameters.String(
                'YQL Token secret name',
                required=True
            )
            yt_token_name = sdk2.parameters.String(
                'YT Token secret name',
                required=True,
            )

        with sdk2.parameters.Group('Temporary cube generation task') as generation_block:
            generate_temporary_cube_step = sdk2.parameters.Bool('Generate temporary cube step enabled', default=True)
            with generate_temporary_cube_step.value[True]:
                accessory_directory_path = sdk2.parameters.String(
                    'Path from /arcadia to directory with cube query and table paths.',
                    required=True
                )
                table_paths_file_name = sdk2.parameters.String(
                    'File name with table paths',
                    required=True
                )
                query_file_name = sdk2.parameters.String(
                    'File name with query',
                    required=True
                )
                outputs = sdk2.parameters.String(
                    'Outputs',
                    description='"QUERY_VAR1:TABLE_PATH1,QUERY_VAR2:TABLE_PATH2..." - which variables where insert to',
                    default='result://home/cloud/billing/...'
                )
                outputs_primary_keys = sdk2.parameters.Dict(
                    'Outputs primary keys',
                    description='result:key1,key2',
                )
                arcadia_revision = sdk2.parameters.String(
                    'Arcadia revision number',
                    required=True
                )

        with sdk2.parameters.Group('Checks') as check_block:
            checks_step = sdk2.parameters.Bool('Checks step enabled', default=True)
            with checks_step.value[True]:
                value_sets_are_subset = CheckQueryResultIsEmptyParams(
                    prefix=CHECK_VALUE_SETS_ARE_SUBSET + '_',
                    label_substs=LABEL_SUBSTS_BY_CHECK[CHECK_VALUE_SETS_ARE_SUBSET]
                )
                value_sets_are_equal = CheckQueryResultIsEmptyParams(
                    prefix=CHECK_VALUE_SETS_ARE_EQUAL + '_',
                    label_substs=LABEL_SUBSTS_BY_CHECK[CHECK_VALUE_SETS_ARE_EQUAL]
                )
                null_segment = CheckQueryResultIsEmptyParams(
                    prefix=CHECK_NULL_SEGMENT + '_',
                    label_substs=LABEL_SUBSTS_BY_CHECK[CHECK_NULL_SEGMENT]
                )
                managed_accounts = CheckQueryResultIsEmptyParams(
                    prefix=CHECK_MANAGED_ACCOUNTS + '_',
                    label_substs=LABEL_SUBSTS_BY_CHECK[CHECK_MANAGED_ACCOUNTS]
                )
                currency_quotes = CheckQueryResultIsEmptyParams(
                    prefix=CHECK_CURRENCY_QUOTES + '_',
                    label_substs=LABEL_SUBSTS_BY_CHECK[CHECK_CURRENCY_QUOTES]
                )

        with sdk2.parameters.Group('Completion') as completion_block:
            main_copy_step = sdk2.parameters.Bool('Copy step enabled', default=True)
            with main_copy_step.value[True]:
                temporary_to_main_paths = sdk2.parameters.Dict(
                    'Paths to copy',
                    description='TMP_SRC1:MAIN_DST1,MAIN_DST1;TMP_SRC2:MAIN_DST2;...',
                    required=True
                )

        with sdk2.parameters.Output:
            output = sdk2.parameters.String('Output')

    class Requirements(AutoBinaryTask.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
        )

    def on_failure(self, prev_status):
        super(Task, self).on_failure(prev_status)  # skip Task.on_failure

        description = 'Scheduled Job Failed. {}.\n{}'.format(self.sandbox_url, self.Parameters.output or "")

        logging.error(description)

        self.send_event_to_juggler('CRIT', description=description)

    def on_execute(self):
        super(BuildAnalyticsCube, self).on_execute()

        step = self._start_temporary_cube_generation()
        self._check_step_completed(step=step)

        step = self._run_checks()
        self._check_step_completed(step=step)

        step = self._start_copying_to_main()
        self._check_step_completed(step=step)

    def _start_temporary_cube_generation(self):
        with self.memoize_stage[self.TEMPORARY_CUBE_GENERATION_STEP]:
            logging.info('Starting generation of temporary cube...')

            if not self.Parameters.generate_temporary_cube_step:
                logging.info('Skip generation of temporary cube!')
                return self.TEMPORARY_CUBE_GENERATION_STEP

            task = ApplyQueryToTable(
                self,
                yt_cluster=self.Parameters.yt_cluster,
                yql_token_name=self.Parameters.yql_token_name,
                directory_path=self.Parameters.accessory_directory_path,
                table_paths_file_name=self.Parameters.table_paths_file_name,
                query_file_name=self.Parameters.query_file_name,
                outputs=self.Parameters.outputs,
                outputs_primary_keys=self.Parameters.outputs_primary_keys,
                arcadia_revision=self.Parameters.arcadia_revision,
                kill_timeout=self.Parameters.kill_timeout,
                tags=self.Parameters.tags,
            )

            self.run_subtasks([task], step=self.TEMPORARY_CUBE_GENERATION_STEP)

        return self.TEMPORARY_CUBE_GENERATION_STEP

    def _run_checks(self):
        with self.memoize_stage[self.CHECKS_STEP]:
            logging.info('Starting checks...')

            if not self.Parameters.checks_step:
                logging.info('Skip checks!')
                return self.CHECKS_STEP

            tasks = []
            yt_cluster = self.Parameters.yt_cluster
            yql_token_name = self.Parameters.yql_token_name

            for check_name, task_type in TASK_TYPE_BY_CHECK.iteritems():
                group_name = get_name('is_enabled', prefix=check_name)
                if not getattr(self.Parameters, group_name):
                    logging.info('Skipping %s check...', check_name)
                    continue

                tasks.append(
                    task_type(
                        self,
                        description=check_name,

                        yt_cluster=yt_cluster,
                        yql_token_name=yql_token_name,
                        tags=self.Parameters.tags,
                        **get_parameters_group_values(task=self, prefix=check_name)
                    ),
                )

            self.run_subtasks(tasks, step=self.CHECKS_STEP)

        return self.CHECKS_STEP

    def _start_copying_to_main(self):
        with self.memoize_stage[self.MAIN_COPY_STEP]:
            logging.info('Starting copying temporary cube to main...')

            if not self.Parameters.main_copy_step:
                logging.info('Skip copying temporary cube to main!')
                return self.MAIN_COPY_STEP

            tasks = []

            for temporary_path, main_paths in self.Parameters.temporary_to_main_paths.items():
                for main_path in main_paths.split(','):
                    tasks.append(YtCopyTable(
                        self,
                        description="{} to {}".format(temporary_path, main_path),

                        yt_cluster=self.Parameters.yt_cluster,
                        yt_token_name=self.Parameters.yt_token_name,
                        src_table_path=temporary_path.strip(),
                        dst_table_path=main_path.strip(),
                        recursive=True,
                        ignore_existing=False,
                        force=True,
                        tags=self.Parameters.tags,
                    ))

            self.run_subtasks(tasks, step=self.MAIN_COPY_STEP)

        return self.MAIN_COPY_STEP

    def _check_step_completed(self, step):
        with self.memoize_stage['check_' + step]:
            logging.info('Checking %s step...', step)

            self.check_tasks_succeed(task_ids=self.subtask_id_by_step.get(step, []))

            logging.info('Step %s was successful', step)

    def check_tasks_succeed(self, task_ids=None, tasks=None):
        tasks = tasks or []

        for task_id in (task_ids or []):
            tasks.append(sdk2.Task[task_id])

        errs = []
        for task in tasks:
            if task.status not in types_task.Status.Group.SUCCEED:
                sandbox_url = self._get_sandbox_url(task)
                errs.append('Subtask failed. {}. {}'.format(sandbox_url, getattr(task.Parameters, 'output', '')))

        if errs:
            self.Parameters.output = '\n'.join(errs)
            raise errors.SandboxTaskFailureError('Not all subtasks were successful.\n{}'.format(self.Parameters.output))
