from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import errors

from sandbox.projects.cloud.billing.common.task import AutoBinaryTask


class CheckQueryResultIsEmpty(AutoBinaryTask):
    """ Task to run YQL query and assert result is empty """

    class Parameters(AutoBinaryTask.Parameters):
        juggler_host_for_result = sdk2.parameters.String(
            'Host Name in Juggler for result monitoring',
            default='',
            required=True
        )
        juggler_service_name_for_result = sdk2.parameters.String(
            'Service name in Juggler for result monitoring',
            default='',
            required=True
        )
        yt_cluster = sdk2.parameters.String(
            'Destination YT cluster',
            default='hahn',
            required=True
        )
        yql_token_name = sdk2.parameters.String(
            'YQL Token secret name',
            required=True
        )
        query_kwargs = sdk2.parameters.String(
            'kwarg1:value1,kwarg2:value2,...'
        )
        arcadia_query_path = sdk2.parameters.String(
            'Svn url to file with query.',
            required=False
        )
        arcadia_revision = sdk2.parameters.String(
            'Arcadia revision number',
            required=True
        )
        raise_on_error = sdk2.parameters.Bool(
            'Raise if result not empty',
            description='Raise error if result is not empty. Otherwise only send notification',
            default=False,
        )

        with sdk2.parameters.Output:
            output = sdk2.parameters.String('Output')

    class Requirements(AutoBinaryTask.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
            environments.PipEnvironment('yql')
        )

    def on_execute(self):
        from cloud.billing.utils.scripts.run_query import run_query
        from yql.api.v1.client import YqlClient
        import yt.wrapper as ytw

        yql_token = sdk2.Vault.data(self.owner, self.Parameters.yql_token_name)
        yt_proxy = self.Parameters.yt_cluster

        query = sdk2.svn.Arcadia.cat(self.Parameters.arcadia_query_path, self.Parameters.arcadia_revision)

        query_kwargs = self.Parameters.query_kwargs
        kwargs = {item.split(':')[0]: item.split(':')[1] for item in query_kwargs.split(',')}
        query = query.format(**kwargs)

        yql_client = YqlClient(db=yt_proxy, token=yql_token)

        ytw.config['token'] = yql_token
        ytw.config['proxy']['url'] = yt_proxy
        result, share_url = run_query(
            query=query,
            yql_client=yql_client,
            yt_wrapper=ytw
        )

        self._check_result(result, share_url)

    def _check_result(self, result, share_url):
        if result:
            self._on_result(share_url)
        else:
            self._on_empty_result(share_url)

    def _on_empty_result(self, share_url):
        description = 'Query result is empty. OK. Operation: {}'.format(share_url)

        self.Parameters.output = description

        self.send_event_to_juggler('OK', self.Parameters.juggler_host_for_result,
                                   self.Parameters.juggler_service_name_for_result, description)

    def _on_result(self, share_url):
        description = 'Query result is not empty!. CRIT. Operation: {}'.format(share_url)

        self.Parameters.output = description

        self.send_event_to_juggler('CRIT', self.Parameters.juggler_host_for_result,
                                   self.Parameters.juggler_service_name_for_result, description)

        if self.Parameters.raise_on_error:
            raise errors.SandboxTaskFailureError(description)
