from sandbox import sdk2
from sandbox.common.types import misc as types_misc
from sandbox.common.types import task as types_task
from sandbox.projects.cloud.analytics.common.analytics_task import AnalyticsTask
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk import errors

SUBTASK_WAIT_STATUSES = (
    types_task.Status.Group.FINISH,
    types_task.Status.Group.BREAK
)


class Task(AnalyticsTask):
    class Parameters(AnalyticsTask.Parameters):
        juggler_host = sdk2.parameters.String(
            'Host Name in Juggler to check task execution success',
            default='cloud-billing-scheduled-sandbox-jobs',
            required=False
        )
        juggler_service_name = sdk2.parameters.String(
            'Service name in Juggler to check task execution success',
            default='',
            required=False
        )

    def on_failure(self, prev_status):
        super(Task, self).on_failure(prev_status)
        self.send_event_to_juggler('CRIT')

    def on_break(self, prev_status, status):
        super(Task, self).on_break(prev_status, status)
        self.send_event_to_juggler('CRIT')

    @property
    def subtask_id_by_step(self):
        if self.Context.subtask_id_by_step is types_misc.NotExists:
            self.Context.subtask_id_by_step = {}
        return self.Context.subtask_id_by_step

    def run_subtasks(self, tasks, step='', wait_statuses=SUBTASK_WAIT_STATUSES, wait_all=True, timeout=None):
        for task in tasks:
            task.enqueue()

        self.subtask_id_by_step.setdefault(step, []).extend(task.id for task in tasks)

        if wait_statuses:
            raise sdk2.WaitTask(tasks, statuses=wait_statuses, wait_all=wait_all, timeout=timeout)

    def get_subtasks(self, step=''):
        return [sdk2.Task[task_id] for task_id in self.subtask_id_by_step.get(step, [])]

    @staticmethod
    def are_tasks_succeed(task_ids=None, tasks=None):
        for task_id in (task_ids or []):
            if sdk2.Task[task_id].status not in types_task.Status.Group.SUCCEED:
                return False

        for task in (tasks or []):
            if task.status not in types_task.Status.Group.SUCCEED:
                return False

        return True

    def check_tasks_succeed(self, task_ids=None, tasks=None):
        if not self.are_tasks_succeed(task_ids=task_ids, tasks=tasks):
            raise errors.SandboxTaskFailureError('Not all subtasks were successful')


class SmallTask(Task):
    class Requirements(Task.Requirements):
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(Task.Requirements.Caches):
            pass  # means that task do not use any shared caches


class AutoBinaryTask(binary_task.LastBinaryTaskRelease, SmallTask):
    class Parameters(SmallTask.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
