import logging

from sandbox import sdk2
from sandbox.projects.cloud.billing.common.task import SmallTask
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk import environments


class YtCopyTable(binary_task.LastBinaryTaskRelease, SmallTask):
    """ Copy source node to destination """

    class Parameters(SmallTask.Parameters):
        yt_cluster = sdk2.parameters.String(
            'YT cluster',
            default='hahn',
            required=True,
        )
        yt_token_name = sdk2.parameters.String(
            'YT Token secret name',
            required=True,
        )

        src_table_path = sdk2.parameters.String(
            'YT Path for source table',
            required=True,
        )
        dst_table_path = sdk2.parameters.String(
            'YT Path for destination table',
            required=True,
        )

        recursive = sdk2.parameters.Bool(
            'Create missing intermediate nodes',
            default=True,
        )
        ignore_existing = sdk2.parameters.Bool(
            'Do not recreate existing nodes',
            default=True,
        )
        force = sdk2.parameters.Bool(
            'If destination exists, it will be deleted and a new one created',
            default=False,
        )

        ext_params = binary_task.binary_release_parameters(stable=True)

    class Requirements(SmallTask.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
        )

    def _get_yt_client(self):
        import yt.wrapper as yt

        yt_proxy = self.Parameters.yt_cluster
        yt_token_name = self.Parameters.yt_token_name

        yt.config['proxy']['url'] = yt_proxy
        yt.config['token'] = sdk2.Vault.data(self.owner, yt_token_name)

        return yt

    def on_execute(self):
        super(YtCopyTable, self).on_execute()

        src_table_path = self.Parameters.src_table_path
        dst_table_path = self.Parameters.dst_table_path
        recursive = self.Parameters.recursive
        ignore_existing = self.Parameters.ignore_existing
        force = self.Parameters.force

        yt = self._get_yt_client()
        result = yt.copy(
            source_path=src_table_path,
            destination_path=dst_table_path,
            recursive=recursive,
            ignore_existing=ignore_existing,
            force=force,
        )

        logging.info('Copy "%s" to "%s" done with result: %s', src_table_path, dst_table_path, str(result))
