# -*- coding: utf-8 -*-

import smtplib
from email.mime.text import MIMEText

from sandbox import sdk2

from sandbox.common import errors
from sandbox.projects.cloud.compliance.CloudsEmployeesInventory.clients import Clients
from sandbox.projects.cloud.compliance.CloudsEmployeesInventory.config import ProdConfig as config


class Output(sdk2.Resource):
    """ Output """


class CloudsEmployeesInventory(sdk2.Task):
    def send_result_email(
        self,
        created_ticket_link,
        closed_tickets,
        stats,
        awereness_tickets,
        subscriptions_recipient,
        courses_tickets,
        open_tickets,
        encr_tickets,
        av_tickets,
    ):
        """
        Отправка email с результатами выполнения задачи.

        :param created_ticket_link: Ссылка на созданый тикет в ST, получена из Clients.st_create.
        :param closed_tickets:       Номер закрытого тикета, получен из Clients.st_get_closed.

        :raises errors.TaskError:   Вылет в случае проблемы с отправкой письма.
        """

        msg = config.MSG_MAIN
        if created_ticket_link == "None":
            msg = config.MSG_MAIN_NO_NOSIGN.format(msg)
        else:
            msg = config.MSG_MAIN_NEW_ST.format(msg, created_ticket_link)
        if len(open_tickets) != 0:
            msg = config.MSG_MAIN_OPENED_ST.format(msg, ", ".join(open_tickets))
        if len(closed_tickets) != 0:
            msg = config.MSG_MAIN_CLOSED_ST.format(msg, ", ".join(closed_tickets))
        msg = config.MSG_STATS.format(
            msg,
            stats["percent_signed"],
            stats["percent_devel"],
            stats["persent_course_15"],
            stats["persent_course_317"],
            stats["persent_course_464"],
            stats["persent_course_184"],
            stats["persent_encrypted"],
            stats["persent_av"],
            stats["persent_fw"],
            stats["persent_role"],
        )
        if len(awereness_tickets) != 0:
            msg = config.MSG_MAIN_AW.format(msg, ", ".join(awereness_tickets))
        if len(subscriptions_recipient) != 0:
            msg = config.MSG_MAIN_DEVEL.format(msg, subscriptions_recipient)
        if len(courses_tickets) != 0:
            msg = config.MSG_MAIN_MOE.format(msg, ", ".join(courses_tickets))
        if len(encr_tickets) != 0:
            msg = config.MSG_MAIN_ENCR.format(msg, ", ".join(encr_tickets))
        if len(av_tickets) != 0:
            msg = config.MSG_MAIN_AV.format(msg, ", ".join(av_tickets))
        msg = MIMEText(msg)
        msg["Subject"] = config.MSG_MAIN_SUBJECT
        msg["From"] = config.EMAIL_SENDER
        msg["To"] = config.EMAIL_TARGET
        server = smtplib.SMTP(config.SMTP_SERVER, port=25)
        try:
            server.sendmail(config.EMAIL_SENDER, config.EMAIL_TARGET, msg.as_string())
            server.quit()
        except smtplib.SMTPException as exc:
            raise errors.TaskError("Did not send the mail to cloud-compliance {}".format(exc))

    def send_subscriptions_mail(self, no_devel):
        recipient = list()
        if len(no_devel) != 0:
            recipient = list(map(lambda login: login + "@yandex-team.ru", no_devel))
            recipient = ", ".join(recipient)
            msg = config.MSG_SUBSCRIPTIONS
            msg = MIMEText(msg)
            msg["To"] = recipient
            msg["From"] = config.EMAIL_SENDER
            msg["Subject"] = "Subscriptions"
            server = smtplib.SMTP(config.SMTP_SERVER, port=25)
            try:
                server.sendmail(config.EMAIL_SENDER, recipient, msg.as_string())
                server.quit()
            except smtplib.SMTPException as exc:
                raise errors.TaskError("Did not send the Subscriptions mail {}".format(exc))

        return recipient

    def on_execute(self):
        clients = Clients()

        logins_list, connected_logins_list = clients.list_logins_make()

        connection = clients.db_connect()
        db_data = clients.db_get(connection)

        devel_list, leads_list = clients.ml_get()

        users_data = clients.users_data_get(
            logins_list, connected_logins_list, devel_list, leads_list
        )

        for login in users_data:
            db_data.setdefault(
                login,
                {
                    "docs_signed": False,
                    "docs_signing_ticket": "No ticket",
                    "added": clients.date,
                    "aw_ticket": "No ticket",
                    "courses_ticket": "No ticket",
                    "encryption_status": {},
                    "av_status": {},
                    "role_check": "not pass",
                    "fw_status": {},
                    "course_1010_ticket": "No ticket",
                    "role_ticket": "No ticket",
                },
            )
            db_data[login].update(users_data[login])

        db_data, closed_tickets, open_tickets = clients.st_get_closed(db_data)
        db_data = clients.db_mark_out(db_data)

        db_data = clients.encryption_status_set(db_data)
        db_data = clients.av_status_set(db_data)
        db_data = clients.fw_status_set(db_data)
        db_data = clients.role_status_set(db_data)
        db_data = clients.st_docs_signed_status_set(db_data)

        awereness_tickets, db_data = clients.st_create_awareness(db_data)
        courses_tickets, db_data = clients.st_create_courses(db_data)
        created_ticket_link, db_data = clients.st_create_sign(db_data)
        encr_tickets, db_data = clients.st_create_encr(db_data)
        av_tickets, db_data = clients.st_create_av(db_data)
        role_tickets, db_data = clients.st_create_role(db_data)

        no_devel = clients.ml_subscribe(db_data)

        stats = clients.stats_get(db_data)

        clients.db_push(db_data, connection)
        connection.close()

        subscriptions_recipient = self.send_subscriptions_mail(no_devel)
        self.send_result_email(
            created_ticket_link,
            closed_tickets,
            stats,
            awereness_tickets,
            subscriptions_recipient,
            courses_tickets,
            open_tickets,
            encr_tickets,
            av_tickets,
        )
