# -*- coding: utf-8 -*-

import os
import json
import datetime
import requests
import logging

from sandbox.sdk2 import yav
from sandbox.projects.cloud.compliance.CloudsEmployeesInventory.config import ProdConfig as config


class Clients:
    def __init__(self):
        """
        Конструктор. Дергает серкреты из YAV и записывает дату.
        """

        secret = yav.Secret(config.SECRET_ID)
        logging.info("Getting secrets.")
        oauth_token = secret.data().get("token", None)

        if oauth_token is not None:
            self.headers = {}
            self.headers["Authorization"] = "OAuth {}".format(oauth_token)
        else:
            raise Exception("Problem with token!")

        yt_token = secret.data().get("yt_token", None)

        if yt_token is not None:
            os.environ["YT_TOKEN"] = yt_token
        else:
            raise Exception("Problem with yt_token!")

        self.db_password = secret.data().get("db_password", None)
        if self.db_password is None:
            raise Exception("Problem with DB password!")

        self.tvm_secret = secret.data().get("tvm_secret", None)
        if self.tvm_secret is None:
            raise Exception("Problem with TVM secret!")

        self.date = datetime.date.today()
        self.service_tickets = self.tvm_tickets_get()

    def db_connect(self):
        """
        Подключение к БД.
        :return: Информация об установленом соединении с БД.
        """

        import psycopg2

        logging.info("Connecting to DB.")
        connection = psycopg2.connect(config.SQL_CONNECTION_PARAMS.format(self.db_password))
        connection.autocommit = True

        return connection

    def db_get(self, connection):
        """
        Выгрузка данный из БД.
        :param connection: информация об установленом соединении с БД.
        :return:           dictionary, словарь с данными из БД.
        """

        from psycopg2.extras import DictCursor

        logging.info("Getting data from DB.")
        cursor = connection.cursor(cursor_factory=DictCursor)
        cursor.execute(config.SQL_QUERY_GET)
        db_output = cursor.fetchall()
        db_data = {}
        for item in db_output:
            login = item[0]
            db_data[login] = {
                "name": item[1],
                "iso_relation": item[2],
                "docs_signed": item[3],
                "docs_signing_ticket": item[4],
                "last_update": item[5],
                "staff_department": item[6],
                "location": item[7],
                "added": item[8],
                "position": item[9],
                "devel_subscribed": item[10],
                "leads_subscribed": item[11],
                "course_15_all_result": item[12],
                "course_15_all_date": item[13],
                "course_15_tech_result": item[14],
                "course_15_tech_date": item[15],
                "course_184_result": item[16],
                "course_184_date": item[17],
                "course_317_result": item[18],
                "course_317_date": item[19],
                "course_464_result": item[20],
                "course_464_date": item[21],
                "uid": item[22],
                "aw_ticket": item[23],
                "courses_ticket": item[24],
                "encryption_status": item[25],
                "av_status": item[26],
                "role_check": item[27],
                "fw_status": item[28],
                "services": item[29],
                "st_docs_signed": item[30],
                "course_1010_result": item[31],
                "course_1010_date": item[32],
                "course_1010_ticket": item[33],
                "role_ticket": item[34],
            }

        cursor.close()

        return db_data

    def db_push(self, db_data, connection):
        """
        Выгрузка данных в БД.
        :param db_data:     dictionary, словарь с инфрмацией, которую нужно записать в БД.
        :param connection:  информация об установленном соединении с БД.
        """

        logging.info("Pushing data to DB.")
        cursor = connection.cursor()
        for key, value in db_data.items():
            cursor.execute(
                config.SQL_QUERY_PREPARE.format(
                    value["name"],
                    key,
                    value["iso_relation"],
                    value["docs_signed"],
                    value["docs_signing_ticket"],
                    value["last_update"],
                    value["staff_department"],
                    value["location"],
                    value["added"],
                    value["position"],
                    value["devel_subscribed"],
                    value["leads_subscribed"],
                    value["course_15_all_result"],
                    value["course_15_all_date"],
                    value["course_15_tech_result"],
                    value["course_15_tech_date"],
                    value["course_184_result"],
                    value["course_184_date"],
                    value["course_317_result"],
                    value["course_317_date"],
                    value["course_464_result"],
                    value["course_464_date"],
                    value["uid"],
                    value["aw_ticket"],
                    value["courses_ticket"],
                    value["encryption_status"],
                    value["av_status"],
                    value["role_check"],
                    value["fw_status"],
                    value["services"],
                    value["st_docs_signed"],
                    value["course_1010_result"],
                    value["course_1010_date"],
                    value["course_1010_ticket"],
                    value["role_ticket"],
                )
            )

        cursor.close()

    def db_mark_out(self, db_data):
        """
        Изменение статуса iso_relation на out для уволившихся/переведенных сотрудников.
        :param db_data:     dictionary, словарь с инфрмацией о пользователях.
        :return:            dictionary, обновленный словарь с инфрмацией о пользователях.
        """
        logging.info("Marking out users.")
        for login in db_data:
            if db_data[login]["last_update"] != str(self.date):
                db_data[login]["iso_relation"] = "out"
                logging.info(
                    "{} marked as out. Last update - {}, now - {}.".format(
                        login, db_data[login]["last_update"], self.date
                    )
                )
        return db_data

    def tvm_tickets_get(self):
        """
        Получение TVM тикетов для аутентификации.
        :return: TVM тикеты для аутентификации.
        """

        from tvm2 import TVM2
        from tvm2.protocol import BlackboxClientId

        logging.info("Getting TVM ticket.")
        tvm = TVM2(
            client_id=config.TVM_ID_CLOUD,
            secret=self.tvm_secret,
            blackbox_client=BlackboxClientId.Prod,
        )

        service_tickets = tvm.get_service_tickets(config.TVM_ID_MOE, config.TVM_ID_ML)

        return service_tickets

    def moe_get(self, login):
        """
        Получение данных из Мебиуса.
        :param login: login сотрудника для которого необходимо выгрузить информацию.
        :return:      данные из Мебиуса.
        """

        logging.info("Getting data from MOE for {}.".format(login))
        moe_tvm_ticket = self.service_tickets.get(config.TVM_ID_MOE)
        users_result = {}
        for course in config.COURSES_IDS:
            users_result[course] = {}
            response = requests.get(
                url="https://api.moe.yandex-team.ru/api/v3/courses/{}/students_results/?login={}".format(
                    course, login
                ),
                headers={"X-Ya-Service-Ticket": moe_tvm_ticket},
                verify=False,
            )
            data = json.loads(response.content)
            if course == "15":
                users_result["15"]["all"], users_result["15"]["tech"] = {}, {}
                users_result["15"]["tech"]["date"] = "Null"
                users_result["15"]["tech"]["results"] = 0
                users_result["15"]["all"]["date"] = "Null"
                users_result["15"]["all"]["results"] = 0
                if len(data["results"]) != 0:
                    if len(data["results"][0]["clessons"]) != 0:
                        for clesson in data["results"][0]["clessons"]:
                            if clesson["clesson_id"] == 46:
                                if len(clesson) != 0:
                                    users_result["15"]["tech"]["results"] = clesson["result"][
                                        "percent"
                                    ]
                                    if users_result["15"]["tech"]["results"] != 0:
                                        users_result["15"]["tech"]["date"] = clesson["result"][
                                            "last_modified"
                                        ]
                            if clesson["clesson_id"] == 69:
                                if len(clesson) != 0:
                                    users_result["15"]["all"]["results"] = clesson["result"][
                                        "percent"
                                    ]
                                    if users_result["15"]["all"]["results"] != 0:
                                        users_result["15"]["all"]["date"] = clesson["result"][
                                            "last_modified"
                                        ]

            if course == "317":
                moudle1 = 0
                module1_date = "Null"
                moudle2 = 0
                module2_date = "Null"
                users_result["317"]["results"] = 0
                users_result["317"]["date"] = "Null"
                if len(data["results"]) != 0:
                    if len(data["results"][0]["clessons"]) != 0:
                        for clesson in data["results"][0]["clessons"]:
                            if clesson["clesson_id"] == 1462:
                                if len(clesson) != 0:
                                    moudle1 = clesson["result"]["percent"]
                                    if moudle1 != 0:
                                        module1_date = clesson["result"]["last_modified"]
                            if clesson["clesson_id"] == 1476:
                                if len(clesson) != 0:
                                    moudle2 = clesson["result"]["percent"]
                                    if moudle2 != 0:
                                        module2_date = clesson["result"]["last_modified"]
                        users_result["317"]["results"] = (moudle1 + moudle2) / 2
                        if module1_date != "Null":
                            users_result["317"]["date"] = module1_date
                        else:
                            users_result["317"]["date"] = module2_date

            if course == "184":
                users_result["184"]["results"] = 0
                users_result["184"]["date"] = "Null"
                if len(data["results"]) != 0:
                    if len(data["results"][0]["clessons"]) != 0:
                        for clesson in data["results"][0]["clessons"]:
                            if clesson["clesson_id"] == 951:
                                if len(clesson) != 0:
                                    users_result["184"]["results"] = clesson["result"]["percent"]
                                    if users_result["184"]["results"] != 0:
                                        users_result["184"]["date"] = clesson["result"][
                                            "last_modified"
                                        ]

            if course == "464":
                users_result["464"]["date"] = "Null"
                users_result["464"]["results"] = 0
                if len(data["results"]) != 0:
                    if len(data["results"][0]["clessons"]) != 0:
                        for clesson in data["results"][0]["clessons"]:
                            if clesson["clesson_id"] == 1978 and len(clesson) != 0:
                                users_result["464"]["results"] = clesson["result"]["percent"]
                                if users_result["464"]["results"] != 0:
                                    users_result["464"]["date"] = clesson["result"]["last_modified"]

            if course == "1010":
                users_result["1010"]["date"] = "Null"
                users_result["1010"]["results"] = 0
                if len(data["results"]) != 0:
                    if len(data["results"][0]["clessons"]) != 0:
                        for clesson in data["results"][0]["clessons"]:
                            if clesson["clesson_id"] == 5096 and len(clesson) != 0:
                                users_result["1010"]["results"] = clesson["result"]["percent"]
                                if users_result["1010"]["results"] != 0:
                                    users_result["1010"]["date"] = clesson["result"][
                                        "last_modified"
                                    ]

        return users_result

    def ml_get(self):
        """
        Получение данных из ML.
        :param service_tickets: TVM тикеты для аутентификации.
        :return:                члены список рассылок.
        """

        ml_tvm_ticket = self.service_tickets.get(config.TVM_ID_ML)
        logging.info("Getting data from ML.")
        devel_list, leads_list = set(), set()
        response = requests.get(
            url="https://ml.yandex-team.ru/apiv3/lists/subscribers?emails=devel@yandex-team.ru,cloud-leads@yandex-team.ru",
            headers={"X-Ya-Service-Ticket": ml_tvm_ticket},
        )
        if response.status_code == 200:
            data = json.loads(response.content)
            data = data["result"]
        else:
            raise Exception(
                "Problem with API (ML GET) connection, status code is {}".format(
                    response.status_code
                )
            )

        for login in data["devel@yandex-team.ru"]["subscribers"]:
            devel_list.add(login["login"])

        for login in data["cloud-leads@yandex-team.ru"]["subscribers"]:
            leads_list.add(login["login"])

        return devel_list, leads_list

    def ml_subscribe(self, db_data):
        uid_list = list()
        no_devel = list()
        for login in db_data:
            if (
                db_data[login]["devel_subscribed"] is False
                and db_data[login]["iso_relation"] == "scope"
            ):
                uid_list.append(db_data[login]["uid"])
                no_devel.append(login)

        if len(uid_list) > 0:
            logging.info("Subscribing users.")
            ml_tvm_ticket = self.service_tickets.get(config.TVM_ID_ML)
            params = {
                "maillist": "devel@yandex-team.ru",
                "subscribers": '{"imap":%s}' % uid_list,
            }
            response = requests.post(
                url="https://ml.yandex-team.ru/apiv3/lists/subscribe",
                headers={"X-Ya-Service-Ticket": ml_tvm_ticket},
                data=params,
            )
            if response.status_code == 200:
                pass
            else:
                raise Exception(
                    "Problem with API (ML SUBSCRIBE) connection, status code is {}".format(
                        response.status_code
                    )
                )

        return no_devel

    def api_get(self, api_url, api_params):
        """
        Получение данных из API.
        :param api_url:    ссылка на API нужного сервиса.
        :param api_params: парметры вызова API.
        :return:           ответ API.
        """

        logging.info(
            "Getting data from api {} with params: {}".format(
                api_url, str(api_params).encode("utf-8")
            )
        )
        response = requests.get(url=api_url, headers=self.headers, params=api_params)
        if response.status_code == 200:
            data = json.loads(response.content)
        else:
            raise Exception(
                "Problem with API ({}) connection, status code is {}. \n Text:\n{}\nData:\n{}".format(
                    api_url, response.status_code, response.text, api_params
                )
            )

        return data

    def api_post(self, api_url, api_params):
        """
        Передача данных через API.
        :param api_url:    ссылка на API нужного сервиса.
        :param api_params: парметры вызова API.
        :return:           ответ API.
        """

        logging.info(
            "Posting data to api {} with params: {}".format(
                api_url, str(api_params).encode("utf-8")
            )
        )
        response = requests.post(url=api_url, headers=self.headers, data=api_params)
        if response.status_code == 201 or response.status_code == 200:
            data = json.loads(response.content)
        elif "внешний пользователь" in response.text:
            data = {"key": "ASSESSOR"}
        else:
            raise Exception(
                "Problem with API ({}) connection, status code is {}. \n Text:\n{}\nData:\n{}".format(
                    api_url, response.status_code, response.text, api_params
                )
            )

        return data

    def st_create_sign(self, db_data):
        """
        Создание нового тикета в ST.
        :param db_data:     dictionary, словарь с инфрмацией, по которой создается тикет.
        :return:            ссылка на тикет, обновленный db_data с инфой о созданых тикетах.
        """

        logging.info("Creating sign ticket.")
        ticket_link = str(None)
        no_sign = set()
        for login in db_data:
            if (
                db_data[login]["docs_signed"] is False
                and db_data[login]["docs_signing_ticket"] == "No ticket"
                and db_data[login]["courses_ticket"] != "ASSESSOR"
                and db_data[login]["aw_ticket"] != "ASSESSOR"
            ):
                no_sign.add("кто:{}".format(login))

        if len(no_sign) != 0:
            ticket_params = {
                "queue": config.TASK_QUEUE,
                "summary": config.TASK_SUMMARY,
                "description": config.TASK_TEXT.format("\n".join(no_sign)),
                "followers": config.TASK_FOLLOWERS,
                "assignee": config.TASK_ASSIGNE,
            }
            response = self.api_post(
                "https://st-api.yandex-team.ru/v2/issues", json.dumps(ticket_params)
            )
            ticket_num = response["key"]
            ticket_link = "https://st.yandex-team.ru/{}".format(response["key"])

            for login in db_data:
                if (
                    db_data[login]["docs_signed"] is False
                    and db_data[login]["docs_signing_ticket"] == "No ticket"
                ):
                    db_data[login]["docs_signing_ticket"] = ticket_num

        return ticket_link, db_data

    def st_create_awareness(self, db_data):
        logging.info("Creating awareness tickets.")
        created_tickets = set()
        for login in db_data:
            if (
                db_data[login]["docs_signed"] is False
                and db_data[login]["docs_signing_ticket"] == "No ticket"
                and db_data[login]["iso_relation"] == "scope"
            ):
                ticket_params = {
                    "queue": config.AWARENESS_TASK_QUEUE,
                    "summary": config.AWARENESS_TASK_SUMMARY,
                    "description": config.AWARENESS_TASK_TEXT,
                    "followers": config.AWARENESS_TASK_FOLLOWERS,
                    "assignee": {"id": login},
                    "checklistItems": config.AWARENESS_TASK_CHECKLIST,
                    "tags": config.AWARENESS_TASK_TAGS,
                }
                response = self.api_post(
                    "https://st-api.yandex-team.ru/v2/issues", json.dumps(ticket_params)
                )
                try:
                    ticket_num = response["key"]
                    created_tickets.add(ticket_num)
                except:
                    if "внешний пользователь" in response["text"]:
                        ticket_num = "ASSESSOR"
                    else:
                        raise Exception(
                            "Problem with awareness ticket. Text:\n{}\nTicket params:\n{}".format(
                                response["text"], ticket_params
                            )
                        )
                db_data[login]["aw_ticket"] = ticket_num

        return created_tickets, db_data

    def st_create_courses(self, db_data):
        logging.info("Creating courses tickets.")
        created_tickets = set()
        for login in db_data:
            courses = list()
            # if (
            #     db_data[login]["iso_relation"] == "scope"
            #     and "азраб" in db_data[login]["position"]
            #     and db_data[login]["course_317_result"] < 70
            # ):
            #     courses.append("- Основы безопасной разработки в Яндекс.Облаке - код 268R89LZ")
            if (
                db_data[login]["iso_relation"] == "scope"
                and db_data[login]["course_15_all_result"] < 70
                and db_data[login]["course_15_tech_result"] < 70
            ):
                courses.append("- Защита приватных данных - код GU32W2SR")
            if (
                db_data[login]["staff_department"] in config.ITDC_COURSE_LIST
                and db_data[login]["course_184_result"] < 70
                and "ежурный" in db_data[login]["position"]
            ):
                courses.append(
                    "- Служба ITDC. Базовый уровень (модуль DCOPS. Регламент работы со стойками CLOUD. (теория+тест)) - код 93S4NNS8"
                )
            if (
                db_data[login]["iso_relation"] == "scope"
                and db_data[login]["course_464_result"] < 70
            ):
                courses.append(
                    "- Требования стандарта ISO-270x в области управления ИБ - код 8F8Z6297"
                )

            if (
                len(courses) != 0
                and db_data[login]["courses_ticket"] == "No ticket"
                and db_data[login]["courses_ticket"] != "ASSESSOR"
                and db_data[login]["aw_ticket"] != "ASSESSOR"
            ):
                logging.info("Creating courses ticket for {}.".format(login))
                courses = "\n".join(courses)
                ticket_params = {
                    "queue": config.COURSES_TASK_QUEUE,
                    "summary": config.COURSES_TASK_SUMMARY,
                    "description": config.COURSES_TASK_TEXT.format(courses),
                    "followers": config.COURSES_TASK_FOLLOWERS,
                    "assignee": {"id": login},
                    "tags": config.COURSES_TASK_TAGS,
                }
                response = self.api_post(
                    "https://st-api.yandex-team.ru/v2/issues", json.dumps(ticket_params)
                )
                try:
                    ticket_num = response["key"]
                    created_tickets.add(ticket_num)
                    logging.info("Created {}.".format(ticket_num))
                except:
                    if "внешний пользователь" in response["text"]:
                        ticket_num = "ASSESSOR"
                    else:
                        raise Exception(
                            "Problem with create courses ticket. Text:\n{}\nTicket params:\n{}".format(
                                response["text"], ticket_params
                            )
                        )
                db_data[login]["courses_ticket"] = ticket_num

        return created_tickets, db_data

    def st_create_encr(self, db_data):
        logging.info("Creating encryption tickets.")
        created_tickets = set()
        for login in db_data:
            if (
                db_data[login]["iso_relation"] == "scope"
                and db_data[login]["courses_ticket"] != "ASSESSOR"
                and db_data[login]["aw_ticket"] != "ASSESSOR"
            ):
                db_data[login]["encryption_status"] = json.loads(
                    db_data[login]["encryption_status"]
                )
                for key, value in db_data[login]["encryption_status"].items():
                    if value["encrypted"] == "false":
                        if "encr_ticket" not in db_data[login]["encryption_status"][key]:
                            logging.info(
                                "Creating encryption tickets for {} (PC {}).".format(login, key)
                            )
                            ticket_params = {
                                "queue": config.ENCR_TASK_QUEUE,
                                "summary": config.ENCR_TASK_SUMMARY,
                                "description": config.ENCR_TASK_TEXT.format(key, value["serial"]),
                                "followers": config.ENCR_TASK_FOLLOWERS,
                                "assignee": {"id": login},
                                "tags": config.ENCR_TASK_TAGS,
                            }
                            response = self.api_post(
                                "https://st-api.yandex-team.ru/v2/issues", json.dumps(ticket_params)
                            )
                            ticket_num = response["key"]
                            logging.info("Created ticket {}".format(ticket_num))
                            db_data[login]["encryption_status"][key]["encr_ticket"] = ticket_num
                            created_tickets.add(ticket_num)
                db_data[login]["encryption_status"] = json.dumps(
                    db_data[login]["encryption_status"]
                )

        return created_tickets, db_data

    def st_create_sec_dev_course(self, db_data):
        logging.info("Creating new sec. dev. course tickets.")
        created_tickets = set()
        for login in db_data:
            if (
                db_data[login]["iso_relation"] == "scope"
                and db_data[login]["course_1010_ticket"] == "No ticket"
                and db_data[login]["course_1010_result"] < 70
                and (
                    "Разработка" in db_data[login]["services"]
                    or "YC Network Infrastructure" in db_data[login]["services"]
                )
            ):
                logging.info("Creating sec. dev. course ticket for {}.".format(login))
                ticket_params = {
                    "queue": config.SEC_DEV_COURSE_TASK_QUEUE,
                    "summary": config.SEC_DEV_COURSE_TASK_SUMMARY,
                    "description": config.SEC_DEV_COURSE_TASK_TEXT,
                    "followers": config.SEC_DEV_COURSE_TASK_FOLLOWERS,
                    "assignee": {"id": login},
                    "tags": config.SEC_DEV_COURSE_TASK_TAGS,
                }
                response = self.api_post(
                    "https://st-api.yandex-team.ru/v2/issues", json.dumps(ticket_params)
                )
                try:
                    ticket_num = response["key"]
                    created_tickets.add(ticket_num)
                    logging.info("Created {}.".format(ticket_num))
                except:
                    if "внешний пользователь" in response["text"]:
                        ticket_num = "ASSESSOR"
                    else:
                        raise Exception(
                            "Problem with create courses ticket. Text:\n{}\nTicket params:\n{}".format(
                                response["text"], ticket_params
                            )
                        )
                db_data[login]["course_1010_ticket"] = ticket_num

        return created_tickets, db_data

    def st_create_av(self, db_data):
        logging.info("Creating AV tickets.")
        created_tickets = set()
        for login in db_data:
            if (
                db_data[login]["iso_relation"] == "scope"
                and db_data[login]["courses_ticket"] != "ASSESSOR"
                and db_data[login]["aw_ticket"] != "ASSESSOR"
            ):
                db_data[login]["av_status"] = json.loads(db_data[login]["av_status"])
                logging.info("Login is {}, status is {}".format(login, db_data[login]["av_status"]))
                for key, value in db_data[login]["av_status"].items():
                    logging.info("Check for AV not passed.")
                    if value["installed"] == "false" or value["installed"] == "disabled":
                        logging.info("Creating AV tickets for {} (PC {}).".format(login, key))
                        ticket_params = {
                            "queue": config.AV_TASK_QUEUE,
                            "summary": config.AV_TASK_SUMMARY,
                            "description": config.AV_TASK_TEXT.format(key),
                            "followers": config.AV_TASK_FOLLOWERS,
                            "assignee": {"id": login},
                            "tags": config.AV_TASK_TAGS,
                        }
                        response = self.api_post(
                            "https://st-api.yandex-team.ru/v2/issues", json.dumps(ticket_params)
                        )
                        ticket_num = response["key"]
                        logging.info("Created ticket {}".format(ticket_num))
                        db_data[login]["av_status"][key]["av_ticket"] = ticket_num
                        created_tickets.add(ticket_num)
                db_data[login]["av_status"] = json.dumps(db_data[login]["av_status"])

            return created_tickets, db_data

    def st_create_role(self, db_data):
        logging.info("Creating role tickets.")
        created_tickets = set()
        for login in db_data:
            if (
                db_data[login]["iso_relation"] == "scope"
                and db_data[login]["courses_ticket"] != "ASSESSOR"
                and db_data[login]["aw_ticket"] != "ASSESSOR"
                and db_data[login]["role_check"] == "not pass"
                and db_data[login]["role_ticket"] == "No ticket"
            ):
                ticket_params = {
                    "queue": config.ROLE_TASK_QUEUE,
                    "summary": config.ROLE_TASK_SUMMARY,
                    "description": config.ROLE_TASK_TEXT,
                    "followers": config.ROLE_TASK_FOLLOWERS,
                    "assignee": {"id": login},
                    "tags": config.ROLE_TASK_TAGS,
                }
                response = self.api_post(
                    "https://st-api.yandex-team.ru/v2/issues", json.dumps(ticket_params)
                )
                ticket_num = response["key"]
                created_tickets.add(ticket_num)
                db_data[login]["role_ticket"] = ticket_num

        return created_tickets, db_data

    def st_get_closed(self, db_data):
        """
        Проверка имеющихся тикетов.
        Если обнаружен закрытый - сотрудники в нем помечаются как подписавшие все документы.
        :param db_data:     dictionary, словарь с инфрмацией о пользователях и открытых по
                            ним тикетах.
        :return:            ссылка на тикет, обновленный db_data.
        """

        logging.info("Getting closed tickets.")
        closed_tickets = set()
        open_tickets = set()
        for login in db_data:
            if (
                db_data[login]["docs_signed"] is False
                and db_data[login]["docs_signing_ticket"] != "No ticket"
            ):
                ticket_num = db_data[login]["docs_signing_ticket"]
                data_from_st = self.api_get(
                    "https://st-api.yandex-team.ru/v2/issues/{}".format(ticket_num), None,
                )
                if data_from_st["status"]["key"] == "closed":
                    db_data[login]["docs_signed"] = True
                    closed_tickets.add(ticket_num)
                else:
                    open_tickets.add(ticket_num)
        open_tickets = list(open_tickets)

        return db_data, closed_tickets, open_tickets

    def list_logins_make(self):
        """
        Формирование списка логинов сотрудников из скоупа задачи.
        :return: списки логинов.
        """

        logging.info("Making logins lists.")
        logins_list = set()
        connected_logins_list = set()
        abc_link = config.ABC_URL

        while abc_link is not None:
            abc_params = {
                "page_size": 1000,
                "service__with_descendants": "1415",
            }
            data_from_abc_cloud = self.api_get(abc_link, abc_params)
            result = data_from_abc_cloud["results"]
            for item in result:
                if item["service"]["id"] not in config.ABC_BLACK_LIST:
                    login = item["person"]["login"]
                    logins_list.add(login)
            abc_link = data_from_abc_cloud["next"]
        abc_link = config.ABC_URL
        abc_params = {
            "page_size": 1000,
            "service__with_descendants": "32922",
        }
        data_from_abc_cloud = self.api_get(abc_link, abc_params)
        result = data_from_abc_cloud["results"]
        for item in result:
            login = item["person"]["login"]
            logins_list.add(login)
        logins_list.add("bkht")  # TMP
        logging.info("Loggins list is {}".format(logins_list))

        for group_url in config.CONNECTED_GROUPS_URLS:
            staff_group_params = {
                "_fields": "person.login,person.name,person.official",
                "group.url": group_url,
            }
            data_from_staff = self.api_get(config.STAFF_GROUPS_URL, staff_group_params)
            connected_result = data_from_staff["result"]
            for item in connected_result:
                login = item["person"]["login"]
                connected_logins_list.add(login)

        logins_list = ",".join(str(login) for login in logins_list)
        connected_logins_list = ",".join(str(login) for login in connected_logins_list)

        logging.info(
            "Logins lists are ready: scope:{}; connected: {}".format(
                logins_list, connected_logins_list
            )
        )

        return logins_list, connected_logins_list

    def users_data_get(self, logins_list, connected_logins_list, devel_list, leads_list):
        """
        Выгрузка данных по логинам из staff.
        :param connected_logins_list: список логинов сотрудников не входящих в скоуп сертификата,
                                      но учавствующих в деятельности.
        :param logins_list:           список логинов сотрудников в скоупе сертификата.
        :return:                      словарь с обновленными данными со staff.
        """

        logging.info("Getting users data.")
        users_data = {}
        staff_params = {
            "_fields": "uid,login,name,official,department_group,location,groups",
            "_limit": "1000",
            "login": logins_list,
        }
        data = self.api_post(config.STAFF_URL, staff_params)
        data = data["result"]
        for user in data:
            if user["official"]["is_robot"] is False and user["official"]["is_dismissed"] is False:
                login = "{}".format(user["login"])
                devel_subscribed = login in devel_list
                leads_subscribed = login in leads_list
                moe_users_data = self.moe_get(login)
                users_data[login] = {
                    "name": "{} {}".format(user["name"]["first"]["en"], user["name"]["last"]["en"]),
                    "iso_relation": "scope",
                    "last_update": "{}".format(self.date),
                    "staff_department": "{}".format(user["department_group"]["name"]),
                    "location": "{}".format(user["location"]["office"]["name"]["en"]),
                    "position": "{}".format(user["official"]["position"]["ru"]),
                    "devel_subscribed": devel_subscribed,
                    "leads_subscribed": leads_subscribed,
                    "course_15_all_result": moe_users_data["15"]["all"]["results"],
                    "course_15_all_date": moe_users_data["15"]["all"]["date"],
                    "course_15_tech_result": moe_users_data["15"]["tech"]["results"],
                    "course_15_tech_date": moe_users_data["15"]["tech"]["date"],
                    "course_184_result": moe_users_data["184"]["results"],
                    "course_184_date": moe_users_data["184"]["date"],
                    "course_317_result": moe_users_data["317"]["results"],
                    "course_317_date": moe_users_data["317"]["date"],
                    "course_464_result": moe_users_data["464"]["results"],
                    "course_464_date": moe_users_data["464"]["date"],
                    "uid": int(user["uid"]),
                    "services": (
                        ", ".join([service["group"]["name"] for service in user["groups"]])
                    ).replace("'", ""),
                    "course_1010_result": moe_users_data["1010"]["results"],
                    "course_1010_date": moe_users_data["1010"]["date"],
                }

        staff_params = {
            "_fields": "uid,login,name,official,department_group,location,groups",
            "_limit": "1000",
            "login": connected_logins_list,
        }
        data = self.api_get(config.STAFF_URL, staff_params)
        data = data["result"]
        for user in data:
            if user["official"]["is_robot"] is False and user["official"]["is_dismissed"] is False:
                login = "{}".format(user["login"])
                moe_users_data = self.moe_get(login)
                users_data[login] = {
                    "name": "{} {}".format(user["name"]["first"]["en"], user["name"]["last"]["en"]),
                    "iso_relation": "connected",
                    "last_update": "{}".format(self.date),
                    "staff_department": "{}".format(user["department_group"]["name"]),
                    "location": "{}".format(user["location"]["office"]["name"]["en"]),
                    "position": "{}".format(user["official"]["position"]["ru"]),
                    "devel_subscribed": devel_subscribed,
                    "leads_subscribed": leads_subscribed,
                    "course_15_all_result": moe_users_data["15"]["all"]["results"],
                    "course_15_all_date": moe_users_data["15"]["all"]["date"],
                    "course_15_tech_result": moe_users_data["15"]["tech"]["results"],
                    "course_15_tech_date": moe_users_data["15"]["tech"]["date"],
                    "course_184_result": moe_users_data["184"]["results"],
                    "course_184_date": moe_users_data["184"]["date"],
                    "course_317_result": moe_users_data["317"]["results"],
                    "course_317_date": moe_users_data["317"]["date"],
                    "course_464_result": moe_users_data["464"]["results"],
                    "course_464_date": moe_users_data["464"]["date"],
                    "uid": int(user["uid"]),
                    "services": (
                        ", ".join([service["group"]["name"] for service in user["groups"]])
                    ).replace("'", ""),
                    "course_1010_result": moe_users_data["1010"]["results"],
                    "course_1010_date": moe_users_data["1010"]["date"],
                }

        return users_data

    def stats_get(self, db_data):
        logging.info("Getting stats.")
        stats = {
            "amount_connected": 0,
            "amount_scope": 0,
            "amount_developers": 0,
            "amount_devel": 0,
            "amount_need_184": 0,
            "percent_devel": 0,
            "amount_signed": 0,
            "percent_signed": 0,
            "amount_course_15": 0,
            "persent_course_15": 0,
            "amount_course_184": 0,
            "persent_course_184": 0,
            "amount_course_317": 0,
            "persent_course_317": 0,
            "amount_course_464": 0,
            "persent_course_464": 0,
            "amount_win_macos": 0,
            "encrypted": 0,
            "persent_encrypted": 0,
            "av": 0,
            "amount_av": 0,
            "persent_av": 0,
            "fw": 0,
            "amount_fw": 0,
            "persent_fw": 0,
            "role": 0,
            "persent_role": 0,
        }

        for login in db_data:
            if db_data[login]["iso_relation"] == "connected":
                stats["amount_connected"] += 1
            if db_data[login]["iso_relation"] == "scope":
                stats["amount_scope"] += 1
            if "азраб" in db_data[login]["position"] and db_data[login]["iso_relation"] == "scope":
                stats["amount_developers"] += 1
            if (
                db_data[login]["iso_relation"] == "scope"
                and db_data[login]["devel_subscribed"] is True
            ):
                stats["amount_devel"] += 1
            if db_data[login]["docs_signed"] is True and db_data[login]["iso_relation"] != "out":
                stats["amount_signed"] += 1
            if db_data[login]["iso_relation"] == "scope" and (
                db_data[login]["course_15_all_result"] > 70
                or db_data[login]["course_15_tech_result"] > 70
            ):
                stats["amount_course_15"] += 1
            if (
                db_data[login]["staff_department"] in config.ITDC_COURSE_LIST
                and "ежурный" in db_data[login]["position"]
                and db_data[login]["iso_relation"] != "out"
            ):
                stats["amount_need_184"] += 1
            if (
                db_data[login]["staff_department"] in config.ITDC_COURSE_LIST
                and db_data[login]["course_184_result"] > 70
                and "ежурный" in db_data[login]["position"]
                and db_data[login]["iso_relation"] != "out"
            ):
                stats["amount_course_184"] += 1
            if (
                db_data[login]["iso_relation"] == "scope"
                and "азраб" in db_data[login]["position"]
                and db_data[login]["course_317_result"] > 70
            ):
                stats["amount_course_317"] += 1
            if (
                db_data[login]["iso_relation"] == "scope"
                and db_data[login]["course_464_result"] > 70
            ):
                stats["amount_course_464"] += 1
            if (
                db_data[login]["iso_relation"] == "scope"
                and db_data[login]["encryption_status"] != {}
            ):
                status = json.loads(db_data[login]["encryption_status"])
                for key, value in status.items():
                    stats["amount_win_macos"] += 1
                    if value["encrypted"] == "true":
                        stats["encrypted"] += 1
            if db_data[login]["iso_relation"] == "scope" and db_data[login]["av_status"] != {}:
                status = json.loads(db_data[login]["av_status"])
                for key, value in status.items():
                    stats["amount_av"] += 1
                    if value["installed"] == "true" or value["installed"] == "enabled":
                        stats["av"] += 1
            if db_data[login]["iso_relation"] == "scope" and db_data[login]["fw_status"] != {}:
                status = json.loads(db_data[login]["fw_status"])
                for key, value in status.items():
                    stats["amount_fw"] += 1
                    if value["enabled"] == "true" or value["enabled"] == "enabled":
                        stats["fw"] += 1
            if (
                db_data[login]["iso_relation"] == "scope"
                and db_data[login]["role_check"] == "not pass"
            ):
                stats["role"] += 1

        stats["percent_devel"] = stats["amount_devel"] / (stats["amount_scope"] * 0.01)
        stats["percent_signed"] = stats["amount_signed"] / (
            (stats["amount_scope"] + stats["amount_connected"]) * 0.01
        )
        stats["persent_course_15"] = stats["amount_course_15"] / (stats["amount_scope"] * 0.01)
        stats["persent_course_464"] = stats["amount_course_464"] / (stats["amount_scope"] * 0.01)
        stats["persent_encrypted"] = stats["encrypted"] / (stats["amount_win_macos"] * 0.01)
        stats["persent_av"] = stats["av"] / (stats["amount_av"] * 0.01)
        stats["persent_fw"] = stats["fw"] / (stats["amount_fw"] * 0.01)
        stats["persent_role"] = stats["role"] / (stats["amount_scope"] * 0.01)

        try:
            stats["persent_course_317"] = stats["amount_course_317"] / (
                stats["amount_developers"] * 0.01
            )
        except:
            stats["persent_course_317"] = 100

        try:
            stats["persent_course_184"] = stats["amount_course_184"] / (
                stats["amount_need_184"] * 0.01
            )
        except:
            stats["persent_course_184"] = 100

        logging.info("Stats is {}".format(stats))

        return stats

    def encryption_status_set(self, db_data):
        import yt.wrapper as yt

        logging.info("Getting encryption status for macOS.")
        yt.config.set_proxy("hahn")
        for record in yt.read_table(config.YT_MAC_TABLE, format="json"):
            login = record["username"]
            if login in db_data:
                logging.info("Setting encryption status for {}.".format(login))
                if record["computer_name"] not in db_data[login]["encryption_status"]:
                    db_data[login]["encryption_status"][record["computer_name"]] = dict()
                    db_data[login]["encryption_status"][record["computer_name"]]["os"] = record[
                        "operating_system"
                    ]
                    db_data[login]["encryption_status"][record["computer_name"]]["serial"] = record[
                        "serial_number"
                    ]
                    if record["filevault_2_status"] == "No Partitions Encrypted":
                        db_data[login]["encryption_status"][record["computer_name"]][
                            "encrypted"
                        ] = "false"
                    else:
                        db_data[login]["encryption_status"][record["computer_name"]][
                            "encrypted"
                        ] = "true"
                    logging.info("Added {}.".format(db_data[login]["encryption_status"]))
                else:
                    if record["filevault_2_status"] == "No Partitions Encrypted":
                        db_data[login]["encryption_status"][record["computer_name"]][
                            "encrypted"
                        ] = "false"
                    else:
                        db_data[login]["encryption_status"][record["computer_name"]][
                            "encrypted"
                        ] = "true"

        logging.info("Getting encryption status for Windows.")
        for record in yt.read_table(config.YT_WIN_ENCRYPTION_TABLE, format="json"):
            login = record["UserName"]
            if login in db_data and record["DriveLetter"] == "C:":
                logging.info("Setting encryption status for {}.".format(login))
                if record["ComputerName"] not in db_data[login]["encryption_status"]:
                    db_data[login]["encryption_status"][record["ComputerName"]] = dict()
                    db_data[login]["encryption_status"][record["ComputerName"]]["os"] = "Win"
                    db_data[login]["encryption_status"][record["ComputerName"]]["serial"] = record[
                        "SerialNumber"
                    ]
                    logging.info("Status is {}.".format(record["ProtectionStatus"]))
                    if record["ProtectionStatus"] == "Зашифрован":
                        db_data[login]["encryption_status"][record["ComputerName"]][
                            "encrypted"
                        ] = "true"
                    else:
                        db_data[login]["encryption_status"][record["ComputerName"]][
                            "encrypted"
                        ] = "false"
                    logging.info("Added {}.".format(db_data[login]["encryption_status"]))
                else:
                    if record["ProtectionStatus"] == "Зашифрован":
                        db_data[login]["encryption_status"][record["ComputerName"]][
                            "encrypted"
                        ] = "true"
                    else:
                        db_data[login]["encryption_status"][record["ComputerName"]][
                            "encrypted"
                        ] = "false"
                    logging.info("Added {}.".format(db_data[login]["encryption_status"]))

        for login in db_data:
            db_data[login]["encryption_status"] = json.dumps(db_data[login]["encryption_status"])

        return db_data

    def av_status_set(self, db_data):
        import yt.wrapper as yt

        logging.info("Getting AV status for macOS.")
        yt.config.set_proxy("hahn")
        for record in yt.read_table(config.YT_MAC_TABLE, format="json"):
            login = record["username"]
            if login in db_data:
                logging.info("Setting AV status for {}.".format(login))
                db_data[login]["av_status"][record["computer_name"]] = dict()
                db_data[login]["av_status"][record["computer_name"]]["os"] = record[
                    "operating_system"
                ]
                db_data[login]["av_status"][record["computer_name"]]["serial"] = record[
                    "serial_number"
                ]
                if "osquery_installed" in record["soc_status"]:
                    db_data[login]["av_status"][record["computer_name"]]["installed"] = "true"
                else:
                    db_data[login]["av_status"][record["computer_name"]]["installed"] = "false"
                logging.info("Added {}.".format(db_data[login]["av_status"]))

        logging.info("Getting AV status for Windows.")
        for record in yt.read_table(config.YT_WIN_AV_FW_TABLE, format="json"):
            login = record["UserName"]
            if login in db_data:
                logging.info("Setting AV status for {}.".format(login))
                db_data[login]["av_status"][record["ComputerName"]] = dict()
                db_data[login]["av_status"][record["ComputerName"]]["os"] = "Win"
                logging.info("Status is {}.".format(record["AntivirusState"]))
                if record["AntivirusState"] == "Enabled":
                    db_data[login]["av_status"][record["ComputerName"]]["installed"] = "enabled"
                else:
                    db_data[login]["av_status"][record["ComputerName"]]["installed"] = "disabled"
                logging.info("Added {}.".format(db_data[login]["av_status"]))

        for login in db_data:
            db_data[login]["av_status"] = json.dumps(db_data[login]["av_status"])

        return db_data

    def fw_status_set(self, db_data):
        import yt.wrapper as yt

        logging.info("Getting FW status for macOS.")
        yt.config.set_proxy("hahn")
        for record in yt.read_table(config.YT_MAC_TABLE, format="json"):
            login = record["username"]
            if login in db_data:
                logging.info("Setting FW status for {}.".format(login))
                db_data[login]["fw_status"][record["computer_name"]] = dict()
                db_data[login]["fw_status"][record["computer_name"]]["os"] = record[
                    "operating_system"
                ]
                db_data[login]["fw_status"][record["computer_name"]]["serial"] = record[
                    "serial_number"
                ]
                if record["firewall"] == "On":
                    db_data[login]["fw_status"][record["computer_name"]]["enabled"] = "true"
                else:
                    db_data[login]["fw_status"][record["computer_name"]]["enabled"] = "false"
                logging.info("Added {}.".format(db_data[login]["fw_status"]))

        logging.info("Getting FW status for Windows.")
        for record in yt.read_table(config.YT_WIN_AV_FW_TABLE, format="json"):
            login = record["UserName"]
            if login in db_data:
                logging.info("Setting FW status for {}.".format(login))
                db_data[login]["fw_status"][record["ComputerName"]] = dict()
                db_data[login]["fw_status"][record["ComputerName"]]["os"] = "Win"
                logging.info("Status is {}.".format(record["FirewallState"]))
                if record["FirewallState"] == "Enabled":
                    db_data[login]["fw_status"][record["ComputerName"]]["enabled"] = "enabled"
                else:
                    db_data[login]["fw_status"][record["ComputerName"]]["enabled"] = "disabled"
                logging.info("Added {}.".format(db_data[login]["fw_status"]))

        for login in db_data:
            db_data[login]["fw_status"] = json.dumps(db_data[login]["fw_status"])

        return db_data

    def role_status_set(self, db_data):
        logging.info("Setting role statuses.")
        for login in db_data:
            db_data[login]["role_check"] = "not pass"
            position = str(db_data[login]["position"])
            position = position.decode("utf-8").lower()
            for role in config.ROLES:
                if role in position:
                    db_data[login]["role_check"] = "pass"

        return db_data

    def st_docs_signed_status_set(self, db_data):
        logging.info("Setting ST Docs Signed statuses.")
        for login in db_data:
            if db_data[login]["docs_signed"] is True:
                db_data[login]["st_docs_signed"] = True
            elif (
                db_data[login]["docs_signed"] is False
                and db_data[login]["docs_signing_ticket"] == "No ticket"
            ):
                db_data[login]["st_docs_signed"] = False
            elif (
                db_data[login]["docs_signed"] is False
                and db_data[login]["docs_signing_ticket"] != "No ticket"
            ):
                docs_signing_ticket = self.api_get(
                    "https://st-api.yandex-team.ru/v2/issues/{}?expand=links".format(
                        db_data[login]["docs_signing_ticket"]
                    ),
                    None,
                )
                if "links" in docs_signing_ticket:
                    for ticket in docs_signing_ticket["links"]:
                        linked_ticket = self.api_get(
                            "https://st-api.yandex-team.ru/v2/issues/{}".format(
                                ticket["object"]["key"]
                            ),
                            None,
                        )
                        if "ЛНА" in linked_ticket["summary"]:
                            db_data[login]["st_docs_signed"] = True
                            if "pendingReplyFrom" in linked_ticket:
                                for wait_login in linked_ticket["pendingReplyFrom"]:
                                    if wait_login["id"] == login:
                                        db_data[login]["st_docs_signed"] = False
                else:
                    db_data[login]["st_docs_signed"] = False

        return db_data
