# -*- coding: utf-8 -*-

import logging
import ssl
import datetime

from urllib.request import HTTPSHandler
import smtplib
from email.mime.text import MIMEText

from sandbox import sdk2
from sandbox.sdk2 import yav
from sandbox.common import errors

from suds.client import Client
from suds.wsse import Security, UsernameToken
from suds.transport.https import HttpAuthenticated

from sandbox.projects.cloud.compliance.CoverityParserPy3.components_map import ComponentsMap


class Secrets:
    def __init__(self):
        secret = yav.Secret("sec-01faqp1bsq29phwyvgc227480a")

        self.st_token = secret.data().get("st_token", None)
        if self.st_token is None:
            raise Exception("Problem with ST token!")
        self.user = secret.data().get("user", None)
        if self.user is None:
            raise Exception("Problem with user!")
        self.password = secret.data().get("password", None)
        if self.password is None:
            raise Exception("Problem with password!")


class Coverity:
    def __init__(self):
        self.defectServiceClient = Coverity.DefectServiceClient()
        self.configServiceClient = Coverity.ConfigServiceClient()
        self.projects = Coverity.Projects(self.configServiceClient, self.defectServiceClient)

        self.getComponentsMapInclusion()
        self.getUnassignedVulns()

    class CustomTransport(HttpAuthenticated):
        def u2handlers(self):
            handlers = HttpAuthenticated.u2handlers(self)
            ctx = ssl.create_default_context()
            ctx.check_hostname = False
            ctx.verify_mode = ssl.CERT_NONE
            handlers.append(HTTPSHandler(context=ctx))

            return handlers

    class WebServiceClient:
        def __init__(self, webservice_type):
            url = "https://coverity.cloud.yandex.net:8443"
            if webservice_type == "configuration":
                self.wsdlFile = url + "/ws/v9/configurationservice?wsdl"
            elif webservice_type == "defect":
                self.wsdlFile = url + "/ws/v9/defectservice?wsdl"

            self.client = Client(self.wsdlFile, transport=Coverity.CustomTransport())
            self.security = Security()
            self.token = UsernameToken(Secrets().user, Secrets().password)
            self.security.tokens.append(self.token)
            self.client.set_options(wsse=self.security)

    class DefectServiceClient:
        def __init__(self):
            Coverity.WebServiceClient.__init__(self, "defect")

        def getVulnsInStreamWithoutTicket(self, stream):
            streamIdDO = self.client.factory.create("streamIdDataObj")
            streamIdDO.name = stream["id"]["name"]

            filterSpecDO = self.client.factory.create("mergedDefectFilterSpecDataObj")
            filterSpecDO.classificationNameList = "Unclassified"
            filterSpecDO.externalReferencePattern = None

            pageSpecDO = self.client.factory.create("pageSpecDataObj")
            pageSpecDO.pageSize = 1000
            pageSpecDO.startIndex = 0

            snapshotScopeDO = self.client.factory.create("mergedDefectFilterSpecDataObj")
            snapshotScopeDO.showSelector = "last()"

            mergedDefects = self.client.service.getMergedDefectsForStreams(streamIdDO, filterSpecDO, pageSpecDO)
            if mergedDefects.totalNumberOfRecords:
                all_vulns = mergedDefects.mergedDefects
                filterSpecDO.externalReferencePattern = "*"
                mergedDefects = self.client.service.getMergedDefectsForStreams(streamIdDO, filterSpecDO, pageSpecDO)
                if mergedDefects.totalNumberOfRecords:
                    ids_vulns_with_ext_ref = [vln["cid"] for vln in mergedDefects.mergedDefectIds]
                    vulns = [vln for vln in all_vulns if vln["cid"] not in ids_vulns_with_ext_ref]
                    return vulns
                else:
                    return all_vulns

        def getVulnInProject(self, project):
            vulns_in_project = dict()
            for stream in project["streams"]:
                logging.info("Analyzing stream {}.".format(stream.id.name))
                stream_vulns = self.getVulnsInStreamWithoutTicket(stream)
                if stream_vulns:
                    for vuln in stream_vulns:
                        vulns_in_project.setdefault(vuln["cid"], dict())
                        vulns_in_project[vuln["cid"]].update(vuln)
            return vulns_in_project

        def setTicketAsExtRef(self, vuln, ticket):
            defectStateSpecDO = self.client.factory.create("defectStateSpecDataObj")

            defectStateAttributeValueDO = self.client.factory.create("defectStateAttributeValueDataObj")
            defectStateAttributeValueDO.attributeDefinitionId.name = "Ext. Reference"
            defectStateAttributeValueDO.attributeValueId.name = ticket

            defectStateSpecDO.defectStateAttributeValues = defectStateAttributeValueDO

            streamDefectFilterSpecDO = self.client.factory.create("streamDefectFilterSpecDataObj")
            streamDefectFilterSpecDO.streamIdList = self.client.factory.create("streamIdDataObj")
            streamDefectFilterSpecDO.streamIdList.name = vuln["lastDetectedStream"]

            mergedDefectIdDO = self.client.factory.create("mergedDefectIdDataObj")
            mergedDefectIdDO.cid = vuln["cid"]

            streamDefectIdDO = self.client.service.getStreamDefects(mergedDefectIdDO, streamDefectFilterSpecDO)

            self.client.service.updateStreamDefects(streamDefectIdDO[0].id, defectStateSpecDO)

    class ConfigServiceClient:
        def __init__(self):
            Coverity.WebServiceClient.__init__(self, "configuration")

        def getProjects(self):
            projects = list()
            all_projects = self.client.service.getProjects()
            for project in all_projects:
                if project["id"]["name"].startswith("[YC]") and len(project["streams"]) != 0:
                    projects.append(project)

            return projects

    class Projects:
        def __init__(self, configServiceClient, defectServiceClient):
            self.projects_list = configServiceClient.getProjects()
            self.all_vulns = dict()
            for project in self.projects_list:
                unassigned_vulns_in_project = 0
                vunls_with_component_not_in_map = 0
                logging.info("Analyzing project {}.".format(project.id.name))
                project.projects_owners = dict()
                project.projects_vulns = defectServiceClient.getVulnInProject(project)

                for vuln, properties in project.projects_vulns.items():
                    owner = properties["defectStateAttributeValues"][7]["attributeValueId"]["name"]
                    self.all_vulns.update({vuln: dict()})
                    project.projects_owners.setdefault(owner, dict())
                    self.all_vulns[vuln]["owner"] = owner
                    self.all_vulns[vuln]["component"] = properties["componentName"]
                    self.all_vulns[vuln]["component_in_map"] = self.inComponentMap(vuln, properties["componentName"], project, owner)
                    self.all_vulns[vuln]["impact"] = properties["displayImpact"]
                    self.all_vulns[vuln]["type"] = properties["displayType"]
                    self.all_vulns[vuln]["kind"] = properties["displayIssueKind"]
                    self.all_vulns[vuln]["project"] = project.id.name
                    if self.all_vulns[vuln]["owner"] == "Unassigned":
                        unassigned_vulns_in_project += 1
                    if self.all_vulns[vuln]["component_in_map"] is False:
                        vunls_with_component_not_in_map += 1

                logging.info("In project found {} vulns without external reference, {} unassigned, {} with component not in map.".format(len(project.projects_vulns), unassigned_vulns_in_project,
                                                                                                                                         vunls_with_component_not_in_map))

        def inComponentMap(self, vuln, component_name: str, project, owner) -> bool:
            is_included = False
            for queue, queues_components in ComponentsMap.MAP.items():
                for component, st_component in queues_components.items():
                    if ("*" in component and component.replace("*", "") in component_name) or (component in component_name):
                        is_included |= True
                        self.all_vulns[vuln]["st_component"] = st_component
                        project.projects_owners[owner].setdefault(queue, set())
                        project.projects_owners[owner][queue].add(vuln)

            return is_included

    def getComponentsMapInclusion(self):
        self.components_not_in_map = dict()
        self.component_in_map = set()
        for vuln, properties in self.projects.all_vulns.items():
            if properties["component_in_map"] is False:
                self.components_not_in_map.setdefault(properties["component"], {"owners": set(), "projects": set()})
                self.components_not_in_map[properties["component"]]["owners"].add(properties["owner"])
                self.components_not_in_map[properties["component"]]["projects"].add(properties["project"])

    def getUnassignedVulns(self):
        self.unassigned_vulns = dict()
        for vuln, item in self.projects.all_vulns.items():
            if item["owner"] == "Unassigned":
                self.unassigned_vulns.setdefault(item["project"], set())
                self.unassigned_vulns[item["project"]].add(vuln)


class Ticket:
    def __init__(self, queue, target_vulns, assigne, project_key, project_name, all_vulns, type, deadline, priority):
        self.queue = queue
        self.target_vulns = target_vulns
        self.assigne = assigne
        self.project_key = project_key
        self.project_name = project_name
        self.all_vulns = all_vulns
        self.deadline = deadline.strftime("%Y-%m-%d")
        self.priority = priority

        date = datetime.date.today()
        self.summary = "[SDL]: SAST {} issues identified for {}, {}".format(type, self.project_name, date)

        self.countVulns()
        self.getTrackerComponents()

        self.tags = ["sdl", "coverity"]
        if "IAM" in self.st_components or "Resource Manager" in self.st_components:
            self.tags.append("tax")

        self.createText()

    def countVulns(self):
        self.count_high = 0
        self.count_medium = 0
        self.count_low = 0
        self.count_audit = 0

        for vuln in self.target_vulns:
            if self.all_vulns[vuln]["impact"] == "High":
                self.count_high += 1
            elif self.all_vulns[vuln]["impact"] == "Medium":
                self.count_medium += 1
            elif self.all_vulns[vuln]["impact"] == "Low":
                self.count_low += 1
            elif self.all_vulns[vuln]["impact"] == "Audit":
                self.count_low += 1

    def createText(self):
        self.vulns_table = str()
        for vuln in self.target_vulns:
            self.vulns_table += ("|| " + str(vuln) + " | " + self.all_vulns[vuln]["component"] + " | " + self.all_vulns[vuln]["type"] + " | " + self.all_vulns[vuln]["impact"] + " ||\n ")

    def getTrackerComponents(self):
        self.st_components = set()
        for vuln in self.target_vulns:
            self.st_components.add(self.all_vulns[vuln]["st_component"])

    def createCloudTicket(self):
        from startrek_client import Startrek
        client = Startrek(useragent="RobotYCComplince", token=Secrets().st_token)

        if self.st_components == {""}:
            components = None
        else:
            components = list(self.st_components)

        logging.info(components)

        issue = client.issues.create(
            queue=self.queue,
            summary=self.summary,
            description=("Привет!\n\nCтатический анализ нашел потенциальные уязвимости в "
                         "проекте {}.\n\n#|\n|| !!High!! | !!(жел)Medium!! | !!(зел)Low!! | Audit ||\n"
                         "|| {} | {} | {} | {} ||\n|#\n\n\n"
                         "<{{**Список потенциальных уязвимостей под катом:**\n"
                         "#|\n|| CID | Component | Type | Impact ||\n{}|#\n}}>\n\n"
                         "Полный отчет доступен в системе "
                         "((https://coverity.cloud.yandex.net/reports.htm#v10014/p{} Coverity Connect)).\n\n"
                         "Инструкции по разбору:\n"
                         "1. Оценить релевантность срабатывания. (посмотреть описание бага, возможность его эксплуатации)\n"
                         "  1.1. Прочитать описание бага.\n"
                         "  1.2. Оценить применимость описанного бага на найденном фрагменте кода.\n"
                         "2. Изменить статус срабатывания в поле Classification на один из 'False Positive', 'Intentional', 'Bug' etc.\n"
                         "3. Для всех 'Bug' случаев проставить Severity и Action, завести тикет на исправление и привязать его к этому тикету.\n"
                         "4. Для всех остальных случаев проставить Action: 'Ignore', а для 'Intentional' случаев оставить комментарий в поле "
                         "Comment о причинах намеренного оставления бага в коде.\n"
                         "5. После создания Pull Request на исправления установить статус Action: Fix Submitted, перевести тикет в статус Тестируется.\n"
                         "6. На следующий день проверить отсутствие ошибок из тикета в Coverity Connect по полю CID и закрыть тикет, если ошибки ушли.\n\n"
                         "((https://wiki.yandex-team.ru/cloud/regulations/security-fixes/ Регламент)) управления уязвимостями в Yandex.Cloud.").format(
                             self.project_name, self.count_high, self.count_medium, self.count_low, self.count_audit, self.vulns_table, self.project_key),
            assignee={"id": self.assigne},
            followers=[{
                "id": "ngergel"
            }],
            tags=self.tags,
            components=components,
            priority=self.priority,
            deadline=self.deadline,
        )
        return issue["key"]

    def createSecwareTicket(self):
        from startrek_client import Startrek
        client = Startrek(useragent="RobotYCComplince", token=Secrets().st_token)

        issue = client.issues.create(
            queue=self.queue,
            summary=self.summary,
            description=("Привет!\n\nCтатический анализ нашел потенциальные уязвимости в "
                         "проекте {}.\n\n#|\n|| !!High!! | !!(жел)Medium!! | !!(зел)Low!! | Audit ||\n"
                         "|| {} | {} | {} | {} ||\n|#\n\n\n"
                         "<{{**Список потенциальных уязвимостей под катом:**\n"
                         "#|\n|| CID | Component | Type | Impact ||\n{}|#\n}}>\n\n"
                         "Полный отчет доступен в системе "
                         "((https://coverity.cloud.yandex.net/reports.htm#v10014/p{} Coverity Connect)).\n\n"
                         "Инструкции по разбору:\n"
                         "Оценить релевантность срабатывания. (посмотреть описание бага, возможность его эксплуатации)\n"
                         "1.1. Прочитать описание бага.\n"
                         "1.2. Оценить применимость описанного бага на найденном фрагменте кода.\n"
                         "Изменить статус срабатывания в поле Classification на один из 'False Positive', 'Intentional', 'Bug' etc.\n"
                         "Для всех 'Bug' случаев проставить Severity и Action. Написать комментарий с деталями о проблеме, перевести тикет в "
                         "статус 'Для Обсуждения' - автоматика призовет ответственного разработчика.\n"
                         "Для всех остальных случаев проставить Action: 'Ignore', а для 'Intentional' случаев оставить комментарий в поле "
                         "Comment платформы Coverity Connect о причинах намеренного оставления бага в коде. В случае необходимости получение "
                         "дополнительной информации от разработки - написать вопрос в комментарии к этому тикету и призвать всех коллег из "
                         "списка Наблюдателей, в противном случае закрыть тикет.\n"
                         "((https://wiki.yandex-team.ru/cloud/regulations/security-fixes/ Регламент)) управления уязвимостями в Yandex.Cloud.").format(
                             self.project_name, self.count_high, self.count_medium, self.count_low, self.count_audit, self.vulns_table, self.project_key),
            assignee={"id": "d-lymbin"},
            followers=[{
                "id": "ngergel"
            }, {
                "id": self.assigne
            }],
            employee=self.assigne,
            tags=self.tags,
            priority=self.priority,
            deadline=self.deadline,
        )
        return issue["key"]


class Mail:
    def __init__(self, coverity) -> None:
        self.msg = str("В процессе анализа уязвимостей в Coverity обнаружено:\n")
        if coverity.components_not_in_map:
            self.msg += ("В карте компонент не указаны компоненты в трекере для следующих компонент в Coverity:\n"
                         "{}\n\n"
                         "Ссылка на карту - https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/cloud/compliance/CoverityParserPy3/"
                         "components_map.py\n\n\n").format("\n".join([
                             "- {}: владельцы - {}, проекты - {}.".format(component, ", ".join(item["owners"]), ", ".join(item["projects"]))
                             for component, item in coverity.components_not_in_map.items()
                         ]))
        if coverity.unassigned_vulns:
            self.msg += ("В следующих проектах обнаружены уязвимости без владельца:\n"
                         "{}\n\n").format("\n".join(["- {} - {} шт.".format(project, len(vulns)) for project, vulns in coverity.unassigned_vulns.items()]))

    def sendMail(self):
        msg = MIMEText(self.msg)
        msg["Subject"] = "Автоматизация Coverity"
        msg["From"] = "robot-yc-compliance@yandex-team.ru"
        recipients = ["semen-tarasov@yandex-team.ru", "ngergel@yandex-team.ru"]
        msg["To"] = ", ".join(recipients)
        server = smtplib.SMTP("yabacks.yandex.ru", port=25)
        try:
            server.sendmail(
                "robot-yc-compliance@yandex-team.ru",
                recipients,
                msg.as_string(),
            )
            server.quit()
        except smtplib.SMTPException as exc:
            raise errors.TaskError("Did not send the mail to cloud-compliance {}".format(exc))


class CoverityParser(sdk2.Task):
    def on_execute(self):
        coverity = Coverity()
        for project in coverity.projects.projects_list:
            project_key = project.projectKey
            project_name = project.id.name
            for owner, queues in project.projects_owners.items():
                if owner != "Unassigned":
                    for queue, vulns in queues.items():
                        target_vulns = [vuln for vuln in vulns if coverity.projects.all_vulns[vuln]["component_in_map"] is True]
                        target_vulns_cloud = {
                            "vulnsHighSecurity": {
                                "vulns": set(),
                                "name": "High Security",
                                "tags": ["security", "high"],
                                "deadline": datetime.date.today() + datetime.timedelta(days=1),
                                "priority": "blocker",
                            },
                            "vulnsMediumSecurity": {
                                "vulns": set(),
                                "name": "Medium Security",
                                "tags": ["security", "medium"],
                                "deadline": datetime.date.today() + datetime.timedelta(days=1),
                                "priority": "blocker",
                            },
                        }
                        target_vulns_secware = {
                            "vulnsLowSecurity": {
                                "vulns": set(),
                                "name": "Low Security",
                                "tags": ["security", "low"],
                                "deadline": datetime.date.today() + datetime.timedelta(days=92),
                                "priority": "minor",
                            },
                            "vulnsHighQuality": {
                                "vulns": set(),
                                "name": "High Quality",
                                "tags": ["quality", "high"],
                                "deadline": datetime.date.today() + datetime.timedelta(days=3),
                                "priority": "critical",
                            },
                            "vulnsMediumQuality": {
                                "vulns": set(),
                                "name": "Medium Quality",
                                "tags": ["quality", "medium"],
                                "deadline": datetime.date.today() + datetime.timedelta(days=31),
                                "priority": "normal",
                            },
                            "vulnsLowQuality": {
                                "vulns": set(),
                                "name": "Low Quality",
                                "tags": ["quality", "low"],
                                "deadline": datetime.date.today() + datetime.timedelta(days=92),
                                "priority": "minor",
                            },
                            "vulnsAudit": {
                                "vulns": set(),
                                "name": "Audit",
                                "tags": ["audit"],
                                "deadline": None,
                                "priority": "trivial",
                            },
                        }

                        for target_vuln in target_vulns:
                            if coverity.projects.all_vulns[target_vuln]["kind"] == "Security":
                                if coverity.projects.all_vulns[target_vuln]["impact"] == "High":
                                    target_vulns_cloud["vulnsHighSecurity"]["vulns"].add(target_vuln)
                                elif coverity.projects.all_vulns[target_vuln]["impact"] == "Medium":
                                    target_vulns_cloud["vulnsMediumSecurity"]["vulns"].add(target_vuln)
                                elif coverity.projects.all_vulns[target_vuln]["impact"] == "Low":
                                    target_vulns_secware["vulnsLowSecurity"]["vulns"].add(target_vuln)
                                elif coverity.projects.all_vulns[target_vuln]["impact"] == "Audit":
                                    target_vulns_secware["vulnsAudit"]["vulns"].add(target_vuln)
                            elif coverity.projects.all_vulns[target_vuln]["kind"] == "Quality":
                                if coverity.projects.all_vulns[target_vuln]["impact"] == "High":
                                    target_vulns_secware["vulnsHighQuality"]["vulns"].add(target_vuln)
                                elif coverity.projects.all_vulns[target_vuln]["impact"] == "Medium":
                                    target_vulns_secware["vulnsMediumQuality"]["vulns"].add(target_vuln)
                                elif coverity.projects.all_vulns[target_vuln]["impact"] == "Low":
                                    target_vulns_secware["vulnsLowQuality"]["vulns"].add(target_vuln)
                                elif coverity.projects.all_vulns[target_vuln]["impact"] == "Audit":
                                    target_vulns_secware["vulnsAudit"]["vulns"].add(target_vuln)

                        for type in target_vulns_secware:
                            if target_vulns_secware[type]["vulns"]:
                                ticket = Ticket("CLOUD", target_vulns_secware[type]["vulns"], owner, project_key, project_name, coverity.projects.all_vulns, target_vulns_secware[type]["name"],
                                                target_vulns_secware[type]["deadline"], target_vulns_secware[type]["priority"])
                                ticket.tags.append("secware")
                                ticket.tags += target_vulns_secware[type]["tags"]

                                logging.info("Will create SecWare ticket with params:\n* Queue - {}\n* Assignee - {}\n* Counters - Hight {}, Medium {}, Low {} "
                                             "Audit {}\n* Tags - {}\n* Project - {}\n* Project Key - {}\n* ST Components - {}\n* "
                                             "Tasks Summary - {} ".format(ticket.queue, ticket.assigne, ticket.count_high, ticket.count_medium, ticket.count_low, ticket.count_audit,
                                                                          ", ".join(ticket.tags), ticket.project_name, ticket.project_key, ", ".join(ticket.st_components), ticket.summary))
                                created_ticket = ticket.createSecwareTicket()
                                logging.info("Created ticket {}.".format(created_ticket))
                                for vuln in target_vulns_secware[type]["vulns"]:
                                    coverity.defectServiceClient.setTicketAsExtRef(project.projects_vulns[vuln], created_ticket)

                        for type in target_vulns_cloud:
                            if target_vulns_cloud[type]["vulns"]:
                                ticket = Ticket(queue, target_vulns_cloud[type]["vulns"], owner, project_key, project_name, coverity.projects.all_vulns, target_vulns_cloud[type]["name"],
                                                target_vulns_cloud[type]["deadline"], target_vulns_cloud[type]["priority"])
                                ticket.tags += target_vulns_secware[type]["tags"]
                                logging.info("Will create Cloud ticket with params:\n* Queue - {}\n* Assignee - {}\n* Counters - Hight {}, Medium {}, Low {} "
                                             "Audit {}\n* Tags - {}\n* Project - {}\n* Project Key - {}\n* ST Components - {}\n* "
                                             "Tasks Summary - {} ".format(ticket.queue, ticket.assigne, ticket.count_high, ticket.count_medium, ticket.count_low, ticket.count_audit,
                                                                          ", ".join(ticket.tags), ticket.project_name, ticket.project_key, ", ".join(ticket.st_components), ticket.summary))
                                created_ticket = ticket.createCloudTicket()
                                logging.info("Created ticket {}.".format(created_ticket))
                                for vuln in target_vulns_cloud[type]["vulns"]:
                                    coverity.defectServiceClient.setTicketAsExtRef(project.projects_vulns[vuln], created_ticket)

        if coverity.unassigned_vulns or coverity.components_not_in_map:
            mail = Mail(coverity)
            mail.sendMail()
