# -*- coding: utf-8 -*-

import requests
import json
import datetime

import logging
from sandbox import sdk2
from sandbox.sdk2 import yav
from sandbox.projects.cloud.compliance.EmployeesStatusesParser.config import Config as config


class Secrets:
    def __init__(self):
        secret = yav.Secret("sec-01dvx104f3y3515tmxc6wx3f53")

        self.db_password = secret.data().get("db_password", None)
        self.tvm_secret = secret.data().get("tvm_secret", None)
        self.st_token = secret.data().get("st_token", None)
        self.oauth_token = secret.data().get("token", None)
        self.tvm_ml_ticket = self.get_tvm_ml_ticket()

        assert self.db_password, "Problem with DB passsword!"
        assert self.tvm_secret, "Problem with TVM secret!"
        assert self.st_token, "Problem with ST token!"
        assert self.oauth_token, "Problem with Staff OAuth token!"
        assert self.tvm_ml_ticket, "Problem with TVM service ticket!"

    def get_tvm_ml_ticket(self):
        from tvm2 import TVM2
        from tvm2.protocol import BlackboxClientId

        tvm = TVM2(
            client_id="2018276",
            secret=self.tvm_secret,
            blackbox_client=BlackboxClientId.Prod,
        )
        tvm_ml_ticket = tvm.get_service_tickets("2014406").get("2014406")

        return tvm_ml_ticket


class DataBase:
    def __init__(self, db_password, table_name):
        self.table_name = table_name
        self.db_password = db_password

        self.connection = self.connect()
        self.columns = self.get_columns()
        self.data_dict = self.get_data()

    def connect(self):
        import psycopg2
        SQL_CONNECTION_PARAMS = ("host=c-mdbl1if2iee71tvvbnq8.rw.db.yandex.net\n" "port=6432\n" "dbname=checkdb\n" "user=robot\n" "password={}\n" "target_session_attrs=read-write")

        connection = psycopg2.connect(SQL_CONNECTION_PARAMS.format(self.db_password))
        connection.autocommit = True

        return connection

    def get_columns(self):
        from psycopg2.extras import DictCursor

        cursor = self.connection.cursor(cursor_factory=DictCursor)
        cursor.execute("SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = N'{}'".format(self.table_name))
        request_output = cursor.fetchall()
        columns = [column[3] for column in request_output]
        cursor.close()

        return columns

    def get_data(self):
        from psycopg2.extras import DictCursor

        cursor = self.connection.cursor(cursor_factory=DictCursor)
        cursor.execute("SELECT * FROM {}".format(self.table_name))
        request_output = cursor.fetchall()
        data_dict = {}
        for item in request_output:
            data_dict.setdefault(item[0], dict())
            i = 1
            for column in self.columns[1:]:
                data_dict[item[0]].update({column: item[i]})
                i += 1

        cursor.close()

        return data_dict

    def push_data(self, db_data):
        from psycopg2.extras import DictCursor

        SQL_QUERY_PREPARE = ("PREPARE push (text) "
                             "AS INSERT INTO employees "
                             "VALUES ($1,$2,$3,$4,$5,$6,$7,$8,$9,$10,$11,$12,$13,$14,$15,$16) ON CONFLICT (login) "
                             "DO UPDATE SET login = $1, cloud_relation = $2, staff_department = $3, "
                             "location = $4, position = $5, devel_subscribed = $6, leads_subscribed = $7, "
                             "uid = $8, services = $9, role_check_passed = $10, added = $11, "
                             "last_update = $12, docs_signed = $13, docs_signing_ticket = $14, is_developer = $15, "
                             "is_external = $16;"
                             "EXECUTE push('{0}','{1}','{2}','{3}','{4}','{5}','{6}','{7}','{8}','{9}','{10}',"
                             "'{11}','{12}','{13}','{14}','{15}');")

        logging.info("Pushing data to DB.")
        cursor = self.connection.cursor(cursor_factory=DictCursor)
        for key, value in db_data.items():
            cursor.execute(
                SQL_QUERY_PREPARE.format(key, value["cloud_relation"], value["staff_department"], value["location"], value["position"], value["devel_subscribed"], value["leads_subscribed"],
                                         value["uid"], value["services"], value["role_check_passed"], value["added"], value["last_update"], value["docs_signed"], value["docs_signing_ticket"],
                                         value["is_developer"], value["is_external"]))

        cursor.close()
        self.connection.close()


class API:
    def __init__(self, oauth_token=None, tvm_ticket=None):
        self.exeption = "Problem with API ({}) connection, status code is {}. \n Text:\n{}\nData:\n{}"
        self.headers = dict()
        if oauth_token:
            self.headers["Authorization"] = "OAuth {}".format(oauth_token)
        if tvm_ticket:
            self.headers["X-Ya-Service-Ticket"] = tvm_ticket

    def get(self, url, params=None):
        logging.info("API get {}.".format(url))
        response = requests.get(url=url, headers=self.headers, params=params)
        if response.status_code == 200:
            data = json.loads(response.content)
        else:
            raise Exception(self.exeption.format(url, response.status_code, response.text, params))

        return data

    def post(self, url, data=None):
        logging.info("API post {}.".format(url))
        response = requests.post(url=url, headers=self.headers, data=data)
        if response.status_code == 201 or response.status_code == 200:
            data = json.loads(response.content)
        else:
            raise Exception(self.exeption.format(url, response.status_code, response.text, data))

        return data


class Tracker:
    def __init__(self, st_token):
        from startrek_client import Startrek
        self.client = Startrek(useragent="RobotYCComplince", token=st_token)

    def check_closed(self, ticket):
        logging.info("Checking if closed {}.".format(ticket))
        issue = self.client.issues[ticket]
        if issue.status.key == "closed":
            return True
        else:
            return False


class ML:
    def __init__(self, secrets):
        self.api = API(tvm_ticket=secrets.tvm_ml_ticket)
        self.devel_list, self.leads_list = self.get_lists_subscribers()

    def get_lists_subscribers(self):
        url = "https://ml.yandex-team.ru/apiv3/lists/subscribers?emails=devel@yandex-team.ru,cloud-leads@yandex-team.ru"
        data = self.api.get(url)
        devel_list = [login["login"] for login in data["result"]["devel@yandex-team.ru"]["subscribers"]]
        leads_list = [login["login"] for login in data["result"]["cloud-leads@yandex-team.ru"]["subscribers"]]

        return devel_list, leads_list


class Employees:
    def __init__(self, secrets):
        self.ml = ML(secrets)
        self.tracker = Tracker(secrets.st_token)
        self.api = API(oauth_token=secrets.oauth_token)

        self.employees_data = dict()
        self.clouds_developers = set()

        self.logins_lists = {
            "clouds_employee": self.get_clouds_logins(),
            "connected_services_employee": self.get_connected_logins(),
        }

    def get_clouds_logins(self):
        clouds_logins = set()
        url = "https://abc-back.yandex-team.ru/api/v4/services/members/?is_robot=false&service__with_descendants=1415"
        while True:
            data = self.api.get(url)
            for employee in data["results"]:
                clouds_logins.add(employee["person"]["login"])
                if employee["role"]["code"] == "developer":
                    self.clouds_developers.add(employee["person"]["login"])

            if data["next"]:
                url = data["next"]
            else:
                break

        return ",".join(str(login) for login in clouds_logins)

    def get_connected_logins(self):
        connected_logins = set()
        url = "https://staff-api.yandex-team.ru/v3/groupmembership/?_page={}&_fields=person.login&person.official.is_dismissed=false&person.official.is_robot=false&group.url={}"
        for group_url in config.CONNECTED_GROUPS_URLS:
            page = 1
            while True:
                data = self.api.get(url.format(page, group_url))
                for employee in data["result"]:
                    connected_logins.add(employee["person"]["login"])
                if page < data["pages"]:
                    page += 1
                else:
                    break

        return ",".join(str(login) for login in connected_logins)

    def get_employees_data(self):
        employees_data = dict()
        for list, logins in self.logins_lists.items():
            url = "https://staff-api.yandex-team.ru/v3/persons/"
            data = {"_fields": "uid,login,name,official,department_group,location,groups", "_page": 1, "_sort": "id", "login": logins}

            while True:
                employees_api_data = self.api.post(url, data)
                for employee in employees_api_data["result"]:
                    login = employee["login"]
                    services = set()
                    for service in employee["groups"]:
                        try:
                            services.add(service["group"]["parent"]["name"])
                        except:
                            services.add(service["group"]["name"])
                    employees_data.setdefault(login, dict())
                    if list == "clouds_employee":
                        employees_data[login]["cloud_relation"] = "clouds_employee"
                    else:
                        employees_data[login]["cloud_relation"] = employees_data[login].get("cloud_relation", "connected_services_employee")
                    employees_data[login]["is_developer"] = login in self.clouds_developers
                    employees_data[login]["last_update"] = datetime.date.today()
                    employees_data[login]["staff_department"] = str(employee["department_group"]["name"])
                    employees_data[login]["location"] = str(employee["location"]["office"]["name"]["ru"])
                    employees_data[login]["position"] = str(employee["official"]["position"]["ru"])
                    employees_data[login]["devel_subscribed"] = login in self.ml.devel_list
                    employees_data[login]["leads_subscribed"] = login in self.ml.leads_list
                    employees_data[login]["uid"] = int(employee["uid"])
                    employees_data[login]["services"] = ", ".join(services).replace("'", "")
                    employees_data[login]["is_external"] = employee["official"]["affiliation"] == "external"

                    employees_data[login]["role_check_passed"] = False
                    position = str(employees_data[login]["position"])
                    position = position.decode("utf-8").lower()
                    for role in config.ROLES:
                        if role in position:
                            employees_data[login]["role_check_passed"] = True

                if data["_page"] < employees_api_data["pages"]:
                    data["_page"] += 1
                else:
                    break

        return employees_data

    def update_db_dict(self, employees_db_dict, employees_data):
        for login in employees_data:
            employees_db_dict.setdefault(login, {"docs_signed": False, "docs_signing_ticket": "No information", "added": datetime.date.today()})
            employees_db_dict[login].update(employees_data[login])
            if employees_db_dict[login]["docs_signing_ticket"] != "No information":
                employees_db_dict[login]["docs_signed"] = self.tracker.check_closed(employees_db_dict[login]["docs_signing_ticket"])

        for login in employees_db_dict:
            if employees_db_dict[login]["last_update"] != datetime.date.today():
                employees_db_dict[login]["cloud_relation"] = "out"

        return employees_db_dict


class EmployeesStatusesParser(sdk2.Task):
    def on_execute(self):
        secrets = Secrets()
        employees = Employees(secrets)
        employees_db = DataBase(secrets.db_password, "employees")
        employees_data = employees.get_employees_data()

        updated_employees_db = employees.update_db_dict(employees_db.data_dict, employees_data)
        employees_db.push_data(updated_employees_db)
