# -*- coding: utf-8 -*-

import logging
import datetime
import requests

from sandbox import sdk2
from sandbox.sdk2 import yav

from sandbox.projects.cloud.compliance.TicketsCreator.config import Config


class Secrets:
    def __init__(self):
        self.secret = yav.Secret("sec-01dvx104f3y3515tmxc6wx3f53")

        self.st_token = self.secret.data().get("st_token", None)
        assert self.st_token, "Problem with ST token!"

        self.db_password = self.secret.data().get("db_password", None)
        assert self.db_password, "Problem with DB password!"

        self.tvm_secret = self.secret.data().get("tvm_secret", None)
        assert self.tvm_secret, "Problem with TVM secret!"


class DataBase:
    def __init__(self, secrets, table_name):
        self.table_name = table_name
        self.db_password = secrets.db_password

        self.connection = self.connect()
        self.columns = self.get_columns()
        self.data_dict = self.get_data()

    def connect(self):
        import psycopg2

        connection_params = ("host=c-mdbl1if2iee71tvvbnq8.rw.db.yandex.net\n"
                             "port=6432\n"
                             "dbname=checkdb\n"
                             "user=robot\n"
                             "password={}\n"
                             "target_session_attrs=read-write").format(self.db_password)

        connection = psycopg2.connect(connection_params)
        connection.autocommit = True

        return connection

    def get_columns(self):
        from psycopg2.extras import DictCursor
        cursor = self.connection.cursor(cursor_factory=DictCursor)
        cursor.execute("SELECT * FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_NAME = N'{}'".format(self.table_name))
        request_output = cursor.fetchall()
        columns = [column[3] for column in request_output]
        cursor.close()

        return columns

    def get_data(self):
        from psycopg2.extras import DictCursor
        cursor = self.connection.cursor(cursor_factory=DictCursor)
        cursor.execute("SELECT * FROM {}".format(self.table_name))
        request_output = cursor.fetchall()
        data_dict = {}
        for item in request_output:
            data_dict.setdefault(item[0], dict())
            i = 1
            for column in self.columns[1:]:
                data_dict[item[0]].update({column: item[i]})
                i += 1

        cursor.close()

        return data_dict


class MoeApi:
    def __init__(self, tvm_secret):
        self.tvm_secret = tvm_secret
        service_tickets = self.tvm_tickets_get()
        self.moe_tvm_ticket = service_tickets.get("2001279")

    def reset_courses_results(self, course_id, login):
        logging.info("Reseting results for {}, course {}.".format(login, course_id))
        response = requests.delete(
            url="https://api.moe.yandex-team.ru/api/v3/results/",
            headers={
                "X-Ya-Service-Ticket": self.moe_tvm_ticket,
                "Content-Type": "application/json"
            },
            json={
                "course_id": course_id,
                "username": login,
            },
        )
        if response.status_code != 204:
            raise Exception("Problem with moe api!")

    def tvm_tickets_get(self):
        from tvm2 import TVM2
        from tvm2.protocol import BlackboxClientId

        tvm = TVM2(
            client_id="2018276",
            secret=self.tvm_secret,
            blackbox_client=BlackboxClientId.Prod,
        )

        service_tickets = tvm.get_service_tickets("2001279")

        return service_tickets


class Tracker:
    def __init__(self, secrets):
        from startrek_client import Startrek
        self.st_token = secrets.st_token
        self.client = Startrek(useragent="RobotYCComplince", token=self.st_token)

        self.moe = MoeApi(secrets.tvm_secret)

    def create_ticket(self, summary, description, tag, login):
        logging.info("Creating {} ticket for {}.".format(tag, login))
        issue = self.client.issues.create(
            queue="CLOUD",
            summary=summary,
            description=description,
            followers={"id": "semen-tarasov"},
            tags=[tag],
            assignee={"id": login},
        )
        logging.info("Created ticket {}.".format(issue.key))
        return issue.key

    def create_courses_tickets(self, employees_db, courses_db, tickets_db):
        for login, employee_values in employees_db.data_dict.items():
            courses_lines = list()
            repassing_courses_lines = list()
            if login in courses_db.data_dict.keys():
                if employee_values["cloud_relation"] == "clouds_employee" and employee_values["is_external"] is False:
                    if courses_db.data_dict[login]["15_result"] < 70:
                        courses_lines.append(Config.COURSES_TASK_STR_15)
                    elif courses_db.data_dict[login]["15_result"] > 70 and courses_db.data_dict[login]["15_date"] < datetime.date.today() - datetime.timedelta(days=365):
                        self.moe.reset_courses_results(15, login)
                        repassing_courses_lines.append("{}, надо проходить ежегодно, пройден {}.".format(Config.COURSES_TASK_STR_15, courses_db.data_dict[login]["15_date"]))
                    if employee_values["is_developer"] is True and courses_db.data_dict[login]["1010_result"] < 70:
                        courses_lines.append(Config.COURSES_TASK_STR_1010)
                    elif courses_db.data_dict[login]["1010_result"] > 70 and courses_db.data_dict[login]["1010_date"] < datetime.date.today() - datetime.timedelta(days=365):
                        self.moe.reset_courses_results(1010, login)
                        repassing_courses_lines.append("{}, надо проходить ежегодно, пройден {}".format(Config.COURSES_TASK_STR_1010, courses_db.data_dict[login]["1010_date"]))
                if (employee_values["cloud_relation"] != "out" and employee_values["staff_department"] in Config.ITDC_COURSE_LIST and "ежурный" in employee_values["position"]
                        and courses_db.data_dict[login]["1846_result"] < 70):
                    courses_lines.append(Config.COURSES_TASK_STR_1846)
                elif (employee_values["cloud_relation"] != "out" and employee_values["staff_department"] in Config.ITDC_COURSE_LIST and "ежурный" in employee_values["position"]
                      and courses_db.data_dict[login]["1846_result"] > 70 and courses_db.data_dict[login]["1846_date"] < datetime.date.today() - datetime.timedelta(days=365)):
                    self.moe.reset_courses_results(1846, login)
                    repassing_courses_lines.append("{}, надо проходить ежегодно, пройден {}".format(Config.COURSES_TASK_STR_1846, courses_db.data_dict[login]["1846_date"]))
            else:
                if employee_values["cloud_relation"] == "clouds_employee" and employee_values["is_external"] is False:
                    courses_lines.append(Config.COURSES_TASK_STR_15)
                    if employee_values["is_developer"] is True:
                        courses_lines.append(Config.COURSES_TASK_STR_1010)
                if employee_values["cloud_relation"] != "out" and employee_values["staff_department"] in Config.ITDC_COURSE_LIST and "ежурный" in employee_values["position"]:
                    courses_lines.append(Config.COURSES_TASK_STR_1846)

            if len(courses_lines) != 0:
                ticket_created = False
                for ticket_values in tickets_db.data_dict.values():
                    if ticket_values["assignee"] == login and ticket_values["tag"] in ("compliance_courses", "compliance_repassing_courses") and ticket_values["status"] != "Закрыт":
                        ticket_created = True
                if ticket_created is False:
                    self.create_ticket(Config.COURSES_TASK_SUMMARY, Config.COURSES_TASK_DESCRIPTION.format("\n".join(courses_lines)), Config.COURSES_TASK_TAG, login)

            if len(repassing_courses_lines) != 0:
                ticket_created = False
                for ticket_values in tickets_db.data_dict.values():
                    if ticket_values["assignee"] == login and ticket_values["tag"] == "compliance_repassing_courses" and ticket_values["status"] != "Закрыт":
                        ticket_created = True
                if ticket_created is False:
                    self.create_ticket(Config.COURSES_REPASSING_TASK_SUMMARY, Config.COURSES_REPASSING_TASK_DESCRIPTION.format("\n".join(repassing_courses_lines)), Config.COURSES_REPASSING_TASK_TAG,
                                       login)

    def create_encryption_ticket(self, employees_db, computers_db, tickets_db):
        for host, computer_values in computers_db.data_dict.items():
            if computer_values["encrypted"] is False:
                login = computer_values["login"]
                if login in employees_db.data_dict.keys():
                    if employees_db.data_dict[login]["cloud_relation"] == "clouds_employee" and employees_db.data_dict[login]["is_external"] is False:
                        ticket_created = False
                        for ticket_values in tickets_db.data_dict.values():
                            if ticket_values["assignee"] == login and ticket_values["tag"] == "compliance_encryption" and ticket_values["status"] != "Закрыт":
                                ticket_created = True
                        if ticket_created is False:
                            self.create_ticket(Config.ENCRYPTION_TASK_SUMMARY, Config.ENCRYPTION_TASK_DESCRIPTION.format(host, computer_values["serial"]), Config.ENCRYPTION_TASK_TAG, login)

    def create_role_ticket(self, employees_db, tickets_db):
        for login, employee_values in employees_db.data_dict.items():
            if employee_values["cloud_relation"] == "clouds_employee" and employee_values["role_check_passed"] is False and employee_values["is_external"] is False:
                ticket_created = False
                for ticket_values in tickets_db.data_dict.values():
                    if ticket_values["assignee"] == login and ticket_values["tag"] == "compliance_role" and ticket_values["status"] != "Закрыт":
                        ticket_created = True
                if ticket_created is False:
                    self.create_ticket(Config.ROLE_TASK_SUMMARY, Config.ROLE_TASK_DESCRIPTION.format(employee_values["position"]), Config.ROLE_TASK_TAG, login)


class TicketsCreator(sdk2.Task):
    def on_execute(self):
        secrets = Secrets()
        employees_db = DataBase(secrets, "employees")
        courses_db = DataBase(secrets, "courses")
        tickets_db = DataBase(secrets, "tickets")
        computers_db = DataBase(secrets, "computers")

        tracker = Tracker(secrets)
        tracker.create_courses_tickets(employees_db, courses_db, tickets_db)
        tracker.create_encryption_ticket(employees_db, computers_db, tickets_db)
        tracker.create_role_ticket(employees_db, tickets_db)
