# -*- coding: utf-8 -*-

import logging
import datetime
import requests
import json

from sandbox import sdk2
from sandbox.sdk2 import yav


class Secrets:
    def __init__(self):
        secret = yav.Secret("sec-01dvx104f3y3515tmxc6wx3f53")

        self.st_token = secret.data().get("st_token", None)
        self.staff_oauth_token = secret.data().get("token", None)
        self.idm_oauth_token = secret.data().get("idm_oauth_token", None)

        assert self.st_token, "Problem with ST token!"
        assert self.staff_oauth_token, "Problem with Staff OAuth token!"
        assert self.idm_oauth_token, "Problem with IDM OAuth token!"


class IDM:
    def __init__(self, secrets):
        self.secrets = secrets
        self.headers = {}
        self.headers["Authorization"] = "OAuth {}".format(self.secrets.idm_oauth_token)
        self.headers["Content-Type"] = "application/json"

    def add_clouds_roles(self, targets):
        for target in targets:
            clouds_roles_ids, clouds_roles_names = self.find_cloud_roles(targets[target]["login"])

            targets[target]["clouds_roles_ids"] = clouds_roles_ids
            targets[target]["clouds_roles_names"] = clouds_roles_names

        return targets

    def find_cloud_roles(self, login):
        api_url = "https://idm-api.yandex-team.ru/api/v1/roles/"
        api_params = {
            "user": login,
            "system": "abc",
            "state": "granted",
            "ownership": "personal",
        }
        response = requests.get(url=api_url, headers=self.headers, params=api_params)
        if response.status_code == 200:
            data = json.loads(response.content)
            clouds_roles_ids = [role["id"] for role in data["objects"] if role["human_short"][42:55] == "Яндекс.Облако"]
            clouds_roles_names = [role["human_short"][42:] for role in data["objects"] if role["human_short"][42:55] == "Яндекс.Облако"]
            pass

            return clouds_roles_ids, clouds_roles_names
        else:
            raise Exception("Problem with API ({}) connection, status code is {}. \n Text:\n{}\nData:\n{}".format(api_url, response.status_code, response.text, api_params))

    def separate_clouds_roles(self, targets):
        for values in targets.values():
            if values["need_separate_roles"]:
                logging.info("Start trying separate roles.")
                self.separate_roles(values["clouds_roles_ids"])

    def separate_roles(self, clouds_roles_ids):
        api_url = "https://idm-api.yandex-team.ru/api/v1/roles/{}/"

        for role in clouds_roles_ids:
            logging.info("Trying separate role {}.".format(role))
            response = requests.request("DELETE", url=api_url.format(role), headers=self.headers)
            if response.status_code == 204:
                logging.info("Role {} separated.".format(role))
            else:
                raise Exception("Problem with API ({}) connection, status code is {}. \n Text:\n{}\n".format(api_url, response.status_code, response.text))


class Tracker:
    TARGETS_MAP = {"test": {"queue": "EDS", "tags": "compliance_awareness", "status": '"Документы на подписании"'}}

    def __init__(self, secrets):
        from startrek_client import Startrek
        self.secrets = secrets
        self.client = Startrek(useragent="RobotYCComplince", token=self.secrets.st_token)

    def find_targets(self):
        logging.info("Finding targets.")
        for params in self.TARGETS_MAP.values():
            logging.info('Queue: {} Tags: {} Status: {}'.format(params["queue"], params["tags"], params["status"]))
            tickets = dict()
            issues = self.client.issues.find(
                query='Queue: {} Status: {} Tags: {} Tags: ! compliance_exclusion'.format(params["queue"], params["status"], params["tags"]),
                per_page=10000,
            )

            for issue in issues:
                tickets[issue.key] = {"login": issue.employee.login, "deadline": issue.deadline, "tags": issue.tags}
                tickets[issue.key]["need_first_notify"] = (issue.deadline == (datetime.date.today() + datetime.timedelta(days=7)).strftime("%Y-%m-%d")
                                                           and ("compliance_first_reminder" not in issue.tags) and ("compliance_second_reminder" not in issue.tags)
                                                           and ("compliance_role_separated" not in issue.tags))
                tickets[issue.key]["need_second_notify"] = (issue.deadline == (datetime.date.today() + datetime.timedelta(days=4)).strftime("%Y-%m-%d") and ("compliance_first_reminder" in issue.tags)
                                                            and ("compliance_second_reminder" not in issue.tags) and ("compliance_role_separated" not in issue.tags))
                tickets[issue.key]["need_separate_roles"] = (issue.deadline == (datetime.date.today()).strftime("%Y-%m-%d") and ("compliance_second_reminder" in issue.tags)
                                                             and ("compliance_first_reminder" not in issue.tags) and ("compliance_role_separated" not in issue.tags))
                tickets[issue.key]["manager"] = self.find_manager(issue.employee.login)
                tickets[issue.key]["deadline_str"] = datetime.datetime.strptime(issue.deadline, '%Y-%m-%d').strftime('%d.%m.%Y')

            targets = tickets.copy()
            for ticket, ticket_params in tickets.items():
                if not ticket_params["need_first_notify"] and not ticket_params["need_second_notify"] and not ticket_params["need_separate_roles"]:
                    targets.pop(ticket)

        return targets

    def put_comments(self, targets):
        NOTIFY_COMMENT = "Привет. \nНапоминаем про необходимость подписать лист ознакомления.\n"\
            "!!В случае, если ознакомление не будет завершено до {}, все роли в Облачных сервисах (находящихся в иерархии ABC под "\
            "((https://abc.yandex-team.ru/services/cloud/ Облаком))) будут отозваны.!!\n\n**Если документ был подписан и передан "\
            "коллегам из HR, но тикет все еще открыт, просьба указать в коментариях дату и место передачи подписанного листа и "\
            "призвать semen-tarasov@ и krasnovaek@.**\n\n"
        SEPARATE_COMMENT = "Привет. \nВ связи с тем, что лист ознакомления не был подписан и призывы были проигнорированы, следующие Облачные роли отзываются:\n{}\n\n"\
            "**Повторная выдача Облачных ролей возможна только после подписания листа ознакомления.**"

        for ticket, params in targets.items():
            logging.info("Adding comments.")

            if params["need_first_notify"]:
                logging.info("Adding first notify comment to {}.".format(ticket))
                issue = self.client.issues[ticket]

                issue["tags"].append("compliance_first_reminder")
                issue.update(tags=issue["tags"])

                issue.comments.create(text=NOTIFY_COMMENT.format(params["deadline_str"]), summonees=params["login"])
            elif params["need_second_notify"]:
                logging.info("Adding second notify comment to {}.".format(ticket))
                issue = self.client.issues[ticket]

                issue["tags"].append("compliance_second_reminder")
                issue["tags"].remove("compliance_first_reminder")
                issue.update(tags=issue["tags"])

                issue.comments.create(text=NOTIFY_COMMENT.format(params["deadline_str"]), summonees=[params["login"], params["manager"]])
            elif params["need_separate_roles"]:
                logging.info("Adding separate notify comment to {}.".format(ticket))
                issue = self.client.issues[ticket]

                issue["tags"].append("compliance_role_separated")
                issue["tags"].remove("compliance_second_reminder")
                issue.update(tags=issue["tags"])

                issue.comments.create(text=SEPARATE_COMMENT.format("\n".join(params["clouds_roles_names"])), summonees=[params["login"], "semen-tarasov", params["manager"]])

    def find_manager(self, login):
        headers = {}
        headers["Authorization"] = "OAuth {}".format(self.secrets.staff_oauth_token)

        api_url = "https://staff-api.yandex-team.ru/v3/persons/"
        api_params = {
            "_fields": "chief.login",
            "login": login,
        }
        response = requests.post(url=api_url, headers=headers, data=api_params)
        if response.status_code == 201 or response.status_code == 200:
            data = json.loads(response.content)
            manager = data["result"][0]["chief"]["login"]
        else:
            raise Exception("Problem with API ({}) connection, status code is {}. \n Text:\n{}\nData:\n{}".format(api_url, response.status_code, response.text, api_params))

        return manager


class Mail:
    def send_mails(self, targets):
        import smtplib
        from email.mime.text import MIMEText

        for ticket, params in targets.items():
            if params["need_first_notify"]:
                text = "Привет. \nНапоминаем про необходимость подписать лист ознакомления в рамках тикета {}.\n"\
                    "В случае, если ознакомление не будет завершено до {}, все роли в Облачных сервисах (находящихся в иерархии ABC под "\
                    "Облаком) будут отозваны.\nЕсли документ был подписан и передан коллегам из HR, но тикет все еще открыт, просьба "\
                    "указать в коментариях дату и место передачи подписанного листа и призвать semen-tarasov@ и krasnovaek@.\n\n"\
                    "P.S. Данное сообщение сгенерировано автоматически. В случае, если в нем есть какая-то ошибка, или "\
                    "ты считаешь, что оно было отправлено тебе по ошибке, просьба написать об этом на рассылку cloud-compliance@yandex-team.ru "\
                    "или в телеграм-чат YC Compliance.".format(ticket, params["deadline_str"])
                targets_mails = ["{}@yandex-team.ru".format(params["login"]), "semen-tarasov@yandex-team.ru"]
            elif params["need_second_notify"]:
                text = "Привет. \nНапоминаем про необходимость подписать лист ознакомления в рамках тикета {}.\n"\
                    "В случае, если ознакомление не будет завершено до {}, все роли в Облачных сервисах (находящихся в иерархии ABC под "\
                    "Облаком) у {} будут отозваны.\nЕсли документ был подписан и передан коллегам из HR, но тикет все еще открыт, просьба "\
                    "указать в коментариях дату и место передачи подписанного листа и призвать semen-tarasov@ и krasnovaek@.\n\n"\
                    "P.S. Данное сообщение сгенерировано автоматически. В случае, если в нем есть какая-то ошибка, или "\
                    "ты считаешь, что оно было отправлено тебе по ошибке, просьба написать об этом на рассылку cloud-compliance@yandex-team.ru "\
                    "или в телеграм-чат YC Compliance.".format(ticket, params["deadline_str"], params["login"])
                targets_mails = ["{}@yandex-team.ru".format(params["login"]), "{}@yandex-team.ru".format(params["manager"]), "semen-tarasov@yandex-team.ru"]
            elif params["need_separate_roles"]:
                text = "Привет. \nВ связи с отсутствием ознакомления с Облачными ЛНА в рамках {} следующие Облачные доступы у {} в ABC отзываются:\n{}\n\n"\
                    "P.S. Данное сообщение сгенерировано автоматически. В случае, если в нем есть какая-то ошибка, или "\
                    "ты считаешь, что оно было отправлено тебе по ошибке, просьба написать об этом на рассылку cloud-compliance@yandex-team.ru "\
                    "или в телеграм-чат YC Compliance.".format(ticket, params["login"], "\n".join(params["clouds_roles_names"]))
                targets_mails = ["{}@yandex-team.ru".format(params["login"]), "semen-tarasov@yandex-team.ru", "ngergel@yandex-team.ru", "{}@yandex-team.ru".format(params["manager"])]
            logging.info("Sending mail to {}, ticket {}.".format(params["login"], ticket))
            msg = text
            msg = MIMEText(msg)
            msg["Subject"] = "Ознакомление с ЛНА Яндекс.Облака"
            msg["From"] = "robot-yc-compliance@yandex-team.ru"
            msg["To"] = ", ".join(targets_mails)
            server = smtplib.SMTP("yabacks.yandex.ru", port=25)
            try:
                server.sendmail("robot-yc-compliance@yandex-team.ru", targets_mails, msg.as_string())
                server.quit()
                logging.info("Mail send.")
            except smtplib.SMTPException as exc:
                raise Exception("Did not send the mail to cloud-compliance {}".format(exc))


class TicketsEscalation(sdk2.Task):
    def on_execute(self):
        secrets = Secrets()
        idm = IDM(secrets)
        tracker = Tracker(secrets)

        targets = tracker.find_targets()
        targets = idm.add_clouds_roles(targets)

        tracker.put_comments(targets)
        Mail().send_mails(targets)

        idm.separate_clouds_roles(targets)
