from sandbox import sdk2

from sandbox.projects.cloud.iam.common import YcIamTask
from sandbox.sandboxsdk import environments


class BlockedAccountsConsumption(YcIamTask):
    class Parameters(YcIamTask.Parameters):
        kill_timeout = 7200  # Task timeout in seconds
        juggler_service_name = YcIamTask.Parameters.juggler_service_name(
            default='blocked-accounts-consumption'
        )
        ignored_billing_account_ids = sdk2.parameters.List(
            'Ignored billing account ids',
            sdk2.parameters.String,
            required=False
        )
        yt_cluster = sdk2.parameters.String(
            'YT cluster',
            default='hahn',
            required=True
        )
        yql_token_name = sdk2.parameters.String(
            'YQL Token secret name',
            required=True
        )

    class Requirements(YcIamTask.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('yql')
        )

    def on_execute(self):
        from yql.api.v1.client import YqlClient
        yql_token = sdk2.Vault.data(self.owner, self.Parameters.yql_token_name)
        yt_proxy = self.Parameters.yt_cluster

        yql_client = YqlClient(db=yt_proxy, token=yql_token)

        result, share_url = find_blocked_accounts_consumption(
            yql_client=yql_client, ignored_billing_account_ids=self.Parameters.ignored_billing_account_ids)

        ok = True
        description = '{}'.format(share_url)
        for row in result:
            if row['today_blocked_hour_plus_3_plus_consumption'] > 0:
                ok = False
                description = '{}\n{}: {}RUB {}'.format(description,
                                                        row['billing_account_id'],
                                                        row['today_blocked_hour_plus_3_plus_consumption'],
                                                        map(lambda x: x.encode('ascii'), row['sku_names']))

        self.send_event_to_juggler('OK' if ok else 'CRIT', self.Parameters.juggler_host,
                                   self.Parameters.juggler_service_name, description)


def find_blocked_accounts_consumption(yql_client, ignored_billing_account_ids):
    query = """
    $accounts_history_table = "//home/cloud/billing/exported-billing-tables/billing_accounts_history_prod";
    $analytics_cube_table = "//home/cloud/billing/analytics_cube/realtime/prod";
    $services_table = "//home/cloud/billing/exported-billing-tables/services_prod";
    $skus_table = "//home/cloud/billing/exported-billing-tables/skus_prod";
    $window= 3600 * 3;
    $ignored_billing_account_ids = [{ignored_ba_ids}];

    $blocked_accounts = (
      SELECT billing_account_id,
             MAX(updated_at) AS blocked_at
      FROM (
        SELECT billing_account_id,
               state,
               LAG(state) OVER w AS previous_state,
               LAST_VALUE(state) OVER w AS current_state,
               updated_at
        FROM $accounts_history_table
        WHERE billing_account_id NOT IN $ignored_billing_account_ids
        WINDOW w AS (
          PARTITION BY billing_account_id
          ORDER BY updated_at
          ROWS BETWEEN UNBOUNDED PRECEDING AND UNBOUNDED FOLLOWING
        )
      )
      WHERE current_state NOT IN ('active', 'payment_required')
        AND (previous_state IS NULL OR previous_state IN ('active', 'payment_required'))
        AND state NOT IN ('active', 'payment_required')
      GROUP BY billing_account_id
    );

    $sku_name_whitelist = (
      SELECT AGGREGATE_LIST_DISTINCT(name)
      FROM $skus_table
      WHERE (name LIKE '%cpu%' OR name LIKE '%ram%') AND name NOT LIKE '%commitment%' AND name NOT LIKE 'ydb.cluster.%'
    );

    $start_ts = DateTime::ToSeconds(CAST(CurrentUtcTimestamp() AS DateTime) - Interval('PT12H'));

    SELECT billing.billing_account_id AS billing_account_id,
           Math::Round(SUM(CASE WHEN billing.billing_record_end_time >= blocked.blocked_at
                                    AND billing.billing_record_end_time < blocked.blocked_at + $window
                                THEN billing.billing_record_cost_rub
                                ELSE 0
                           END
           ), -2) AS blocked_hour_plus_3_consumption,
           Math::Round(SUM(CASE WHEN billing.billing_record_end_time >= blocked.blocked_at + $window
                                THEN billing.billing_record_cost_rub
                                ELSE 0
                           END
           ), -2) AS blocked_hour_plus_3_plus_consumption,
           Math::Round(SUM(CASE WHEN billing.billing_record_end_time >= blocked.blocked_at + $window
                                    AND billing.billing_record_end_time >= $start_ts
                                THEN billing.billing_record_cost_rub
                                ELSE 0
                           END
           ), -2) AS today_blocked_hour_plus_3_plus_consumption,
           AGGREGATE_LIST_DISTINCT(CASE WHEN billing.billing_record_end_time >= blocked.blocked_at + $window
               AND billing.billing_record_end_time >= $start_ts THEN billing.sku_name ELSE NULL END) AS sku_names
    FROM $analytics_cube_table AS billing
      JOIN $blocked_accounts AS blocked ON blocked.billing_account_id = billing.billing_account_id
    WHERE billing.billing_record_end_time >= blocked.blocked_at
      AND ListHas($sku_name_whitelist, billing.sku_name)
    GROUP BY billing.billing_account_id
    HAVING
        SUM(CASE WHEN billing.billing_record_end_time >= $start_ts THEN billing.billing_record_cost_rub ELSE 0 END) > 0
    ORDER BY today_blocked_hour_plus_3_plus_consumption DESC;
    """

    request = yql_client.query(query.format(
        ignored_ba_ids=", ".join(["'" + ba_id + "'" for ba_id in ignored_billing_account_ids])
    ), syntax_version=1)
    request.run()
    result = request.get_results()
    if not result.is_success:
        raise RuntimeError(
            "YQL Query is not succeeded. Operation id: {}. Errors: \n{}".format(str(request.operation_id), "\n".join(
                issue.format_issue() for issue in result.errors)))

    res = []
    for table in result:
        columns = table.columns
        rows = table.rows
        keys = map(lambda (key, _): key, columns)

        for row in rows:
            res.append(dict(zip(keys, row)))

    return res, request.share_url
