import logging
import datetime
import sys

from sandbox import sdk2
from sandbox.projects.common.binary_task import LastBinaryTaskRelease, binary_release_parameters
from dateutil import parser as time_parser


def parse_date(date_str):
    return time_parser.parse(date_str).date()


class PlanningTool(LastBinaryTaskRelease, sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        bert = binary_release_parameters(stable=True)

        revision = sdk2.parameters.Integer('revision', default_value=0)
        checkout_arcadia_from_url = sdk2.parameters.String(
            'svn url for arcadia:',
            default_value='arcadia:/arc/trunk/arcadia/'
        )
        config_path = sdk2.parameters.String(
            'config path',
            default_value='cloud/iam/planning_tool/configs/iam.yaml'
        )

        release = sdk2.parameters.Bool('make a release page /{date}', default_value=False)
        token = sdk2.parameters.YavSecret('token')
        mds_key = sdk2.parameters.YavSecret('mds_key_id and mds_key')

    def on_execute(self):
        from cloud.iam.planning_tool.library import config
        from cloud.iam.planning_tool.library import report

        logging.info('Started: {}'.format(datetime.datetime.now()))

        config_path = self.Parameters.config_path
        separate_dir_index = config_path.rfind('/')
        config_dir = config_path[:separate_dir_index]
        config_name = config_path[separate_dir_index + 1:]

        url = '/'.join((self.Parameters.checkout_arcadia_from_url, config_dir))
        config_path = sdk2.svn.Arcadia.checkout(url, 'config', depth='files', revision=self.Parameters.revision)
        logging.info('Checkout config from revision = {}'.format(self.Parameters.revision))
        logging.info('config_path = {}'.format(config_path))
        config_file_name = '/'.join((config_path, config_name))
        self.config = config.Config(config_file_name,
                                    self.Parameters.token.data()['token'],
                                    self.Parameters.mds_key.data()['mds_key_id'],
                                    self.Parameters.mds_key.data()['mds_key'])
        self.util_builder = report.UtilBuilder(self.config)

    def build_and_upload(self, release_date=None):
        self.util_builder.build().upload(release_date=release_date)


class PlanningToolIamDaily(PlanningTool):
    class Parameters(PlanningTool.Parameters):
        use_current_date = sdk2.parameters.Bool('use current date', default=False)
        with use_current_date.value[False]:
            date = sdk2.parameters.String('report date', default=datetime.date.today().isoformat())
        shuffle = sdk2.parameters.Bool('shuffle', default=False)

    def on_execute(self):
        super(PlanningToolIamDaily, self).on_execute()
        from cloud.iam.planning_tool.library import daily

        date = datetime.date.today() if self.Parameters.use_current_date else parse_date(self.Parameters.date)
        logging.info('Generating Daily Report, date: {}'.format(date))
        self.util_builder.set_report_cls(daily.DailyReport)
        self.util_builder.prepare_from_web(date, self.Parameters.shuffle)
        self.build_and_upload(release_date=date if self.Parameters.release else None)


class PlanningToolIamMonthly(PlanningTool):
    class Parameters(PlanningTool.Parameters):
        from_date = sdk2.parameters.String('from date (inclusively), e.g. 2019-11-17', required=True)
        to_date = sdk2.parameters.String('to date (exclusively), e.g. 2020-01-31',
                                         default=datetime.datetime.today().isoformat())
        new_tag = sdk2.parameters.String('new_tag', default=None)
        prev_tag = sdk2.parameters.String('prev_tag', default=None)

    def on_execute(self):
        super(PlanningToolIamMonthly, self).on_execute()
        from cloud.iam.planning_tool.library import monthly

        from_date = parse_date(self.Parameters.from_date)
        to_date = parse_date(self.Parameters.to_date)

        logging.info('Generating Monthly Report, from date: {}, to date: {}'.format(from_date, to_date))
        self.util_builder.set_report_cls(monthly.MonthlyReport)
        self.util_builder.prepare_from_web(from_date, to_date, self.Parameters.new_tag, self.Parameters.prev_tag)
        self.build_and_upload(release_date=to_date if self.Parameters.release else None)


class PlanningToolIamTimesheet(PlanningTool):
    class Parameters(PlanningTool.Parameters):
        from_date = sdk2.parameters.String('from date (inclusively), e.g. 2019-11-17', required=True)
        to_date = sdk2.parameters.String('to date (exclusively), e.g. 2020-01-31',
                                         default=datetime.datetime.today().isoformat())

    def on_execute(self):
        super(PlanningToolIamTimesheet, self).on_execute()
        from cloud.iam.planning_tool.library import timesheet

        from_date = parse_date(self.Parameters.from_date)
        to_date = parse_date(self.Parameters.to_date)

        logging.info('Generating Timesheet Report, from date: {}, to date {}'.format(from_date, to_date))
        self.util_builder.set_report_cls(timesheet.Timesheet)
        self.util_builder.prepare_from_web(from_date, to_date)
        self.build_and_upload(release_date=to_date if self.Parameters.release else None)


class PlanningToolIamTasks(PlanningTool):
    class Parameters(PlanningTool.Parameters):
        from_date = sdk2.parameters.String('created from date (inclusively), e.g. 2019-11-17')
        show_closed = sdk2.parameters.Bool('show closed tasks', default=False)

    def on_execute(self):
        super(PlanningToolIamTasks, self).on_execute()
        from cloud.iam.planning_tool.library import tasks

        created_from_date = parse_date(self.Parameters.from_date)

        logging.info(
            'Generating Tasks Report, created from date: {}, show closed: {}'.format(created_from_date,
                                                                                     self.Parameters.show_closed))
        self.util_builder.set_report_cls(tasks.TaskReport)
        self.util_builder.prepare_from_web(created_from_date, self.Parameters.show_closed)
        self.build_and_upload(datetime.date.today() if self.Parameters.release else None)
