import logging
import time
import traceback
import uuid

import requests

from sandbox import sdk2


def find_build(job, token, trigger_id):
    """
    Find build by trigger_id
    """
    deadline = time.time() + 1800
    headers = {'Authorization': 'OAuth {token}'.format(token=token)}
    while time.time() < deadline:
        try:
            builds = requests.get(
                'https://jenkins.db.yandex-team.ru/job/{job}/api/json'.format(job=job), headers=headers
            ).json()['builds']
            for build in builds:
                build_url = '{url}api/json'.format(url=build['url'])
                build_info = requests.get(build_url, headers=headers).json()
                for action in build_info['actions']:
                    if action['_class'] == 'hudson.model.ParametersAction':
                        for parameter in action['parameters']:
                            if parameter['name'] == 'trigger_id' and parameter['value'] == trigger_id:
                                return '{url}api/json'.format(url=build['url'])
        except Exception:
            logging.error('Error on find build: %s', traceback.format_exc())
        time.sleep(5)

    raise RuntimeError('Unable to find running build within timeout (30 minutes)')


def poll_build(token, build_url, timeout):
    """
    Poll jenkins build until terminate state
    """
    deadline = time.time() + timeout
    headers = {'Authorization': 'OAuth {token}'.format(token=token)}
    while time.time() < deadline:
        try:
            build_info = requests.get(build_url, headers=headers).json()
            if build_info['result'] is not None:
                if build_info['result'] == 'SUCCESS':
                    return True
                return False
        except Exception:
            logging.error('Error on poll build: %s', traceback.format_exc())
        time.sleep(5)

    raise RuntimeError('Poll build timeout ({timeout} seconds)'.format(timeout=timeout))


class MdbJenkinsCiCommon(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        job = sdk2.parameters.String("jenkins job name", required=True)
        is_post = sdk2.parameters.Bool("is this job a post commit one", required=False)
        arcadia_patch = sdk2.parameters.String("arcadia patch value", required=False)
        poll_timeout = sdk2.parameters.Integer("poll timeout of jenkins job", default=6 * 60 * 60)

    def on_execute(self):
        logging.info('arcadia_patch: {patch}'.format(patch=self.Parameters.arcadia_patch))
        if not self.Parameters.arcadia_patch and not self.Parameters.is_post:
            logging.info('Empty arcadia patch, assuming its a "prepatch" run. Not doing anything.')
            return

        job = self.Parameters.job
        poll_timeout = self.Parameters.poll_timeout
        gsid = self.Context.__GSID
        logging.info('gsid: {gsid}'.format(gsid=gsid))

        token = sdk2.Vault.data('MDB', 'mdb-ci-token')

        arcanum_id = None
        for value in gsid.split():
            if value.startswith('ARCANUM:'):
                arcanum_id = value.split(':')[1]
                break

        trigger_id = str(uuid.uuid4())

        params = {'token': 'token', 'trigger_id': trigger_id}
        if arcanum_id is not None:
            params['ARCANUMID'] = arcanum_id
        res = requests.get(
            'https://jenkins.db.yandex-team.ru/job/{job}/buildWithParameters'.format(job=job),
            params=params,
            headers={'Authorization': 'OAuth {token}'.format(token=token)},
        )
        res.raise_for_status()

        build = find_build(job, token, trigger_id)

        assert poll_build(token, build, poll_timeout), 'Jenkins build failed: {build}'.format(
            build=build.replace('api/json', 'console')
        )
