import logging
import os
import shutil

from sandbox import sdk2, common
from sandbox.sandboxsdk import process
from sandbox.sdk2.paths import get_logs_folder

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr


class DefaultToLastResourceMixin(object):
    @common.utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            attrs=cls.attrs,
            state=ctr.State.READY,
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class LastResource(DefaultToLastResourceMixin, sdk2.parameters.Resource):
    pass


class PackerBinary(sdk2.Resource):
    """
    Packer static binary (linux/amd64)

    Executable file from https://www.packer.io/
    """

    releasable = True
    any_arch = False
    auto_backup = True
    executable = True
    ttl = 365
    release_version = sdk2.parameters.String("Release version")


class PackerBuilderBinary(sdk2.Resource):
    """
    Packer Builder static binary (linux/amd64)

    See details about Packer builders at https://www.packer.io/docs/builders/index.html
    """

    releasable = True
    any_arch = False
    auto_backup = True
    executable = True
    ttl = 365
    release_version = sdk2.parameters.String("Release version")


class CloudBuildImageWithPacker(sdk2.Task):
    """
    Build Yandex Cloud Image with Packer and builder for YC
    """

    class Requirements(sdk2.Requirements):
        cores = 1  # exactly 1 core
        ram = 1024  # 8GiB or less
        disk_space = 5 * 1024  # 5GiB
        dns = ctm.DnsType.DNS64

        client_tags = ctc.Tag.Group.LINUX

    class Caches(sdk2.Requirements.Caches):
        pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        max_restarts = 1
        kill_timeout = 1800

        packer_resource = LastResource("Resource with Packer", resource_type=PackerBinary)
        packer_builder_resource = LastResource("Resource with custom binary Packer builder",
                                               resource_type=PackerBuilderBinary)

        # custom parameters
        svn_dir_url = sdk2.parameters.SvnUrl("Svn url of a directory with Packer template file", required=True)
        template_file = sdk2.parameters.String("Packer template filename", required=True)
        env_vars = sdk2.parameters.Dict("Environment variables", required=False)
        vault_env = sdk2.parameters.String("space-separated list of ENVVAR=vault_owner:vault_key", required=False)
        service_account_key = sdk2.parameters.String("Service account private key (json file with RSA keypair "
                                                     "generated by IAM) in form FILENAME=vault_owner:vault_key",
                                                     required=False)
        log_packer_to_file = sdk2.parameters.Bool("Write Packer log to file (set PACKER_LOG)", required=False,
                                                  default=False)
        extra_resources = sdk2.parameters.Dict("Extra resources")
        extra_secrets = sdk2.parameters.Dict("Extra secrets")
        debug_mode = sdk2.parameters.Bool("Run Packer build step with '-debug'", required=False, default=False)
        with sdk2.parameters.RadioGroup("If the build fails do", per_line=2) as on_error_mode:
            on_error_mode.values['cleanup'] = on_error_mode.Value(value="cleanup", default=True)
            on_error_mode.values['abort'] = on_error_mode.Value(value="abort")

    def prepare_resources(self, workdir):
        for resource_target_path, resource_id_text in self.Parameters.extra_resources.items():
            try:
                resource_id = int(resource_id_text)
            except ValueError:
                raise common.errors.TaskError("Invalid resource id {}. It must be integer".format(resource_id_text))
            resources = list(sdk2.Resource.find(id=resource_id).limit(1))
            if not resources:
                raise common.errors.TaskError("Resource id {} not found".format(resource_id))
            resource_path = str(sdk2.ResourceData(resources[0]).path)
            target_path = str(sdk2.Path(workdir).joinpath(resource_target_path))
            if not os.path.isfile(resource_path):
                raise common.errors.TaskError("Resource id {} is not file".format(resource_id))
            mkdir_path = os.path.split(target_path)[0]
            if mkdir_path and not os.path.isdir(mkdir_path):
                os.makedirs(mkdir_path)
            shutil.copyfile(resource_path, target_path)

    def prepare_secrets(self, workdir):
        for secret_target_path, vault in self.Parameters.extra_secrets.items():
            vault_owner, vault_key = vault.split(':', 1)
            secret_data = sdk2.Vault.data(vault_owner, vault_key)
            target_path = str(sdk2.Path(workdir).joinpath(secret_target_path))
            mkdir_path = os.path.split(target_path)[0]
            if mkdir_path and not os.path.isdir(mkdir_path):
                os.makedirs(mkdir_path)
            with open(target_path, "w+") as fd:
                fd.write(secret_data)

    def prepare_dir(self, workdir):
        url = self.Parameters.svn_dir_url
        if url:
            process.run_process(['svn', 'co', url, workdir], log_prefix='checkout')

        # copy and chmod packer builder if specified in task
        if self.Parameters.packer_builder_resource:
            packer_builder_path = sdk2.ResourceData(self.Parameters.packer_builder_resource).path
            target_path = sdk2.Path(workdir).joinpath(packer_builder_path.name)
            shutil.copyfile(str(packer_builder_path), str(target_path))
            target_path.chmod(0o775)

        self.prepare_resources(workdir)
        self.prepare_secrets(workdir)

    def packer_validate_cmdline(self):
        cmd_args = [
            self._packer_bin(),
            "validate",
            self.Parameters.template_file
        ]
        return ' '.join(cmd_args)

    def packer_build_cmdline(self):
        cmd_args = [
            self._packer_bin(),
            "build",
            "-color=false",
            self.Parameters.template_file
        ]
        return ' '.join(cmd_args)

    def _packer_bin(self):
        return str(sdk2.ResourceData(self.Parameters.packer_resource).path)

    def build_options(self):
        build_options = ["-color=false"]

        if self.Parameters.debug_mode:
            build_options.append("-debug")

        build_options.append("-on-error={}".format(self.Parameters.on_error_mode))

        return build_options

    def prepare_env(self):
        env = os.environ.copy()
        env.update(self.Parameters.env_vars or {})
        vault_env = self.Parameters.vault_env
        if vault_env:
            for pair in vault_env.split():
                (name, value) = pair.split('=')
                (owner, key) = value.split(':')
                value = sdk2.Vault.data(owner, key)
                env[name] = value
        return env

    def on_execute(self):
        logging.info("Current environments: %r", os.environ)
        workdir = self.path('packer-workdir')
        workdir.mkdir()

        logging.info("Prepare environment in dir {0}".format(workdir))
        self.prepare_dir(str(workdir))

        env = self.prepare_env()

        service_account_key = self.Parameters.service_account_key

        if service_account_key:
            service_account_key_path = self.create_sa_keyfile(service_account_key, workdir)

        build_opts = self.build_options()
        workdir_file_list = os.listdir(str(workdir))

        # prepare all steps
        steps = [
            {
                "name": "version",
                "cmd": "{} version".format(self._packer_bin())
            },
            {
                "name": "validate",
                "cmd": "{} validate {}".format(self._packer_bin(), self.Parameters.template_file)
            },
            {
                "name": "build",
                "cmd": "{} build {} {}".format(self._packer_bin(), ' '.join(build_opts),
                                               self.Parameters.template_file)
            }
        ]

        for step in steps:
            if self.Parameters.log_packer_to_file:
                packer_log_file = os.path.join(get_logs_folder(), "PACKER_step_{}.log".format(step.get("name")))
                env.update({"PACKER_LOG": "1",
                            "PACKER_LOG_PATH": packer_log_file})

            logging.info("Run 'packer {}' step".format(step.get("name")))
            sp = process.run_process(step.get("cmd"), shell=True, work_dir=(str(workdir)), environment=env,
                                     log_prefix='packer_{}'.format(step.get("name")), wait=False)
            sp.wait()
            process.check_process_return_code(sp)

        if service_account_key and os.path.exists(str(service_account_key_path)):
            logging.info("Remove SA key file {}".format(service_account_key_path))
            os.remove(str(service_account_key_path))

        # collect all new file and move them to log dit
        new_files = list(set(os.listdir(str(workdir))) - set(workdir_file_list))
        if len(new_files) > 0:
            logging.info("Moving new files to TASK_LOG resource:")
            for new_file in new_files:
                dst_file = str(os.path.join(get_logs_folder(), new_file))
                src_file = str(sdk2.Path(workdir).joinpath(new_file))
                logging.info("  {} -> {}".format(src_file, dst_file))
                shutil.move(src_file, dst_file)

    @staticmethod
    def create_sa_keyfile(service_account_key, workdir):
        (filename, vault) = service_account_key.split('=')
        (vault_owner, vault_key) = vault.split(':')
        service_account_key_data = sdk2.Vault.data(vault_owner, vault_key)
        service_account_key_path = sdk2.Path(workdir).joinpath(filename)
        with service_account_key_path.open('wb') as f:
            f.write(service_account_key_data)
            f.close()
        return service_account_key_path
