import json
import logging
import os
import requests
import shutil
import datetime

from sandbox import sdk2
from sandbox.sandboxsdk import process, errors
from sandbox.sdk2.paths import get_logs_folder

import sandbox.common.types.resource as ctr

from sandbox.projects.cloud.platform.CloudBuildImageWithPacker import CloudBuildImageWithPacker


class PackerManifestResource(sdk2.Resource):
    """
    Packer Builder static binary (linux/amd64)

    See details about Packer builders at https://www.packer.io/docs/builders/index.html
    """

    releasable = True
    any_arch = False
    auto_backup = False
    executable = False
    ttl = 5
    release_version = sdk2.parameters.String("Release version")


class CloudBuildImageWithPackerWithRelease(CloudBuildImageWithPacker):
    """
    Build Yandex Cloud Image with Packer and builder for YC
    """

    def on_success(self, prev_status):
        resdir = self.path('res-workdir')
        resdir.mkdir()
        shutil.copyfile(os.path.join(get_logs_folder(), "manifest.json"),
                        os.path.join(str(resdir), "manifest.json"))

        PackerManifestResource(self, "test", os.path.join(str(resdir), "manifest.json"))

    def on_release(self, parameters_):
        workdir = self.path('yc-workdir')
        workdir.mkdir()

        logging.info("Prepare yc environment in dir {0}".format(workdir))

        if self.Parameters.svn_dir_url:
            process.run_process(['svn', 'co', self.Parameters.svn_dir_url, str(workdir)], log_prefix='checkout')

        self.prepare_dir(str(workdir))
        env = self.prepare_env()

        service_account_key = self.Parameters.service_account_key
        if service_account_key:
            service_account_key_path = self.create_sa_keyfile(service_account_key, workdir)

        recipe_path = os.path.join(str(workdir), self.Parameters.template_file)
        logging.info("Recipe path: %s", recipe_path)
        yc_cmd_path = self._install_yc(str(workdir))
        logging.info("YC path: %s", yc_cmd_path)
        yc_config_path = self._create_yc_config(yc_cmd_path, str(workdir), str(service_account_key_path))
        logging.info("YC config path: %s", yc_config_path)
        artifact_id, build_time = self._source_image_info_from_manifest()
        logging.info("Artifact_id, Build time: %s, %s", artifact_id, build_time)
        if not artifact_id or not build_time:
            raise errors.SandboxTaskFailureError("couldn't parse artifact_id or build_time from manifest")

        artifact_family = self._extract_family_from_recipe(recipe_path)
        logging.info("Artifact family: %s", artifact_family)
        if not artifact_family:
            raise errors.SandboxTaskFailureError("couldn't parse artifact_family from recipe")

        artifact_name = self._generate_image_name(build_time, artifact_family)
        logging.info("Artifact name: %s", artifact_name)
        if not artifact_name:
            raise errors.SandboxTaskFailureError("couldn't parse artifact_name from recipe")

        logging.info("Current environments: %r", env)

        self._create_image_from_source(yc_cmd_path, yc_config_path, artifact_id, artifact_name, artifact_family, env)
        self._delete_tmp_image(yc_cmd_path, yc_config_path, artifact_id, env)

        if service_account_key and os.path.exists(str(service_account_key_path)):
            logging.info("Remove SA key file {}".format(service_account_key_path))
            os.remove(str(service_account_key_path))

    @staticmethod
    def _install_yc(workdir):
        installer_local_path = os.path.join(workdir, "install.sh")
        logging.info("Installer local path is: %s", installer_local_path)
        try:
            r = requests.get("https://storage.yandexcloud.net/yandexcloud-yc/install.sh", timeout=10)
        except Exception as e:
            raise Exception(e)
        else:
            with open(installer_local_path, 'w') as f:
                f.write(r.text)

        install_yc_cmd = ["bash", installer_local_path, "-n", "-i", workdir]
        logging.info("install_yc_cmd is: %r", install_yc_cmd)
        sp = process.run_process(install_yc_cmd, log_prefix="yc_installer", shell=True, wait=False)
        sp.wait()
        process.check_process_return_code(sp)

        yc_cmd_path = os.path.join(os.path.join(workdir, "bin"), "yc")
        validate_installer_cmd = [yc_cmd_path, "version"]
        sp = process.run_process(validate_installer_cmd, log_prefix="yc_installer", shell=True, wait=False)
        sp.wait()
        process.check_process_return_code(sp)
        return yc_cmd_path

    @staticmethod
    def _create_yc_config(yc_cmd_path, workdir, sa_key_file_path):
        yc_config_path = os.path.join(workdir, "yc.config")
        set_sa_key = [
            yc_cmd_path, "config",
            "set", "service-account-key", "--config", yc_config_path,
            sa_key_file_path
        ]
        logging.info("set_sa_key is: %r", set_sa_key)
        sp = process.run_process(set_sa_key, log_prefix="set_sa_key", shell=True, wait=False)
        sp.wait()
        process.check_process_return_code(sp)
        return yc_config_path

    # packer-platform-image-base-kubelet-static-salt-test.json
    @staticmethod
    def _create_image_from_source(yc_cmd_path, yc_config_path, artifact_id, artifact_name, artifact_family, env):
        create_image_from_source = [
            yc_cmd_path, "--config", yc_config_path,
            "compute", "image", "create",
            "--folder-id", env["YC_STABLE_FOLDER_ID"],
            "--source-image-id", artifact_id,
            "--name", artifact_name,
            "--family", artifact_family,
            "--endpoint", env["YC_ENDPOINT"]
        ]
        logging.info("create image from source cmd is: %r", create_image_from_source)
        sp = process.run_process(create_image_from_source, log_prefix="create_image_from_source",
                                 shell=True, wait=False)
        sp.wait()
        process.check_process_return_code(sp)

    @staticmethod
    def _delete_tmp_image(yc_cmd_path, yc_config_path, artifact_id, env):
        delete_tmp_image = [
            yc_cmd_path, "--config", yc_config_path,
            "compute", "image", "delete", artifact_id,
            "--endpoint", env["YC_ENDPOINT"]
        ]
        logging.info("delete image from source cmd is: %r", delete_tmp_image)
        CloudBuildImageWithPackerWithRelease._run_subproccess(delete_tmp_image, "delete_tmp_image")

    @staticmethod
    def _run_subproccess(delete_tmp_image, log_prefix):
        sp = process.run_process(delete_tmp_image, log_prefix=log_prefix, shell=True, wait=False)
        sp.wait()
        process.check_process_return_code(sp)

    def _source_image_info_from_manifest(self):
        resources = [r for r in
                     sdk2.Resource.find(task=self, state=ctr.State.READY, resource_type=PackerManifestResource).limit(
                         0)]
        logging.info("Resources: %r", resources)
        with open(str(sdk2.ResourceData(resources[0]).path)) as f:
            data = json.loads(f.read())
        builds = data.get('builds')
        if builds is not None:
            artifact_id = builds[0].get("artifact_id")
            build_time = builds[0].get("build_time")
            if artifact_id and build_time:
                return artifact_id, build_time

        return '', ''

    @staticmethod
    def _extract_family_from_recipe(recipe_path):
        with open(recipe_path) as f:
            data = json.loads(f.read())
        return data["builders"][0]["image_family"]

    @staticmethod
    def _generate_image_name(build_time, image_family):
        artifact_name = "{}-{}-release".format(
            image_family,
            datetime.datetime.fromtimestamp(build_time).strftime("%Y-%m-%dt%H-%M-%S")
        )
        return artifact_name
