import logging
import time
import os
import shutil

from sandbox import sdk2, common
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import paths

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr


class CloudInfraYtr(sdk2.Resource):
    """
    Yandex.Cloud infrasTructure Reconciler AKA ytr (linux/amd64)

    Executable build from cloud-go/api/infra/ytr/cmd/ytr
    """

    releasable = True
    any_arch = False
    auto_backup = True
    executable = True
    ttl = 365
    release_version = sdk2.parameters.String("Release version")


class DefaultToLastResourceMixin(object):
    @common.utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            attrs=cls.attrs,
            state=ctr.State.READY,
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class LastResource(DefaultToLastResourceMixin, sdk2.parameters.Resource):
    pass


class CloudInfraYtrReconcile(sdk2.Task):
    """
    Reconcile Yandex Cloud infrasTructure by ytr
    """

    class Requirements(sdk2.Requirements):
        cores = 1  # vCores
        ram = 1024  # Mb
        disk_space = 5 * 1024  # MB
        dns = ctm.DnsType.DNS64

        client_tags = ctc.Tag.Group.LINUX

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        max_restarts = 1
        kill_timeout = 300

        # main binary
        ytr_resource = LastResource("Resource with YTR binary", resource_type=CloudInfraYtr)

        # custom parameters
        config_file = sdk2.parameters.String("YTR config file path", required=True)
        overrides_file = sdk2.parameters.String("YTR config overrides file path", required=False)
        custom_ca_pem = sdk2.parameters.String("Custom CA in PEM format to trust", required=False)
        diff_mode = sdk2.parameters.Bool("Use '--diff' mode", default=False)
        env_vars = sdk2.parameters.Dict("Environment variables", required=False)
        env_vars_vault = sdk2.parameters.Dict("Secret Environment variables ENVVAR=vault_owner:vault_key",
                                              required=False)
        files_vault = sdk2.parameters.Dict("Files provisoned through Vault", required=False)
        files_res = sdk2.parameters.Dict("Files provisoned through Resources", required=False)

        with sdk2.parameters.Output:
            ytr_log = sdk2.parameters.Url(
                "URL to ytr log file", default=""
            )

    @staticmethod
    def secure_delete(path, passes=1):
        with open(path, "r+") as delfile:
            length = delfile.tell()
            for i in range(passes):
                delfile.seek(0)
                delfile.write(os.urandom(length))
        os.remove(path)

    def cleanup(self):
        if self.Parameters.files_vault:
            for file_name in self.Parameters.files_vault.keys():
                try:
                    self.secure_delete(file_name)
                except IOError:
                    logging.info("IOError exception while secure remove file {}".format(file_name))
                else:
                    logging.info("File {} secure removed".format(file_name))

    def _ytr_bin(self):
        return str(sdk2.ResourceData(self.Parameters.ytr_resource).path)

    def run_options(self):
        run_options = ["-c", self.Parameters.config_file]

        if self.Parameters.overrides_file:
            run_options.extend(["--overrides", self.Parameters.overrides_file])

        if self.Parameters.diff_mode:
            run_options.append("--diff")

        if self.Parameters.custom_ca_pem:
            run_options.extend(["--ca", self.Parameters.custom_ca_pem])

        return run_options

    def on_prepare(self):
        self.Context.test_on_prepare = True
        msg = "There is on_prepare stage"
        logging.info(msg)
        logging.debug(msg + " (debug)")
        self.set_info(msg)
        super(CloudInfraYtrReconcile, self).on_prepare()

    def on_break(self, prev_status, status):
        self.Context.test_on_break = True
        super(CloudInfraYtrReconcile, self).on_break(prev_status, status)
        msg = "on_break: switching from {} to {}".format(prev_status, status)
        logging.debug(msg)
        self.set_info(msg)
        time.sleep(1)
        self.cleanup()

    def on_wait(self, prev_status, status):
        super(CloudInfraYtrReconcile, self).on_wait(prev_status, status)
        msg = "on_wait: switching from {} to {}".format(prev_status, status)
        logging.debug(msg)
        time.sleep(1)

    def on_finish(self, prev_status, status):
        self.Context.test_on_finish = True
        super(CloudInfraYtrReconcile, self).on_finish(prev_status, status)
        logging.debug("on_finish: switching from %s to %s", prev_status, status)
        resource_id = self.Context.custom_log_resource_id
        if resource_id:
            file_resource_url = getattr(resource_id, "http_proxy", None)
            if not file_resource_url:
                resources = self.server.resource.read(id=resource_id, limit=1)
                if resources["items"]:
                    resource_id = resources["items"][0]
                else:
                    return "Invalid resource_id {!r}".format(resource_id)
                self.ytr_log = resource_id["http"]["proxy"]

        self.cleanup()

    def on_execute(self):
        def get_secret_env_vars():
            result = {}
            if self.Parameters.env_vars_vault:
                for env_name, secret_id in self.Parameters.env_vars_vault.iteritems():
                    (owner, name) = secret_id.split(':')
                    value = sdk2.Vault.data(owner, name)
                    result[env_name] = value
            return result

        def prepare_secret_files():
            if self.Parameters.files_vault:
                for file_name, secret_id in self.Parameters.files_vault.iteritems():
                    (owner, name) = secret_id.split(':')
                    value = sdk2.Vault.data(owner, name)
                    with open(file_name, "w+") as f:
                        f.write(value)

        def prepare_resources():
            for resource_target_path, resource_id_text in self.Parameters.files_res.items():
                try:
                    resource_id = int(resource_id_text)
                except ValueError:
                    raise common.errors.TaskError("Invalid resource id {}. It must be integer".format(resource_id_text))
                resources = list(sdk2.Resource.find(id=resource_id).limit(1))
                if not resources:
                    raise common.errors.TaskError("Resource id {} not found".format(resource_id))
                resource_path = str(sdk2.ResourceData(resources[0]).path)
                target_path = str(sdk2.Path(os.getcwd()).joinpath(resource_target_path))
                if not os.path.isfile(resource_path):
                    raise common.errors.TaskError("Resource id {} is not file".format(resource_id))
                mkdir_path = os.path.split(target_path)[0]
                if mkdir_path and not os.path.isdir(mkdir_path):
                    os.makedirs(mkdir_path)
                shutil.copyfile(resource_path, target_path)

        prepare_resources()
        logging.info("Current environments: {}".format(os.linesep.join(os.environ)))
        logging.info("Current work directory: {}".format(os.getcwd()))
        logging.info("Current dir list: {}".format(" ".join(paths.list_dir(os.getcwd()))))

        # prepare OS env variables and files on disk
        env = os.environ.copy()
        env.update(self.Parameters.env_vars or {})
        env.update(get_secret_env_vars() or {})

        prepare_secret_files()

        if self.scheduler:
            logging.info(
                "This is a %s run under scheduler #%d",
                "scheduled" if self.scheduler > 0 else "manual",
                abs(self.scheduler)
            )

        # prepare command with options
        ytr_options = self.run_options()
        ytr_command = self._ytr_bin() + " " + " ".join(ytr_options)

        msg = "Run command '{}' ".format(ytr_command)
        logging.info(msg)
        self.set_info(msg)

        sp = process.run_process(cmd=ytr_command, shell=True, environment=env,
                                 work_dir=os.getcwd(), log_prefix='ytr', wait=False, outputs_to_one_file=True,
                                 close_fds=True)
        sp.wait()

        html_ytr_log = sdk2.helpers.gdb.get_html_view_for_logs_file(
            "ytr stdout and stderr",
            os.path.basename(sp.stdout_path),
            self.log_resource.id
        )
        self.set_info("YTR logs is available in custom logs: </br> {0}".format(html_ytr_log), do_escape=False)

        # prepare custom ytr logs
        custom_log_resource = sdk2.service_resources.TaskCustomLogs(self, "YTR Logs", "ytr_logs")
        custom_log_data = sdk2.ResourceData(custom_log_resource)
        custom_log_data.path.mkdir(0o755, parents=True, exist_ok=True)
        paths.copy_path(sp.stdout_path, str(custom_log_data.path))
        self.Context.custom_log_resource_id = custom_log_resource.id

        # check execution result
        process.check_process_return_code(sp)

        logging.info("All done. Finishing.")
