import contextlib
import json
import logging
import os
import re
import shutil
import tarfile
import tempfile

from sandbox import sdk2
from sandbox.common import urls as common_urls
from sandbox.sandboxsdk import process
from sandbox.sdk2.helpers import process as sdk2_process

from sandbox.common.types import misc as ctm
from sandbox.common.types import resource as ctr
from sandbox.common.types import client as ctc
from sandbox.common import utils as cu
from sandbox.sdk2.vcs import svn

from sandbox.projects.common import binary_task
from sandbox.projects.common.yappy import resources as yappy_resources
from sandbox.projects.common.yappy.utils import node_manager
from sandbox.projects.common.build import parameters as build_parameters
from sandbox.projects.common.arcadia import sdk as arcadia_sdk

from sandbox.projects import resource_types

logger = logging.getLogger(__name__)


class YfmTool(sdk2.Resource):
    executable = False
    auto_backup = True

    platform = sdk2.parameters.String
    version = sdk2.parameters.String


class BuildYfm(binary_task.LastBinaryTaskRelease, sdk2.Task):
    A_BEGIN_RE = re.compile(r'<a\s+name\s*=\s*["\']', re.UNICODE)
    A_END_RE = re.compile(r'["\']>[^<]*</a>', re.UNICODE)

    ARCADIA_URL = svn.Arcadia.DEFAULT_SCHEME + '://' + svn.Arcadia.ARCADIA_RW + '/arc/trunk/arcadia'
    DEFAULT_COMMITTER = 'robot-cozmo'

    GITHUB_URL_PATTERN = 'https://raw.githubusercontent.com/yandex-cloud/yfm-transform/{}/'
    GITHUB_DOCS_PATTERN = 'DOCS{}.md'
    GITHUB_PICS_PATH = 'docsAssets/'
    GITHUB_PICS_NAMES = ('note-alert.jpg', 'note-tip.jpg', 'note-warning.jpg', 'note.jpg', 'tabs.jpg')


    NPM_TOOL_NAME = 'docs'
    TOOL_NAME = 'yfm-docs'

    YATOOL_RE_STR = '"{tool_name}": {{\n\s+"formula": {{.*?}}'
    YATOOL_SNIPPET = (
        '"{tool_name}": {{\n'
        '            "formula": {{\n'
        '                "sandbox_id": {sandbox_id},\n'
        '                "match": "{tool_name}"\n'
        '            }}'
    )

    YAV_SSH_KEY = 'ssh-id_rsa'
    YAV_TOKEN = 'robot-cozmo:robot-cozmo-yav-token'
    YAV_UUID = 'sec-01drh3z694nqvd4wa9qwc0y39z'

    class Parameters(sdk2.Parameters):
        version = sdk2.parameters.String('yfm tool version', default='latest')
        update_docs = sdk2.parameters.Bool('Update documentation from GH', default=True)
        create_pull_request = sdk2.parameters.Bool('Create pull request', default=True)
        checkout_arcadia_from_url = build_parameters.ArcadiaUrl(default_value='arcadia-arc:/#trunk')
        plugins_directory = sdk2.parameters.String('Plugins directory', default='data-ui/yfm-custom-plugins')
        arc_secret = build_parameters.ArcSecret(default='sec-01eyxes68z3wmr6k1kkjddv27g#arc-token')

        ssh_vault_owner = sdk2.parameters.String('Vault item owner')
        ssh_vault_name = sdk2.parameters.String('Vault item with ssh key for git access')

        resource_inf = sdk2.parameters.Bool('Store built resource forever', default=True)

        nodejs_archive = sdk2.parameters.Resource(
            'Node JS',
            resource_type=yappy_resources.NodeJsArchive,
            state=(ctr.State.READY,),
            required=True
        )
        npm_registry = sdk2.parameters.String(
            'NPM registry',
            default='https://registry.npmjs.org/',
            required=False
        )

        ext_params = binary_task.binary_release_parameters(stable=True)

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_BIONIC

    ssh_key_params = {}
    platform_sandbox_resources = dict()

    tool_version = None
    docs_version = None

    @cu.singleton_property
    def node(self):
        return node_manager.NodeManager(self, registry=self.Parameters.npm_registry)

    def on_execute(self):
        super(BuildYfm, self).on_execute()

        if self.Parameters.ssh_vault_owner and self.Parameters.ssh_vault_name:
            self.ssh_key_params = {'key_owner': self.Parameters.ssh_vault_owner, 'key_name': self.Parameters.ssh_vault_name}
        else:
            from library.python.vault_client import instances as yav_instances

            logger.debug('Trying to get ssh secret from yav')
            yav_token = sdk2.Vault.data(*self.YAV_TOKEN.split(':'))

            yav_client = yav_instances.Production(
                rsa_auth=False,
                authorization='OAuth {}'.format(yav_token),
                decode_files=True,
            )

            yav_secret = yav_client.get_version(self.YAV_UUID)
            self.ssh_key_params = {'private_part': yav_secret['value'][self.YAV_SSH_KEY]}

        self.node.setup('', self.Parameters.nodejs_archive)

        checkout_arcadia_from_url = self.Parameters.checkout_arcadia_from_url
        plugins_directory = self.Parameters.plugins_directory.strip()
        arc_token = self.Parameters.arc_secret.value()

        with arcadia_sdk.mount_arc_path(
            checkout_arcadia_from_url,
            fetch_all=False,
            minimize_mount_path=False,
            use_arc_instead_of_aapi=True,
            arc_oauth_token=arc_token,
        ) as arcadia_path:
            checkout_dir = os.path.join(arcadia_path, plugins_directory)
            result_dir = os.path.join(checkout_dir, 'bin')

            with process.CustomOsEnviron(self.node._update_env()), sdk2_process.ProcessLog(self, logger='pkg') as pl:
                pkg_cmd = ['scripts/pkg', 'version', self.Parameters.version]
                rc = sdk2_process.subprocess.call(
                    pkg_cmd,
                    stdout=pl.stdout, stderr=pl.stdout,
                    cwd=checkout_dir,
                )
                if rc:
                    raise sdk2_process.subprocess.CalledProcessError(rc, pkg_cmd)

            with sdk2_process.ProcessLog(self, logger=self.NPM_TOOL_NAME) as pl:
                output = sdk2_process.subprocess.check_output([os.path.join(result_dir, '{}-linux'.format(self.NPM_TOOL_NAME)), '--version'], stderr=pl.stderr)
            self.tool_version = output.strip()

            platform_mapping = {'data': {}}
            platform_aliases = {
                'linux': 'linux',
                'macos': 'darwin',
                'win.exe': 'win32',
            }
            for pl in ['linux', 'macos', 'win.exe']:
                platform = platform_aliases[pl]
                binary_name = '{}-{}'.format(self.NPM_TOOL_NAME, pl)
                platform_dir = os.path.abspath(platform)
                os.mkdir(platform_dir)
                shutil.copy(os.path.join(result_dir, binary_name), os.path.join(platform_dir, self.TOOL_NAME))
                self.store_target_resource(platform, platform_mapping['data'])

            by_platform_attrs = {}
            if self.Parameters.resource_inf:
                by_platform_attrs['ttl'] = 'inf'

            by_platform_res = resource_types.PLATFORM_MAPPING(self, self.TOOL_NAME + ' tool', 'by_platform.json', **by_platform_attrs)
            result_resource = sdk2.ResourceData(by_platform_res)
            result_resource.path.write_text(unicode(json.dumps(platform_mapping)))
            result_resource.ready()

            self.create_pull_requests()

    @contextlib.contextmanager
    def info_section(self, name):
        # NodeManager expects this (usually inherited from yappy.base_build_ui task class)
        yield

    def store_target_resource(self, target_platform, platform_mapping):
        attrs = {'platform': target_platform, 'version': self.tool_version}
        if self.Parameters.resource_inf:
            attrs['ttl'] = 'inf'

        target_name = os.path.join(target_platform, self.TOOL_NAME)
        tgz_resource_name = '{}-{}.tgz'.format(self.TOOL_NAME, target_platform)
        with tarfile.open(str(self.path(tgz_resource_name)), 'w:gz') as tar:
            tar.add(os.path.abspath(target_name), arcname=self.TOOL_NAME)

        res = YfmTool(self, '{} for {}'.format(self.TOOL_NAME, target_platform), self.path(tgz_resource_name), **attrs)

        self.platform_sandbox_resources[target_platform] = res.id

        sdk2.ResourceData(res).ready()

        platform_mapping[target_platform] = {
            'url': res.http_proxy,
            'urls': [res.http_proxy],
            'md5': res.md5,
        }

    def create_pull_requests(self):
        if not self.Parameters.create_pull_request:
            return

        arcadia_dir = tempfile.mkdtemp('arcadia')
        svn.Svn.checkout(url=self.ARCADIA_URL, path=arcadia_dir, depth='empty')

        opts = svn.Svn.SvnOptions()
        opts.depth = 'empty'
        opts.ignore_externals = False
        opts.parents = True
        opts.force = True

        ya_tool_conf = 'build/ya.conf.json'
        resources_library = 'build/platform/yfm/ya.make'
        docs_dir = 'devtools/dummy_arcadia/hello_docs/yfm'

        svn.Svn.svn('update', opts=opts, path=os.path.join(arcadia_dir, ya_tool_conf), summary=True, svn_ssh=self.ARCADIA_URL)
        svn.Svn.svn('update', opts=opts, path=os.path.join(arcadia_dir, resources_library), summary=True, svn_ssh=self.ARCADIA_URL)

        opts.depth = 'infinity'
        svn.Svn.svn('update', opts=opts, path=os.path.join(arcadia_dir, docs_dir), summary=True, svn_ssh=self.ARCADIA_URL)

        with open(os.path.join(arcadia_dir, ya_tool_conf), 'r+') as f:
            content = f.read()
            f.seek(0)
            f.truncate()

            pattern = re.compile(self.YATOOL_RE_STR.format(tool_name=self.TOOL_NAME), re.DOTALL | re.MULTILINE)
            snippet = self.YATOOL_SNIPPET.format(
                tool_name=self.TOOL_NAME,
                sandbox_id=self.id,
            )
            content = pattern.sub(snippet, content)

            f.write(content)

        self.create_pr(os.path.join(arcadia_dir, ya_tool_conf), 'Update {} for ya tool to {}'.format(self.TOOL_NAME, self.tool_version))

        self.update_docs(os.path.join(arcadia_dir, docs_dir))

        with open(os.path.join(arcadia_dir, resources_library), 'r+') as f:
            content = f.read()
            f.seek(0)
            f.truncate()

            for platform, sbr in self.platform_sandbox_resources.items():
                snippet = 'sbr:{} FOR {}'.format(sbr, platform.upper())
                content = re.compile(r'sbr:\d+ FOR {}'.format(platform.upper())).sub(snippet, content)

            f.write(content)

        self.create_pr([os.path.join(arcadia_dir, resources_library), os.path.join(arcadia_dir, docs_dir)], 'Update {} for ya make to {}'.format(self.TOOL_NAME, self.tool_version))

    def create_pr(self, path, message):

        revprops = {
            'arcanum:json': 'yes',
            'arcanum:review': 'new',
            'arcanum:review-publish': 'yes',
            'arcanum:review-reviewers': ','.join(['arivkin', 'birman111', 'tsufiev', 'workfork']),
        }

        try:
            message += '\nAuthor: {}\nSandbox task: {}/task/{}'.format(self.author, common_urls.server_url(), self.id)
            if self.docs_version != 'v' + self.tool_version:
                message += '\nDocs version: {}'.format(self.docs_version)

            with sdk2.ssh.Key(**self.ssh_key_params):
                commit_result = svn.Arcadia.commit(
                    path, message,
                    user=self.Parameters.ssh_vault_owner or self.DEFAULT_COMMITTER,
                    with_revprop=['{}={}'.format(k, v) for k, v in revprops.iteritems()],
                )
            self.set_info('Failed to create pull request:\n{}'.format(commit_result))
        except sdk2.svn.SvnError as e:
            logger.exception('Result of pull request creating')
            err = str(e)
            STDERR_START = '=== stderr:\n'
            index = err.find(STDERR_START)
            if index != -1:
                err = err[index + len(STDERR_START):]
            output = err.split("\n", 2)[2]

        try:
            info = json.loads(output)
            pr_id = info['pr']['id']
            self.set_info('Pull request <a href="https://a.yandex-team.ru/review/{id}" target="_blank">{id}</a> created.'.format(id=pr_id), do_escape=False)
        except Exception:
            logger.exception('Failed to get pull request ID from \'{}\''.format(output))
            self.set_info('Failed to get pull request ID. See logs')
            raise

    def update_docs(self, docs_dir):
        import itertools
        import requests
        import requests.exceptions
        import library.python.retry

        if not self.Parameters.update_docs:
            return

        request_timeout = 20
        connection_timeout = 5
        retry_conf = library.python.retry.DEFAULT_CONF.clone(
            retriable=lambda e: isinstance(e, (requests.exceptions.ConnectionError, requests.exceptions.Timeout))
                                or (isinstance(e, requests.exceptions.HTTPError) and e.response.status_code in (500, 502, 504))) \
            .upto(request_timeout)

        def get_docs(url, fail_on_error=False):
            logger.debug('requesting docs from url %s', url)
            response = library.python.retry.retry_call(
                requests.get,
                f_args=[url],
                f_kwargs=dict(timeout=(connection_timeout, request_timeout)),
                conf=retry_conf,
            )
            logger.debug('github responded %s', response)

            if fail_on_error:
                response.raise_for_status()
            return response

        def line_is_header(l):
            if not l:
                return False

            if not l.startswith(u'#'):
                return False

            l = l.lstrip(u'#')
            if not l.startswith(u' '):
                return False

            return True

        # at first try to fetch docs corresponding to tool_version, if failed then get docs from master
        # for both attempts try to get russian localization first then the default one
        for (version, lang_suffix) in itertools.product(['v' + self.tool_version, 'master'], ['.ru', '']):
            origin_docs_url = (self.GITHUB_URL_PATTERN + self.GITHUB_DOCS_PATTERN).format(version, lang_suffix)
            response = get_docs(origin_docs_url)
            if response.status_code == 200:
                original_docs_content = response.text
                self.docs_version = version
                break

        # fail the task if the attempt failed
        response.raise_for_status()

        # fix anchors (see DOCSTOOLS-289) and other roughnesses
        docs_lines = []
        for line in original_docs_content.splitlines():
            newline = line
            if line_is_header(line):
                newline = self.A_BEGIN_RE.sub(u'{#', newline)
                newline = self.A_END_RE.sub(u'}', newline)

            newline = newline.replace('`{% list tabs %}`', '`{% list tabs  %}`')
            newline = newline.replace('{{', 'not_var{{')

            docs_lines.append(newline)

            if newline != line:
                logger.debug('Line changed\n%s\n->%s', line, newline)

            # cut off possible language selector at the top of the page
            if newline.strip() == '- - -':
                docs_lines = []

        # add empty line at the end of a file
        if docs_lines[-1]:
            docs_lines.append('')

        # fetch some assets and store all the stuff to disk
        with open(os.path.join(docs_dir, 'examples.md'), 'w') as f:
            f.write('\n'.join(docs_lines))

        for pic_name in self.GITHUB_PICS_NAMES:
            r = get_docs(self.GITHUB_URL_PATTERN.format(self.docs_version) + self.GITHUB_PICS_PATH + pic_name, fail_on_error=True)

            with open(os.path.join(docs_dir, self.GITHUB_PICS_PATH, pic_name), 'wb') as f:
                f.write(r.content)
