import logging
import os

from sandbox import sdk2
from sandbox import common
from sandbox.projects import resource_types
import sandbox.common.types.task as ctt

from sandbox.projects.cmnt.cmnt_env import CmntEnv
from sandbox.projects.cmnt.DbBackup import CmntDbBackup
from sandbox.projects.cmnt.DbCleanup import CmntDbCleanupLastAction, CmntDbCleanupChats, CmntDbCleanupChatDescOrphans


TASK_TIMEOUT_SEC = 3600 * 23  # 23 hours
DEFAULT_MAX_BACKUP_COUNT = 7
WAIT_STATUSES = ctt.Status.Group.BREAK | ctt.Status.Group.FINISH


class CmntDbActions(sdk2.Task):
    """ A task, which runs subtasks for Commentator's DB, like backup & cleanups. """

    class Requirements(sdk2.Task.Requirements):
        ram = 2 * 1024
        disk_space = 5000
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass  # the task does not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = TASK_TIMEOUT_SEC

        with sdk2.parameters.Group("Binaries") as bin_block:
            bin_ydb = sdk2.parameters.Resource("Resource id for ydb client", resource_type=resource_types.ARCADIA_PROJECT, required=True)
            bin_cmnt_yql = sdk2.parameters.Resource("Resource id for cmnt-yql", resource_type=resource_types.ARCADIA_PROJECT, required=True)
            bin_cleanup_last_action = sdk2.parameters.Resource("Resource id for cmnt-cleanup-last-action", resource_type=resource_types.ARCADIA_PROJECT, required=True)
            bin_cleanup_chats = sdk2.parameters.Resource("Resource id for cmnt-cleanup-chats", resource_type=resource_types.ARCADIA_PROJECT, required=True)
            bin_cleanup_desc_orphans = sdk2.parameters.Resource("Resource id for cmnt-cleanup-desc-orphans", resource_type=resource_types.ARCADIA_PROJECT, required=True)

        with sdk2.parameters.Group("YT parameters") as yt_block:
            yt_proxy = sdk2.parameters.String("YT proxy", default=CmntEnv.DEFAULT_YT_PROXY, required=True)
            yt_cleanup_data_folder = sdk2.parameters.String("YT folder to store cleanup data", required=True)

        with sdk2.parameters.Group("Security credentials") as in_block:
            yt_token = sdk2.parameters.Vault("Vault secret name with YT token",
                                             default='robot-cmnt:yt_token',
                                             required=True)
            ydb_token = sdk2.parameters.Vault("Vault secret name with YDB token",
                                              default='robot-cmnt:ydb_token',
                                              required=True)

        with sdk2.parameters.RadioGroup("Environment") as env:
            env.values[CmntEnv.ENV_TESTING] = env.Value(value=CmntEnv.ENV_TESTING, default=True)
            env.values[CmntEnv.ENV_ALPHA] = env.Value(value=CmntEnv.ENV_ALPHA)
            env.values[CmntEnv.ENV_PRODUCTION] = env.Value(value=CmntEnv.ENV_PRODUCTION)

        with sdk2.parameters.Group("Backup parameters") as backup_block:
            yt_backups_folder = sdk2.parameters.String("YT path where backup will be stored", required=True)
            yt_backup_count = sdk2.parameters.Integer("Max backup count", default=DEFAULT_MAX_BACKUP_COUNT, required=True)

        with sdk2.parameters.Group("Clean last action data parameters") as clean_last_action_block:
            clean_la_days = sdk2.parameters.Integer("Cleanup older than given amount of days", default=30, required=True)
            clean_la_threads = sdk2.parameters.Integer("Cleanup threads", default=10, required=True)
            clean_la_limit = sdk2.parameters.Integer("Cleanup limit", default=1000000, required=True)
            clean_la_tables_count = sdk2.parameters.Integer("Max cleanup tables count", default=7, required=True)

        with sdk2.parameters.Group("Clean chats parameters") as clean_chats_block:
            clean_chats_srv = sdk2.parameters.String("Cleanup chats for servuices (space separated)", required=True)
            clean_chats_days = sdk2.parameters.Integer("Cleanup older than given amount of days", default=30, required=True)
            clean_chats_threads = sdk2.parameters.Integer("Cleanup threads", default=10, required=True)
            clean_chats_limit = sdk2.parameters.Integer("Cleanup limit", default=1000000, required=True)
            clean_chats_tables_count = sdk2.parameters.Integer("Max cleanup tables count", default=7, required=True)

        with sdk2.parameters.Group("Clean chat desc orphans parameters") as clean_desc_orphans_block:
            clean_desc_threads = sdk2.parameters.Integer("Cleanup threads", default=10, required=True)
            clean_desc_limit = sdk2.parameters.Integer("Cleanup limit", default=1000000, required=True)
            clean_desc_tables_count = sdk2.parameters.Integer("Max cleanup tables count", default=7, required=True)

    def make_cleanup_folder_path(self, folder):
        return str(os.path.join(CmntEnv.get_yt_backup_path(self.Parameters.env, self.Parameters.yt_cleanup_data_folder), folder))

    def make_backup_task(self):
        task = CmntDbBackup(
            self,
            description="CmntDbBackup: child of {}".format(self.id),
            owner=self.Parameters.owner,
            endpoint=CmntEnv.get_ydb_endpoint(self.Parameters.env),
            db_path=CmntEnv.get_ydb_database(self.Parameters.env),
            ydb_client=self.Parameters.bin_ydb,
            yt_token=self.Parameters.yt_token,
            ydb_token=self.Parameters.ydb_token,
            backup_count=self.Parameters.yt_backup_count,
            yt_proxy=self.Parameters.yt_proxy,
            yt_path=CmntEnv.get_yt_backup_path(self.Parameters.env, self.Parameters.yt_backups_folder)
        )
        task.save()
        return task.enqueue()

    def make_cleanup_last_action_task(self):
        task = CmntDbCleanupLastAction(
            self,
            description="Child of {}".format(self.id),
            owner=self.Parameters.owner,
            cmnt_yql_client=self.Parameters.bin_cmnt_yql,
            cleanup_client=self.Parameters.bin_cleanup_last_action,
            yt_proxy=self.Parameters.yt_proxy,
            yt_cleanup_data_folder=self.make_cleanup_folder_path("last_action"),
            yt_token=self.Parameters.yt_token,
            ydb_token=self.Parameters.ydb_token,
            env=self.Parameters.env,
            days=self.Parameters.clean_la_days,
            cleanup_threads=self.Parameters.clean_la_threads,
            cleanup_limit=self.Parameters.clean_la_limit,
            cleanup_tables_count=self.Parameters.clean_la_tables_count
        )
        task.save()
        return task.enqueue()

    def make_cleanup_chats_task(self):
        task = CmntDbCleanupChats(
            self,
            description="Child of {}".format(self.id),
            owner=self.Parameters.owner,
            cmnt_yql_client=self.Parameters.bin_cmnt_yql,
            cleanup_client=self.Parameters.bin_cleanup_chats,
            yt_proxy=self.Parameters.yt_proxy,
            yt_cleanup_data_folder=self.make_cleanup_folder_path("chats"),
            yt_token=self.Parameters.yt_token,
            ydb_token=self.Parameters.ydb_token,
            env=self.Parameters.env,
            services=self.Parameters.clean_chats_srv,
            days=self.Parameters.clean_chats_days,
            cleanup_threads=self.Parameters.clean_chats_threads,
            cleanup_limit=self.Parameters.clean_chats_limit,
            cleanup_tables_count=self.Parameters.clean_chats_tables_count
        )
        task.save()
        return task.enqueue()

    def make_cleanup_desc_orphans_task(self):
        task = CmntDbCleanupChatDescOrphans(
            self,
            description="Child of {}".format(self.id),
            owner=self.Parameters.owner,
            cmnt_yql_client=self.Parameters.bin_cmnt_yql,
            cleanup_client=self.Parameters.bin_cleanup_desc_orphans,
            yt_proxy=self.Parameters.yt_proxy,
            yt_cleanup_data_folder=self.make_cleanup_folder_path("desc_orphans"),
            yt_token=self.Parameters.yt_token,
            ydb_token=self.Parameters.ydb_token,
            env=self.Parameters.env,
            cleanup_threads=self.Parameters.clean_desc_threads,
            cleanup_limit=self.Parameters.clean_desc_limit,
            cleanup_tables_count=self.Parameters.clean_desc_tables_count
        )
        task.save()
        return task.enqueue()

    def on_execute(self):
        with self.memoize_stage.backup:
            logging.info("Start backup")
            raise sdk2.WaitTask([self.make_backup_task()], WAIT_STATUSES, wait_all=True)

        with self.memoize_stage.after_backup:
            backup_success = self.find(CmntDbBackup, status=ctt.Status.SUCCESS).first() is not None
            if not backup_success:
                raise 'Backup failed, skip after_backup tasks'

            logging.info("Start after backups tasks")

            after_backup_tasks = [
                self.make_cleanup_last_action_task(),
                self.make_cleanup_chats_task(),
                self.make_cleanup_desc_orphans_task()
            ]

            raise sdk2.WaitTask(after_backup_tasks, WAIT_STATUSES, wait_all=True)

        clean_la_success = self.find(CmntDbCleanupLastAction, status=ctt.Status.SUCCESS).first() is not None
        clean_chats_success = self.find(CmntDbCleanupChats, status=ctt.Status.SUCCESS).first() is not None
        clean_desc_orphans_success = self.find(CmntDbCleanupChatDescOrphans, status=ctt.Status.SUCCESS).first() is not None
        if clean_la_success and clean_chats_success and clean_desc_orphans_success:
            logging.info("Done")
        else:
            raise common.errors.TaskError("Some of cleanup tasks failed")
