import logging
import os
import datetime
import json
import time

from sandbox import common
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.sandboxsdk.environments as sdk_environments
from sandbox.projects import resource_types

from sandbox.projects.cmnt.cmnt_env import CmntEnv
import ydb_scheme


BACKUP_WAIT_SEC = 6 * 3600
DEFAULT_MAX_BACKUP_COUNT = 7
WAIT_SLOT_SEC = 180
BACKUP_PARAM_RETRIES = 1000


class CmntDbBackup(sdk2.Task):
    """ A task, which backups Commentator's database. """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            sdk_environments.PipEnvironment("yandex-yt"),
        ]
        ram = 2 * 1024
        disk_space = 5000
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass  # the task does not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = BACKUP_WAIT_SEC

        with sdk2.parameters.Group("Input parameters") as in_block:
            endpoint = sdk2.parameters.String("Database endpoint (path:port)", default=CmntEnv.get_ydb_endpoint(CmntEnv.ENV_TESTING), required=True)
            db_path = sdk2.parameters.String("Database path", default=CmntEnv.get_ydb_database(CmntEnv.ENV_TESTING), required=True)
            ydb_client = sdk2.parameters.Resource("YDB client to use", resource_type=resource_types.ARCADIA_PROJECT, required=True)
            yt_token = sdk2.parameters.Vault("Vault secret name with YT token",
                                             default='robot-cmnt:yt_token',
                                             required=True)
            ydb_token = sdk2.parameters.Vault("Vault secret name with YDB token",
                                              default='robot-cmnt:ydb_token',
                                              required=True)
            backup_count = sdk2.parameters.Integer("Max backup count", default=DEFAULT_MAX_BACKUP_COUNT, required=True)

        with sdk2.parameters.Group("Output parameters") as out_block:
            yt_proxy = sdk2.parameters.String("YT proxy", default=CmntEnv.DEFAULT_YT_PROXY, required=True)
            yt_path = sdk2.parameters.String('YT path where backup will be stored', required=True)

    def on_create(self):
        self.Context.operation_id = None

    def get_ydb_cli_path(self):
        ydb_cli_resource = sdk2.ResourceData(self.Parameters.ydb_client)
        return str(ydb_cli_resource.path)

    def get_ydb_table_list(self, ydb_cli):
        output = ''
        with sdk2.helpers.ProcessLog(self, logger="ydb_cli_table_list") as pl:
            output = sp.check_output(
                [
                    ydb_cli,
                    '--endpoint', str(self.Parameters.endpoint),
                    '--database', str(self.Parameters.db_path),
                    'scheme', 'ls',
                    str(self.Parameters.db_path)
                ],
                env={
                    'YDB_TOKEN': self.Parameters.ydb_token.data()
                },
                stderr=pl.stderr,
                timeout=60
            )
            logging.info('ydb cli list tables result: %s', str(output))

        result = []
        for line in str(output).strip().split('\n'):
            item, typ = line.split()
            if typ == 'Table':
                result.append(item)

        return result

    def make_yt_folder(self, folder):
        self.yt_client.mkdir(folder)

    def get_env(self):
        return {
            'YT_TOKEN': self.Parameters.yt_token.data(),
            'YDB_TOKEN': self.Parameters.ydb_token.data()
        }

    def get_ydb_common_args(self, ydb_cli):
        return [
            ydb_cli,
            '--endpoint', str(self.Parameters.endpoint),
            '--database', str(self.Parameters.db_path),
        ]

    def create_dst_tables(self, ydb_cli, yt_dst_path, table_names):
        for tbl in table_names:
            ydb_sch = self.get_ydb_table_scheme(ydb_cli, tbl)
            schema = ydb_scheme.get_yt_scheme_from_ydb_scheme(ydb_sch)
            full_path = os.path.join(yt_dst_path, tbl)
            logging.info('Creating dest table %s with schema: %s' % (full_path, schema))
            self.yt_client.create_table(full_path, attributes={"schema": schema, "strict": True})

    def start_backup(self, ydb_cli, yt_dst_path, table_names, description):
        self.create_dst_tables(ydb_cli, yt_dst_path, table_names)

        args = self.get_ydb_common_args(ydb_cli)
        args.extend([
            'export', 'yt',
            '--json',
            '--proxy', str(self.Parameters.yt_proxy),
            '--description', description,
            '--retries', str(BACKUP_PARAM_RETRIES),
        ])

        for tbl in table_names:
            args.append('--item')
            args.append("src=%(src)s,dst=%(dst)s" % {
                'src': os.path.join(self.Parameters.db_path, tbl),
                'dst': os.path.join(yt_dst_path, tbl),
            })

        with sdk2.helpers.ProcessLog(self, logger="ydb_cli_start_backup") as pl:
            p = sp.Popen(
                args,
                env=self.get_env(),
                stdout=sp.PIPE,
                stderr=pl.stderr,
            )
            out, err = p.communicate()
            if p.returncode != 0:
                raise common.errors.TaskError("Could not start backup process")

            logging.info('Got json: %s' % out)
            json_data = json.loads(out)
            if json_data['status'] != u'SUCCESS':
                raise common.errors.TaskError("Something went wrong. See ydb_cli log")

            return json_data['id']

    def call_ydb_cli_with_json_out(self, args, cli_call):
        with sdk2.helpers.ProcessLog(self, logger="ydb_cli_%s.%s" % (cli_call, time.time())) as pl:
            p = sp.Popen(
                args,
                env=self.get_env(),
                stdout=sp.PIPE,
                stderr=pl.stderr,
            )
            out, err = p.communicate()
            if p.returncode == 0:
                try:
                    return json.loads(out)
                except:
                    logging.error('Not json response: %s' % out)
                    raise

            logging.error('ydb cli failed with message:')
            logging.error(out)

            return None

    def get_ydb_operation(self, ydb_cli, operation_id, retry_count=3):
        args = self.get_ydb_common_args(ydb_cli)
        args.extend([
            'operation', 'get', operation_id,
            '--json',
        ])

        for i in xrange(retry_count):
            data = self.call_ydb_cli_with_json_out(args, 'get_operation')

            if data is not None:
                return data

            time.sleep(WAIT_SLOT_SEC)

        raise common.errors.TaskError("Could not get operation status")

    def cancel_operation(self, ydb_cli, operation_id):
        args = self.get_ydb_common_args(ydb_cli)
        args.extend([
            'operation', 'cancel', operation_id,
            '--json',
        ])

        with sdk2.helpers.ProcessLog(self, logger="ydb_cli_cancel_operation") as pl:
            p = sp.Popen(
                args,
                env=self.get_env(),
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            out, err = p.communicate()

    def forget_operation(self, ydb_cli, operation_id):
        args = self.get_ydb_common_args(ydb_cli)
        args.extend([
            'operation', 'forget', operation_id
        ])

        with sdk2.helpers.ProcessLog(self, logger="ydb_cli_forget_operation") as pl:
            p = sp.Popen(
                args,
                env=self.get_env(),
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            out, err = p.communicate()

    def get_ydb_table_scheme(self, ydb_cli, table_name, retry_count=3):
        logging.info('ydb_cli: %s' % ydb_cli)
        args = self.get_ydb_common_args(ydb_cli)
        args.extend([
            'table', 'describe', os.path.join(self.Parameters.db_path, table_name),
            '--json',
        ])

        for i in xrange(retry_count):
            table_scheme = self.call_ydb_cli_with_json_out(args, 'describe_table')
            if table_scheme:
                return table_scheme

            time.sleep(WAIT_SLOT_SEC)

        raise common.errors.TaskError("Could not get ydb table scheme")

    def wait_backup_complete(self, ydb_cli, operation_id):
        start_time = time.time()
        while True:
            if time.time() - start_time > self.Parameters.kill_timeout:
                logging.info('Time is out. Cancelling operation')
                self.cancel_operation(ydb_cli, operation_id)
                raise common.errors.TaskError("Backup process is timed out. Operation %s has been canceled" % operation_id)

            op_data = self.get_ydb_operation(ydb_cli, operation_id)
            if op_data.get('ready'):
                logging.info('Operation is ready now')
                break

            logging.info('Operation is still running')
            time.sleep(WAIT_SLOT_SEC)

    def check_backup(self, yt_dst_folder, ydb_table_list):
        yt_file_list = self.yt_client.list(yt_dst_folder, absolute=True)

        if not yt_file_list:
            raise common.errors.TaskError('Zero number of created backup files')

        if len(ydb_table_list) != len(yt_file_list):
            raise common.errors.TaskError('Number of created backup files differs from number of YDB tables')

    def do_backup(self, ydb_cli, yt_dst_path, table_names, description):
        self.Context.operation_id = self.start_backup(ydb_cli, yt_dst_path, table_names, description)
        try:
            self.wait_backup_complete(ydb_cli, self.Context.operation_id)
        finally:
            op = self.Context.operation_id
            self.Context.operation_id = None
            self.forget_operation(ydb_cli, op)

    def on_terminate(self):
        logging.info("Terminating")
        if self.Context.operation_id:
            ydb_cli_path = self.get_ydb_cli_path()
            self.cancel_operation(ydb_cli_path, self.Context.operation_id)
            self.forget_operation(ydb_cli_path, self.Context.operation_id)

    def on_execute(self):
        import yt.wrapper as yt

        self.yt_client = yt.YtClient(self.Parameters.yt_proxy, self.Parameters.yt_token.data())
        ydb_cli_path = self.get_ydb_cli_path()

        now = datetime.datetime.now()

        backup_description = 'CMNT database backup created on %s' % now.strftime('%Y-%m-%d %H:%M:%S')

        table_names = self.get_ydb_table_list(ydb_cli_path)
        if not table_names:
            raise common.errors.TaskError('Empty list of tables in database. Nothing to do')

        logging.info("Table list to backup: ", table_names)

        yt_dst_folder = os.path.join(str(self.Parameters.yt_path), CmntEnv.make_yt_name(now))
        self.make_yt_folder(yt_dst_folder)
        self.do_backup(ydb_cli_path, yt_dst_folder, table_names, backup_description)
        self.check_backup(yt_dst_folder, table_names)
        CmntEnv.remove_old_data(self.yt_client, str(self.Parameters.yt_path), self.Parameters.backup_count, True)

        logging.info("Done")
