import logging
import os
import datetime
import json

from sandbox import common
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.sandboxsdk.environments as sdk_environments
from sandbox.projects import resource_types

from sandbox.projects.cmnt.cmnt_env import CmntEnv

CLEANUP_WAIT_SEC = 3600 * 16  # 16 hours
DEFAULT_DAYS = 30
DEFAULT_MAX_DATA_COUNT = 7
DEFAULT_CLEAN_THREADS = 10
DEFAULT_CLEAN_LIMIT = 1000000


class CmntDbCleanupBase:
    """ Base CMNT DB cleanup obsolete data task """

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = CLEANUP_WAIT_SEC

        with sdk2.parameters.Group("Binaries") as bin_block:
            cmnt_yql_client = sdk2.parameters.Resource("Resource id for cmnt-yql", resource_type=resource_types.ARCADIA_PROJECT, required=True)
            cleanup_client = sdk2.parameters.Resource("Resource id for cleanup tool", resource_type=resource_types.ARCADIA_PROJECT, required=True)

        with sdk2.parameters.Group("YT parameters") as yt_block:
            yt_proxy = sdk2.parameters.String("YT proxy", default=CmntEnv.DEFAULT_YT_PROXY, required=True)
            yt_cleanup_data_folder = sdk2.parameters.String("YT folder to store cleanup data", required=True)

        with sdk2.parameters.Group("Security credentials") as in_block:
            yt_token = sdk2.parameters.Vault("Vault secret name with YT token",
                                             default='robot-cmnt:yt_token',
                                             required=True)
            ydb_token = sdk2.parameters.Vault("Vault secret name with YDB token",
                                              default='robot-cmnt:ydb_token',
                                              required=True)

        with sdk2.parameters.RadioGroup("Environment") as env:
            env.values[CmntEnv.ENV_TESTING] = env.Value(value=CmntEnv.ENV_TESTING, default=True)
            env.values[CmntEnv.ENV_ALPHA] = env.Value(value=CmntEnv.ENV_ALPHA)
            env.values[CmntEnv.ENV_PRODUCTION] = env.Value(value=CmntEnv.ENV_PRODUCTION)

        with sdk2.parameters.Group("Exec parameters") as run_block:
            cleanup_threads = sdk2.parameters.Integer("Cleanup threads", default=DEFAULT_CLEAN_THREADS, required=True)
            cleanup_limit = sdk2.parameters.Integer("Cleanup limit", default=DEFAULT_CLEAN_LIMIT, required=True)
            cleanup_tables_count = sdk2.parameters.Integer("Max cleanup tables count", default=DEFAULT_MAX_DATA_COUNT, required=True)

    def get_yql_cli_path(self):
        yql_cli_resource = sdk2.ResourceData(self.Parameters.cmnt_yql_client)
        return str(yql_cli_resource.path)

    def get_clean_cli_path(self):
        cleanup_tool_resource = sdk2.ResourceData(self.Parameters.cleanup_client)
        return str(cleanup_tool_resource.path)

    def get_cmnt_yql_env(self):
        return {
            'YQL_TOKEN': self.Parameters.yt_token.data()
        }

    def get_cmnt_yql_args(self):
        """
        Common args to run cmnt-yql cleanup queries.

        Must be extended in tasks.
        """
        cmnt_yql = self.get_yql_cli_path()
        result_table = os.path.join(self.Parameters.yt_cleanup_data_folder, CmntEnv.make_yt_name(datetime.datetime.now()))

        args = [
            cmnt_yql,
            '--env', CmntEnv.get_backup_env(self.Parameters.env),
            '--param', 'result=%s' % result_table,
            '--param', 'limit=%s' % self.Parameters.cleanup_limit,
            '--dump-result-json',
        ]
        return args

    def dump_obsolete_data(self):
        """
        Runs specified cmnt-yql cleanup query.

        Query params must be defined in tasks in get_cmnt_yql_args() method.

        All cleanup queries return result in common format.
        """
        logging.info("Fetch cleanup data from DB backup")
        args = self.get_cmnt_yql_args()

        with sdk2.helpers.ProcessLog(self, logger="cmnt-yql") as pl:
            p = sp.Popen(
                args,
                env=self.get_cmnt_yql_env(),
                stdout=sp.PIPE,
                stderr=pl.stderr
            )
            out, err = p.communicate()
            if p.returncode == 0:
                try:
                    result_json = json.loads(out)
                    table_name = result_json[0][0]['result']
                    if table_name is None or len(table_name) == 0:
                        raise common.errors.TaskError("Failed to get table with cleanup data")
                    return CmntEnv.yt_abs_path(table_name)
                except:
                    logging.error('Not json or bad response: %s' % out)
                    raise
            logging.error('cmnt-yql failed with message:')
            logging.error(out)
            return None

    def run_cleanup_tool(self, data_table, tool_name):
        """
        Runs cleanup tool.

        Tools have similar invoke interface, but expect data_table with appropriate schemas and data.
        """
        logging.info("Run cleanup tool %s" % tool_name)
        cmnt_cleanup_tool = self.get_clean_cli_path()

        args = [
            cmnt_cleanup_tool,
            '--env', CmntEnv.get_ydb_env(self.Parameters.env),
            '--token', self.Parameters.ydb_token.data(),
            '--yt-cluster', self.Parameters.yt_proxy,
            '--yt-token', self.Parameters.yt_token.data(),
            '--limit', str(self.Parameters.cleanup_limit),
            '--threads', str(self.Parameters.cleanup_threads),
            '--data-table', str(data_table)
        ]

        with sdk2.helpers.ProcessLog(self, logger=tool_name) as pl:
            p = sp.Popen(
                args,
                stdout=pl.stdout,
                stderr=pl.stderr
            )
            out, err = p.communicate()
            if p.returncode != 0:
                raise common.errors.TaskError("Could not cleanup data")


class CmntDbCleanupLastAction(sdk2.Task, CmntDbCleanupBase):
    """ A task, which cleans obsolete rows from cmnt_last_action_data table of Commentator's database. """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            sdk_environments.PipEnvironment("yandex-yt"),
        ]
        ram = 24 * 1024
        disk_space = 50000
        cores = 2

        class Caches(sdk2.Requirements.Caches):
            pass  # the task does not use any shared caches

    class Parameters(CmntDbCleanupBase.Parameters):
        with sdk2.parameters.Group("Exec parameters") as run_block:
            days = sdk2.parameters.Integer("Cleanup older than given amount of days", default=DEFAULT_DAYS, required=True)

    def get_cmnt_yql_args(self):
        args = super(CmntDbCleanupLastAction, self).get_cmnt_yql_args()
        args.extend([
            '--param', 'days=%s' % str(self.Parameters.days),
            '--run', 'rc:cleanup_action_data.sql'
        ])
        return args

    def on_execute(self):
        import yt.wrapper as yt

        self.yt_client = yt.YtClient(self.Parameters.yt_proxy, self.Parameters.yt_token.data())

        data_table = self.dump_obsolete_data()
        self.run_cleanup_tool(data_table, "cmnt-cleanup-last-action")

        logging.info("Drop old cleanup data tables")
        CmntEnv.remove_old_data(self.yt_client, self.Parameters.yt_cleanup_data_folder, self.Parameters.cleanup_tables_count, False)

        logging.info("Done")


class CmntDbCleanupChats(sdk2.Task, CmntDbCleanupBase):
    """ A task, which cleans obsolete rows from cmnt_chat_data and cmnt_chat_desc tables of Commentator's database. """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            sdk_environments.PipEnvironment("yandex-yt"),
        ]
        ram = 24 * 1024
        disk_space = 50000
        cores = 2

        class Caches(sdk2.Requirements.Caches):
            pass  # the task does not use any shared caches

    class Parameters(CmntDbCleanupBase.Parameters):
        with sdk2.parameters.Group("Exec parameters") as run_block:
            services = sdk2.parameters.String("Services to clean", required=True)
            days = sdk2.parameters.Integer("Cleanup older than given amount of days", default=DEFAULT_DAYS, required=True)

    def formatted_services(self):
        raw_srv = self.Parameters.services
        res = []
        for srv_id in raw_srv.split():
            res.append('"' + srv_id + '"')
        return ', '.join(res)

    def get_cmnt_yql_args(self):
        args = super(CmntDbCleanupChats, self).get_cmnt_yql_args()
        args.extend([
            '--param', 'days=%s' % str(self.Parameters.days),
            '--param', 'srv=%s' % self.formatted_services(),
            '--run', 'rc:cleanup_chats.sql'
        ])
        return args

    def on_execute(self):
        import yt.wrapper as yt

        self.yt_client = yt.YtClient(self.Parameters.yt_proxy, self.Parameters.yt_token.data())

        data_table = self.dump_obsolete_data()
        self.run_cleanup_tool(data_table, "cmnt-cleanup-chats")

        logging.info("Drop old cleanup data tables")
        CmntEnv.remove_old_data(self.yt_client, self.Parameters.yt_cleanup_data_folder, self.Parameters.cleanup_tables_count, False)

        logging.info("Done")


class CmntDbCleanupChatDescOrphans(sdk2.Task, CmntDbCleanupBase):
    """ A task, which cleans orphaned chat descriptions of Commentator's database. """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            sdk_environments.PipEnvironment("yandex-yt"),
        ]
        ram = 2 * 1024
        disk_space = 5000
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass  # the task does not use any shared caches

    class Parameters(CmntDbCleanupBase.Parameters):
        pass

    def get_cmnt_yql_args(self):
        args = super(CmntDbCleanupChatDescOrphans, self).get_cmnt_yql_args()
        args.extend([
            '--run', 'rc:cleanup_orphan_chatdesc.sql'
        ])
        return args

    def on_execute(self):
        import yt.wrapper as yt

        self.yt_client = yt.YtClient(self.Parameters.yt_proxy, self.Parameters.yt_token.data())

        data_table = self.dump_obsolete_data()  # just make chat desc orphans table
        self.run_cleanup_tool(data_table, "cmnt-cleanup-desc-orphans")

        logging.info("Drop old cleanup data tables")
        CmntEnv.remove_old_data(self.yt_client, self.Parameters.yt_cleanup_data_folder, self.Parameters.cleanup_tables_count, False)

        logging.info("Done")
