import os
import logging
import tarfile

from sandbox import common
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.client as ctc

from sandbox.projects.cmnt.cmnt_env import CmntEnv
from sandbox.projects.cmnt import CMNT_BUNDLE


CMNT_BRANCH_PREFIX = 'cmnt/'


class CmntPumpkinSettingsGenerator(sdk2.Task):
    """Task to generate resource with settings for CMNT Pumpkin servant."""

    class Requirements(sdk2.Task.Requirements):
        ram = 2 * 1024
        disk_space = 5000
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass  # the task does not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600  # 10 minutes

        with sdk2.parameters.Group("Input data") as bin_block:
            ydb_token = sdk2.parameters.Vault("Vault secret name with YDB token",
                                              default='robot-cmnt:ydb_token',
                                              required=True)
            s3_credentials = sdk2.parameters.Vault("Vault secret with S3 uploader credentials",
                                                   default='robot-cmnt:S3_UPLOADER_CREDENTIALS',
                                                   required=True)

        with sdk2.parameters.RadioGroup("Environment") as env:
            env.values[CmntEnv.ENV_TESTING] = env.Value(value=CmntEnv.ENV_TESTING, default=True)
            env.values[CmntEnv.ENV_ALPHA] = env.Value(value=CmntEnv.ENV_ALPHA)
            env.values[CmntEnv.ENV_PRODUCTION] = env.Value(value=CmntEnv.ENV_PRODUCTION)

    @staticmethod
    def _get_branch_ver(resource, released_value=None):
        attr = getattr(resource, 'branch', '')
        if not attr:
            return (0, 0)

        if released_value is not None and getattr(resource, 'released') != released_value:
            return (0, 0)

        fs = attr.lstrip(CMNT_BRANCH_PREFIX).split('.')
        if len(fs) != 2:
            return (0, 0)

        return (int(fs[0]), int(fs[1]))

    def _get_cmnt_bundle_resource_trunk(self):
        return CMNT_BUNDLE.find(attrs={"released": "unstable"}).first()

    def _get_cmnt_bundle_resource_branch(self):
        resources = CMNT_BUNDLE.find(attrs={}).limit(50)

        max_stable_resource = max(resources, key=lambda r: self._get_branch_ver(r, 'stable'))
        max_prestable_resource = max(resources, key=lambda r: self._get_branch_ver(r, 'prestable'))

        logging.info('max_stable_resource: %s' % str(max_stable_resource))
        logging.info('max_prestable_resource: %s' % str(max_prestable_resource))

        if self.Parameters.env == CmntEnv.ENV_ALPHA and \
           self._get_branch_ver(max_prestable_resource) > self._get_branch_ver(max_stable_resource):
            return max_prestable_resource

        return max_stable_resource

    def _extract_generator_bin(self, cmnt_bundle_tar_gz_path):
        generator_bin = 'cmnt-pumpkin-settings-generator'
        result_path = os.path.join('.', 'app', generator_bin)

        logging.info('Extracting generator binary from %s' % cmnt_bundle_tar_gz_path)
        with tarfile.open(cmnt_bundle_tar_gz_path, 'r:gz') as tar:
            tar.extract(os.path.join('app', generator_bin), '.')
            if not os.path.exists(result_path):
                raise common.errors.TaskError("Unexpected error: could not find extracted file '%s' from tarball '%s'" % (result_path, cmnt_bundle_tar_gz_path))

            return result_path

        return None

    def get_generator_bin_path(self):
        cmnt_bundle_resource = None
        if self.Parameters.env == CmntEnv.ENV_TESTING:
            cmnt_bundle_resource = self._get_cmnt_bundle_resource_trunk()
        else:
            cmnt_bundle_resource = self._get_cmnt_bundle_resource_branch()

        if not cmnt_bundle_resource:
            raise common.errors.TaskError('Could not find appropriate resource of type %s' % CMNT_BUNDLE)

        resource_data = sdk2.ResourceData(cmnt_bundle_resource)
        logging.info("Using generator binary from resource %s" % str(cmnt_bundle_resource))

        return self._extract_generator_bin(str(resource_data.path))

    def on_execute(self):
        tool_name = self.get_generator_bin_path()
        if not tool_name:
            raise common.errors.TaskError('Could not get generator binary path')

        logging.info("Run generator tool %s" % tool_name)
        args = [
            tool_name,
            '--env', CmntEnv.get_ydb_env(self.Parameters.env),
            '--token', self.Parameters.ydb_token.data(),
            '--s3-credentials', self.Parameters.s3_credentials.data()
        ]

        with sdk2.helpers.ProcessLog(self, logger=tool_name):
            p = sp.Popen(
                args,
                stdout=sp.PIPE,
                stderr=sp.PIPE
            )
            out, err = p.communicate()
            if p.returncode != 0:
                raise common.errors.TaskError(err)

            logging.info("Done!")
