import logging
import os

from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import parameters

from sandbox.projects import resource_types
from sandbox.projects.images.bans import yt_task
from sandbox.projects.collections import resources as collections_resources
from sandbox.projects.common import apihelpers
from sandbox.projects.common.mediasearch import ban as media_ban
from sandbox.projects.common import utils


class BoardsIndexPrefixParameter(parameters.SandboxStringParameter):
    name = "boards_index_prefix"
    description = "Boards index prefix"
    default_value = "//home/collections/boards_search"
    required = True


class BanFromTimestamp(parameters.SandboxIntegerParameter):
    name = "boards_ban_timestamp"
    description = "From (timestamp)"


class BanLagFromTimestamp(parameters.SandboxIntegerParameter):
    name = "boards_ban_timestamp_lag"
    description = "Lag (sec)"
    default_value = 0


class CollectionsBuildAndReleaseBoardsBan(yt_task.YtTask, media_ban.ImagesBaseReleaseBanTask):
    """
        Build boards ban
    """

    type = "COLLECTIONS_BUILD_BOARDS_BAN"
    client_tags = media_ban.ImagesBaseReleaseBanTask.client_tags & yt_task.YtTask.client_tags

    vault_owner = "IMAGES-BAN"

    input_parameters = \
        media_ban.ImagesBaseReleaseBanTask.create_input_parameters(enable_semaphore=True) + \
        yt_task.YtTask.create_params("arnold.yt.yandex.net", "//home/logfeller/logs/collections-banned-log") + \
        (BoardsIndexPrefixParameter, BanFromTimestamp, BanLagFromTimestamp)

    release_subject = "collections/middle/boards-ban-{timestamp}"
    release_comment = "collections boards ban file"
    push_signal_name = "collections_boards_ban"

    ban_resource = collections_resources.CollectionsMiddlesearchBoardsBanTrie
    version_resource = collections_resources.CollectionsMiddlesearchBoardsBanTrieVer
    release_resources = (ban_resource, version_resource)

    def _build_ban(self):
        boards_index_prefix = utils.get_or_default(self.ctx, BoardsIndexPrefixParameter)

        version_path = self.abs_path(self.version_resource)

        yt_token = self.get_vault_data(self.vault_owner, "yt_token")

        trie_path = self.abs_path(self.ban_resource)
        builder = self._tool(collections_resources.CollectionsBoardsBanBuilder)

        extra_params = []
        ban_timestamp = utils.get_or_default(self.ctx, BanFromTimestamp)
        if ban_timestamp:
            extra_params += ["--timestamp", ban_timestamp]

        ban_lag = utils.get_or_default(self.ctx, BanLagFromTimestamp)
        if ban_lag:
            extra_params += ["--lag", str(ban_lag)]

        process.run_process([
            builder,
            "--server", utils.get_or_default(self.ctx, yt_task.YtProxyParameter),
            "--index-prefix", utils.get_or_default(self.ctx, BoardsIndexPrefixParameter),
            "--board-ban-log-prefix", utils.get_or_default(self.ctx, yt_task.YtTableParameter),
            "--output", trie_path,
            "--text-output", version_path] + extra_params, environment={"YT_TOKEN": yt_token, "YT_LOG_LEVEL": "INFO"}, outputs_to_one_file=False, log_prefix="boards_ban")

        if self._skip_build(version_path, self.version_resource):
            logging.info("Nothing to release")
            return 0

        self._register_ban(self.descr, version_path, self.version_resource)
        self._register_ban(self.descr, trie_path, self.ban_resource)

        return os.stat(trie_path).st_size

    def _skip_build(self, version_path, version_resource):
        if self.ctx.get(media_ban.ForceBuildParameter.name):
            return False

        released_resource = apihelpers.get_last_released_resource(version_resource, release_status=self.ctx.get(media_ban.ReleaseStatusParameter.name))
        if not released_resource:
            return False

        released_path = self.sync_resource(released_resource.id)

        with open(version_path) as version_file:
            with open(released_path) as released_file:
                return version_file.read().strip() == released_file.read().strip()

    def _test_ban(self, build_task_id):
        return []


__Task__ = CollectionsBuildAndReleaseBoardsBan
