# -*- coding: utf-8 -*-

import os
import logging
import tarfile
import json

import sandbox.common.types.misc as ctm

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.process import run_process

FRONTEND_PACKAGE_RESOURCE = 'YASAP_PDB_FRONTEND_PACKAGE'
ARCHIVE = 'pdb-frontend.tar.gz'
ARCHIVE_TESTING = 'pdb-frontend-testing.tar.gz'
STATICS_SERVICE_ID = 'pdb_static_production'
TAR_MODE = 'w:gz'


def unarchive(fname, newfname=None):
    if tarfile.is_tarfile(fname):
        if not newfname:
            idx = fname.find('.tar')
            newfname = fname[:idx]
        tar = tarfile.open(fname)
        tar.extractall(path=newfname)
        tar.close()
        os.unlink(fname)
        return newfname
    return fname


class NodeJSPackage(LastReleasedResource):
    name = 'YASAP_PDB_NODEJS_PACKAGE'
    description = 'node.js'
    resource_type = resource_types.YASAP_PDB_NODEJS_PACKAGE
    limit = 5


class PdbFrontendBranch(SandboxStringParameter):
    name = "branch"
    description = "branch"
    default_value = "master"


class PdbFrontendRebase(SandboxBoolParameter):
    name = "rebase"
    description = "rebase"
    default_value = False


class PdbFrontendRebaseIgnore(SandboxBoolParameter):
    name = "rebase_ignore"
    description = "ignore rebase errors"
    default_value = False


class PdbFrontendYarn(SandboxBoolParameter):
    name = "yarn"
    description = "Install dependencies with Yarn"
    default_value = False


class PdbFrontendProd(SandboxBoolParameter):
    name = "production"
    description = "Production"
    default_value = True


class PdbFrontendTesting(SandboxBoolParameter):
    name = "testing"
    description = "Testing"
    default_value = True


class PdbFrontendExtensions(SandboxBoolParameter):
    name = "extensions"
    description = "Build extension"
    default_value = False


class PdbFrontendPRNumber(SandboxStringParameter):
    name = "pr_number"
    description = "Pull request number"


class PdbFrontendSize(SandboxBoolParameter):
    name = "size"
    description = "Check bundles size"
    default_value = False


class BuildPdbFrontend(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'BUILD_YASAP_PDB_FRONTEND'
    dns = ctm.DnsType.DNS64
    GIT_URL = 'https://github.yandex-team.ru/pdb/pdb-frontend.git'
    _footer = ''

    input_parameters = [
        NodeJSPackage,
        PdbFrontendBranch,
        PdbFrontendRebase,
        PdbFrontendRebaseIgnore,
        PdbFrontendYarn,
        PdbFrontendProd,
        PdbFrontendTesting,
        PdbFrontendExtensions,
        PdbFrontendPRNumber,
        PdbFrontendSize
    ]

    execution_space = 1200

    release_to = ['ftdebugger', "tarkus", "menjoy"]

    def _create_tar(self, filenames, tarname=ARCHIVE, tarmode=TAR_MODE):
        tar = tarfile.open(tarname, tarmode)

        def excludes_fn(name):
            return '.git' in name

        for fname in filenames:
            logging.info('Add %s to archive', fname)
            tar.add(fname, exclude=excludes_fn)
        tar.close()

    def _sync_res(self, resource):
        path = self.sync_resource(self.ctx[resource.name])
        copy_path(path, self.path())
        return path

    def _checkout_sources(self, tag, rebase, rebase_ignore):
        logging.info('Checking out %s tag of pdb-frontend', tag)
        src_dir = self.path('pdb-frontend')
        run_process(['git', 'clone', self.GIT_URL, src_dir], log_prefix='git_clone')
        run_process(['git', 'checkout', tag], work_dir=src_dir, log_prefix='git_checkout')

        try:
            if rebase:
                run_process(['git', 'config', '--global', 'user.email', 'polubob@yandex.ru'], work_dir=src_dir,
                            log_prefix='git_config')
                run_process(['git', 'config', '--global', 'user.name', 'polubob'], work_dir=src_dir,
                            log_prefix='git_config')
                run_process(['git', 'rebase', 'master'], work_dir=src_dir, log_prefix='git_rebase')
        except SandboxSubprocessError:
            if rebase_ignore:
                run_process(['git', 'rebase', '--abort'], work_dir=src_dir, log_prefix='git_rebase_abort')
            else:
                raise

        current_tag = run_process(
            ['git', 'describe', '--tags'],
            outs_to_pipe=True,
            work_dir=src_dir
        ).stdout.read().strip()

        self.ctx['git-tag'] = current_tag
        logging.info('git-tag is %s', current_tag)

        return src_dir

    def _clean_mess(self, sources_folder):
        if self.ctx['extensions']:
            run_process(['rm', '-rf', os.path.join(sources_folder, 'dist/chrome')], log_prefix='clean',
                        work_dir=sources_folder)

    def _calc_size(self, sources_folder, node_bin, gulp_bin, environment, project_env):
        if self.ctx['size']:
            run_process(
                [node_bin, gulp_bin, 'size:calc', '--' + project_env],
                log_prefix='gulp_size',
                work_dir=sources_folder,
                environment=environment
            )

            if project_env == 'production':
                self.create_resource(
                    description='Bundle size',
                    resource_path=os.path.join(sources_folder, 'dist/web/size.json'),
                    resource_type=resource_types.YASAP_PDB_FRONTEND_SIZE,
                    arch='any'
                )

    def on_execute(self):
        files = []

        sources_folder = self._checkout_sources(self.ctx['branch'], self.ctx['rebase'], self.ctx['rebase_ignore'])

        node_archive = self._sync_res(NodeJSPackage)
        node_base_name = os.path.basename(node_archive)
        node_root = unarchive(node_base_name)
        bin_root = os.path.abspath(os.path.join(node_root, 'bin'))
        node_bin = os.path.abspath(os.path.join(bin_root, 'node'))
        npm_bin = os.path.abspath(os.path.join(bin_root, 'npm'))
        yarn_bin = os.path.abspath(os.path.join(bin_root, 'yarn'))
        gulp_bin = os.path.abspath(os.path.join(sources_folder, 'node_modules/.bin/gulp'))

        environment = os.environ.copy()
        environment['PATH'] = bin_root + ':' + os.environ['PATH']
        environment['GITHUB_OAUTH_TOKEN'] = self.get_vault_data('YASAP', 'GITHUB_OAUTH_TOKEN')
        environment['GITHUB_HASH'] = self.ctx['branch']

        files.append(os.path.basename(node_root))
        files.append(os.path.basename(sources_folder))

        # Build app
        if self.ctx['yarn']:
            run_process(
                [node_bin, yarn_bin, 'config', 'set', 'registry', 'https://npm.yandex-team.ru'],
                log_prefix='yarn_config'
            )

            run_process(
                [node_bin, yarn_bin, 'install', '--production'],
                work_dir=sources_folder,
                environment=environment,
                log_prefix="yarn_install"
            )
        else:
            run_process(
                [node_bin, npm_bin, '--registry=https://npm.yandex-team.ru', 'install', '--production'],
                log_prefix='npm_install',
                work_dir=sources_folder,
                environment=environment
            )

        # Build production

        if self.ctx['production']:
            run_process(
                [node_bin, npm_bin, 'run', 'release'],
                log_prefix='release',
                work_dir=sources_folder,
                environment=environment
            )

            # desktop_ls_output = run_process(
            #     ['ls', '-skahS1R'],
            #     outs_to_pipe=True,
            #     work_dir=sources_folder,
            #     environment=environment
            # ).stdout.read().strip()
            #
            # mobile_ls_output = run_process(
            #     ['ls', '-skahS1R'],
            #     outs_to_pipe=True,
            #     work_dir=sources_folder,
            #     environment=environment
            # ).stdout.read().strip()
            #
            # self.descr = self.descr + '\n\nDesktop:\n' + desktop_ls_output + '\n\nMobile:\n' + mobile_ls_output

            # Clean
            self._clean_mess(sources_folder)

            # Build archive
            self._create_tar(files)

            self.create_resource(
                description=FRONTEND_PACKAGE_RESOURCE,
                resource_path=ARCHIVE,
                resource_type=resource_types.YASAP_PDB_FRONTEND_PACKAGE,
                arch='any'
            )

            self._calc_size(sources_folder, node_bin, gulp_bin, environment, 'production')

        # Build testing
        if self.ctx['testing']:
            if self.ctx['pr_number'] and self.ctx['pr_number'] != 'test':
                environment['PR'] = self.ctx['pr_number']

            run_process(
                [node_bin, npm_bin, 'run', 'release:testing'],
                log_prefix='release',
                work_dir=sources_folder,
                environment=environment
            )

            # Clean
            self._clean_mess(sources_folder)

            # Build archive
            self._create_tar(files, tarname=ARCHIVE_TESTING)

            # Create resource
            self.create_resource(
                description='YASAP_PDB_FRONTEND_TESTING_PACKAGE',
                resource_path=ARCHIVE_TESTING,
                resource_type=resource_types.YASAP_PDB_FRONTEND_TESTING_PACKAGE,
                arch='any'
            )

            self._calc_size(sources_folder, node_bin, gulp_bin, environment, 'testing')

            # Create extensions resources
            if self.ctx['extensions']:
                zip_file = ''

                for filename in os.listdir(os.path.join(sources_folder, 'dist')):
                    root, ext = os.path.splitext(filename)

                    if root.startswith('chrome') and ext == '.zip':
                        zip_file = filename

                if zip_file:
                    self.create_resource(
                        description='YASAP_PDB_FRONTEND_CHROME_EXTENSION',
                        resource_path=os.path.join(sources_folder, 'dist', zip_file),
                        resource_type=resource_types.YASAP_PDB_FRONTEND_CHROME_EXTENSION,
                        arch='any'
                    )

    @property
    def footer(self):
        if self.ctx['size']:
            path = os.path.join(self.path('pdb-frontend'), 'dist/web/size.html')

            if os.path.isfile(path):
                report = open(path, 'r')
                return report.read()

        return "Footer is not ready"

    def generate_new_service_resource(self):
        logging.info('git-tag is %s', self.ctx['git-tag'])
        service_resource = {
            'task_id': str(self.id),
            'resource_type': FRONTEND_PACKAGE_RESOURCE,
            'task_type': self.type,
            'local_path': '{}.tar.gz'.format(self.ctx['git-tag'])
        }
        return service_resource

    def update_sandbox_files_on_release(self, sandbox_files, new_sandbox_file, max_count=5):
        def by_task_id_comparator(a, b):
            a_id = int(a['task_id'])
            b_id = int(b['task_id'])
            return a_id - b_id

        def by_task_type_filter(file):
            return file['task_type'] == self.type

        other_files = filter(lambda a: not by_task_type_filter(a), sandbox_files)
        needed_files = sorted(filter(by_task_type_filter, sandbox_files), by_task_id_comparator)
        needed_files = filter(lambda f: f['task_id'] != new_sandbox_file['task_id'], needed_files)
        needed_files.append(new_sandbox_file)
        return other_files + needed_files[-max_count:]

    def on_release(self, additional_parameters):
        service_id = STATICS_SERVICE_ID

        nanny_oauth_token = self.get_vault_data('YASAP', 'nanny_oauth_token')
        nanny_client = nanny.NannyClient(api_url='http://nanny.yandex-team.ru/', oauth_token=nanny_oauth_token)

        service_current_state = nanny_client.get_service_current_state(service_id)
        active_snapshot = next(s for s in service_current_state['content']['active_snapshots'] if
                               s['state'] in ['ACTIVE', 'DEACTIVATE_PENDING'])

        if active_snapshot:
            service_runtime_attrs = nanny_client.get_history_runtime_attrs(active_snapshot['snapshot_id'])
            logging.debug('service_runtime_attrs dump: %s',
                          json.dumps(service_runtime_attrs, sort_keys=True, indent=4, separators=(',', ': ')))
            service_resources_content = service_runtime_attrs['content']['resources']
            snapshot_id = active_snapshot['snapshot_id']
        else:
            service_resources = nanny_client.get_service_resources(service_id)
            logging.debug('service_resources dump: %s',
                          json.dumps(service_resources, sort_keys=True, indent=4, separators=(',', ': ')))
            service_resources_content = service_resources['content']
            snapshot_id = service_resources['snapshot_id']

        logging.info('%s: Updating nanny service_resources', service_id)

        new_service_resource = self.generate_new_service_resource()
        service_resources_content['sandbox_files'] = self.update_sandbox_files_on_release(
            service_resources_content['sandbox_files'],
            new_service_resource
        )

        logging.debug('new service_resources_content dump: %s',
                      json.dumps(service_resources_content, sort_keys=True, indent=4, separators=(',', ': ')))

        comment = 'Automatically add new resource: {}'.format(json.dumps(new_service_resource))
        logging.info('Generated comment: %s', comment)

        snapshot_info = nanny_client.update_service_resources(service_id, {
            'content': service_resources_content,
            'comment': comment,
            'snapshot_id': snapshot_id,
        })

        logging.info('%s/%s: nanny service was updated successfully', service_id, snapshot_id)
        logging.debug('%s: snapshot_info %s', service_id,
                      json.dumps(snapshot_info, sort_keys=True, indent=4, separators=(',', ': ')))

        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        SandboxTask.on_release(self, additional_parameters)

        logging.info('on_release success')


__Task__ = BuildPdbFrontend
