# -*- coding: utf-8 -*-

import os
import shutil
import hashlib
import logging
import platform

import requests
from sandbox import common

import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.errors as sdk_errors
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.projects.resource_types


class BuildPdbNodeJSPackage(sdk_task.BuildForAllMode):
    """
    Build NodeJS binary package of specified version
    """

    class Version(sdk_parameters.SandboxStringParameter):
        name = "version"
        description = "NodeJS version"
        default_value = "4.2.2"

    input_parameters = [Version]
    type = "BUILD_YASAP_PDB_NODEJS_PACKAGE"

    resource_type = str(sandbox.projects.resource_types.YASAP_PDB_NODEJS_PACKAGE)

    @classmethod
    def required_resources(cls):
        return [(cls.resource_type, "NodeJS package for {plat}".format(plat=platform.platform()))]

    def on_execute(self):
        version = self.ctx[self.Version.name]
        srcname = "node-v{}".format(version)
        srcfile = srcname + "-linux-x64.tar.gz"
        checksum_url = "http://nodejs.org/dist/v{}/SHASUMS256.txt.asc".format(version)
        package_url = "http://nodejs.org/dist/v{}/{}".format(version, srcfile)

        logging.info("Looking for source package's checksum at '%s'.", checksum_url)
        r = requests.get(checksum_url)
        checksum = next(
            (
                parts[0] for parts in (
                    line.split() for line in r.text.split('\n')
                ) if len(parts) == 2 and parts[1] == srcfile
            ),
            None
        )
        if not checksum:
            raise sdk_errors.SandboxTaskFailureError(
                "Unable to find a checksum for source file '{}'".format(srcfile)
            )
        logging.debug("sha256 checksum of file '%s' is '%s'.", srcfile, checksum)

        with self.current_action("Downloading sources package from '{}'".format(package_url)):
            r = requests.get(package_url, stream=True)
            sha256 = hashlib.sha256()
            with open(srcfile, 'wb') as f:
                for chunk in r.iter_content(chunk_size=4095):
                    if chunk:  # filter out keep-alive new chunks
                        sha256.update(chunk)
                        f.write(chunk)
                        f.flush()

        logging.debug("sha256 of downloaded file '%s' is '%s'.", srcfile, sha256.hexdigest())
        if checksum != sha256.hexdigest():
            raise sdk_errors.SandboxTaskFailureError(
                "Downloaded file's '{}' checksum '{}' doesn't match expected '{}'.".format(
                    package_url, sha256.hexdigest(), checksum
                )
            )

        srcpath = self.abs_path(srcname)
        with self.current_action("Extracting archive to '{}'".format(srcpath)):
            if os.path.exists(srcpath):
                shutil.rmtree(srcpath)
            os.mkdir(srcpath)
            sdk_process.run_process(
                ["tar", "xf", srcfile, "--strip-components=1", "--directory", srcpath],
                log_prefix="extract"
            )

        with self.current_action("Install yarn"):
            bindir = os.path.join(srcpath, "bin")
            yarn = os.path.join(srcpath, "node_modules", ".bin", "yarn")

            sdk_process.run_process(
                ["./bin/node", "bin/npm", "install", "yarn", "--registry=https://npm.yandex-team.ru"],
                log_prefix="npm", work_dir=srcpath
            )

            common.fs.make_symlink(
                os.path.relpath(yarn, bindir),
                os.path.join(bindir, "yarn")
            )

        with self.current_action("Packaging distributive"):
            distpkg = self.abs_path("node.tar.gz")
            sdk_process.run_process(
                ["tar", "-zcvf", distpkg, "-C", srcpath, '.'],
                log_prefix="pack"
            )

        # Create resource
        self.create_resource(
            description='YASAP_PDB_NODEJS_PACKAGE',
            resource_path="node.tar.gz",
            resource_type=sandbox.projects.resource_types.YASAP_PDB_NODEJS_PACKAGE,
            arch='any'
        )


__Task__ = BuildPdbNodeJSPackage
