# coding: utf-8
import requests
from requests.exceptions import RequestException
import logging

from sandbox import sdk2

import sandbox.common.types.task as ctt
import sandbox.common.types.misc as ctm

from itertools import izip_longest


def grouper(iterable, n, fillvalue=None):
    args = [iter(iterable)] * n
    return izip_longest(*args, fillvalue=fillvalue)


logger = logging.getLogger(__name__)


class CollectionsBulkBan(sdk2.Task):
    """
        Collections bulk ban task
    """
    class Requirements(sdk2.Requirements):
        disk_space = 1200
        dns = ctm.DnsType.DNS64
        semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='COLLECTIONS_BULK_BAN_SEMAPHORE'),
            ],
        )

    class Parameters(sdk2.Parameters):
        targets = sdk2.parameters.String(
            'Targets to ban',
            required=True,
            multiline=True,
        )
        url = sdk2.parameters.String('URL', default='https://pdb-admin.n.yandex-team.ru/collections/api/actions')
        chunk_size = sdk2.parameters.Integer('chunk_size', default=10)

    @property
    def chunked_targets(self):
        targets = str(self.Parameters.targets).split()
        for chunk in grouper(targets, int(self.Parameters.chunk_size)):
            yield chunk

    def on_execute(self):
        _comment = "spam"
        _reason = "Другое нарушение ПС"
        _type = "ban"
        _restrict = "editing"
        _content = "hide"
        _subject = "user"

        s = requests.Session()

        for i, _targets in enumerate(self.chunked_targets):
            _targets = [t for t in _targets if t is not None]
            if not _targets:
                continue
            try:
                r = s.post(
                    str(self.Parameters.url),
                    json={
                        "comment": _comment,
                        "subject": _subject,
                        "type": _type,
                        "reason": _reason,
                        "restrict": _restrict,
                        "targets": _targets,
                        "content": _content,
                    },
                    headers={'Authorization': 'OAuth ' + sdk2.Vault.data('YASAP', 'backend_oauth_token')},
                )
                r.raise_for_status()
                logger.info("Finished part %d, starting from %s ending with %s", _targets[0], _targets[-1])
            except RequestException:
                logger.exception("Request error")
