# -*- coding: utf-8 -*-

import logging
import re

import sandbox.common.types.task as ctt

from datetime import datetime
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.collections.CollectionsFrontendAssessorsRun import CollectionsFrontendAssessorsRun
from sandbox.projects.common import link_builder as lb
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.components.configs.collections_backend import CollectionsBackendCfg
from sandbox.projects.release_machine.notify_helper import telegram as notify_tg
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine import security as rm_sec
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.vcs.svn import Arcadia


class CollectionsBackendAssessorsRun(sdk2.Task):
    """
        Collections backend assessors run
    """

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Arcadia info'):
            arcadia_url = sdk2.parameters.String('Arcadia url')
        with sdk2.parameters.Group('Notifications'):
            notify_telegram = sdk2.parameters.Bool('Notify in telegram', default=False)
            notify_startrek = sdk2.parameters.Bool('Notify in startrek', default=False)
        with sdk2.parameters.Group('Assessors'):
            preset = sdk2.parameters.String('Preset', required=True, default='backend')
            dry_run = sdk2.parameters.Bool('Dry run', default=False)

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment('startrek-client==1.5.8'),
        ]
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    TESTPALM_LINK_TEMPLATE = 'https://testpalm2.yandex-team.ru/collections_v{version}'
    PODBAS_DASHBOARD_LINK = 'https://st.yandex-team.ru/dashboard/15953'

    @staticmethod
    def _get_notification_template(with_release_ticket=False):
        lines = [
            'Assessors started on {target}',
            'SB task {task_link}',
            'Testpalm {testpalm_link}',
            'PODBAS {podbas_dashboard}',
        ]
        if with_release_ticket:
            lines.append('Backend release ticket {release_ticket}')
        return '\n'.join(lines)

    def _get_testpalm_link(self):
        return self.TESTPALM_LINK_TEMPLATE.format(version=self.Context.assessors_version)

    def _build_telegram_notification(self):
        release_ticket = lb.st_link(
            self.Context.release_ticket, self.Context.release_ticket
        ) if self.Context.release_ticket else 'unknown'
        return self._get_notification_template(with_release_ticket=True).format(
            target=self.Context.assessors_target,
            task_link=lb.task_link(self.id, link_name=str(self.id)),
            testpalm_link=lb.HREF_TO_ITEM.format(
                link=self._get_testpalm_link(),
                name='link',
            ),
            podbas_dashboard=lb.HREF_TO_ITEM.format(
                link=self.PODBAS_DASHBOARD_LINK,
                name='dashboard',
            ),
            release_ticket=release_ticket,
        )

    def _build_telegram_notification_on_fail(self):
        tmpl = '\n'.join([
            'Failed to start assessors on {target}',
            'SB task {task_link}',
        ])
        return tmpl.format(
            target=self.Context.assessors_target,
            task_link=lb.task_link(self.id, link_name=str(self.id)),
        )

    def _build_startrek_notification(self):
        return self._get_notification_template(with_release_ticket=False).format(
            target=self.Context.assessors_target,
            task_link=lb.task_wiki_link(self.id, link_name=str(self.id)),
            testpalm_link=lb.WIKILINK_TO_ITEM.format(
                link=self._get_testpalm_link(),
                name='link',
            ),
            podbas_dashboard=lb.WIKILINK_TO_ITEM.format(
                link=self.PODBAS_DASHBOARD_LINK,
                name='dashboard',
            ),
        )

    def _parse_arcadia_url(self):
        if self.Context.assessors_version:
            return

        self.Context.assessors_target = CollectionsBackendCfg.name
        self.Context.assessors_version = self.Parameters.preset
        if not self.Parameters.arcadia_url:
            return
        parsed_url = Arcadia.parse_url(self.Parameters.arcadia_url)

        if parsed_url.revision:
            self.Context.assessors_target = 'r{}'.format(parsed_url.revision)
            self.Context.assessors_version = 'r{}-{}'.format(parsed_url.revision, self.Parameters.preset)

        if parsed_url.tag:
            tag_regex = r'.*-([0-9]+)-([0-9]+)'
            match = re.match(tag_regex, parsed_url.tag)
            self.Context.release_num = int(match.group(1))
            self.Context.assessors_target = parsed_url.tag
            self.Context.assessors_version = '{}-{}-{}'.format(int(match.group(1)), int(match.group(2)), self.Parameters.preset)

        self.Context.assessors_version = '{}-{}'.format(
            self.Context.assessors_version,
            datetime.now().strftime('%Y%m%d-%H%M%S')
        )
        self.Context.save()

    def _get_st_helper(self):
        token = rm_sec.get_rm_token(self)
        return STHelper(token)

    def _find_release_ticket(self):
        if not self.Context.release_num:
            return
        c_info = rmc.COMPONENTS[CollectionsBackendCfg.name]()
        st_issue = self._get_st_helper().find_ticket_by_release_number(
            self.Context.release_num,
            c_info,
            fail=False
        )
        if st_issue:
            self.Context.release_ticket = st_issue.key
            self.Context.save()

    def _run_assessors_task(self):
        if self.Context.assessors_task:
            assessors_task = sdk2.Task[self.Context.assessors_task]
            if assessors_task.status != ctt.Status.SUCCESS:
                if self.Parameters.notify_telegram:
                    telegram_notification = self._build_telegram_notification_on_fail()
                    logging.info('Telegram notification text: %s', telegram_notification)
                    notify_tg(
                        self,
                        telegram_notification,
                        people=CollectionsBackendCfg.Notify.Telegram.release_chats,
                    )
                raise TaskFailure('Subtask failed')
            return

        assessors_task = CollectionsFrontendAssessorsRun(
            self,
            description='Run assessors on backend',
            hash='dev',
            dir='.',
            reuse_workcopy_cache=False,
            preset=self.Parameters.preset,
            version=self.Context.assessors_version,
            dry_run=self.Parameters.dry_run,
        )
        assessors_task.save()
        task_id = assessors_task.enqueue().id

        self.Context.assessors_task = task_id
        self.Context.save()

        raise sdk2.WaitTask(
            [task_id],
            (ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
            wait_all=True
        )

    def on_execute(self):
        self._parse_arcadia_url()
        self._run_assessors_task()
        self._find_release_ticket()

        telegram_notification = self._build_telegram_notification()
        logging.info('Telegram notification text: %s', telegram_notification)
        startrek_notification = self._build_startrek_notification()
        logging.info('Startrek notification text: %s', startrek_notification)

        if self.Parameters.notify_telegram:
            notify_tg(
                self,
                telegram_notification,
                people=CollectionsBackendCfg.Notify.Telegram.release_chats,
            )

        if self.Parameters.notify_startrek and self.Context.release_ticket:
            st_issue = self._get_st_helper().get_ticket_by_key(self.Context.release_ticket)
            st_issue.comments.create(text=startrek_notification)
            try:
                st_issue.transitions['testingByAssessors'].execute()
            except Exception as e:
                logging.exception('Failed to execute ticket transition: %s', e)


__TASK__ = CollectionsBackendAssessorsRun
