# -*- coding: utf-8 -*-

import os
import json
import re
import logging

import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.common.utils import singleton_property
from sandbox.projects.sandbox_ci import SANDBOX_CI_LXC_IMAGE
from sandbox.projects.collections.mixins import NannyMixin, get_vault
from sandbox.projects.collections.CollectionsDeployToNannyService.gencfg_api import GenCfgApi


class CollectionsDeployToNannyService(sdk2.Task, NannyMixin):
    """
        Collections custom nanny deploy PR
    """
    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.LOCAL

    class Parameters(sdk2.Parameters):
        kill_timeout = 3600
        _container = sdk2.parameters.Container(
            'Build environment',
            resource_type=SANDBOX_CI_LXC_IMAGE,
            platform='linux_ubuntu_16.04_xenial',  # necessary for correct default resource searching
            required=True,
        )
        with sdk2.parameters.Group('Stand params') as stand_params:
            stand_url_template = sdk2.parameters.String(
                'Stand URl template',
                required=True,
                default='pull-{}.collections.test.yandex.ru',
                description='URL, по которому будет доступен стенд для пулл-реквеста',
            )

        with sdk2.parameters.Group('Sandbox params') as sandbox_params:
            task_id = sdk2.parameters.Integer(
                'Sandbox task id',
                description='Id Sandbox задачи с ресурсом, который нужно задеплоить.',
                required=True,
            )

        with sdk2.parameters.Group('Nanny configuration params') as nanny_params:
            category = sdk2.parameters.String(
                'Category',
                required=True,
                default='/pdb/features/',
                description='На какаом уровне относительно корня должна лежать конфигурация сервиса.',
            )
            service_prefix = sdk2.parameters.String(
                'Service prefix',
                required=True,
                default='pdb_nodejs_test_feature_pr',
                description='Префикс нового или уже существующего сервиса {prefix}{pr_num}.',
            )
            default_service = sdk2.parameters.String(
                'Default service',
                required=True,
                default='pdb_nodejs_test',
                description='Конфигурация данного сервиса будет скопирована для новой.',
            )
            default_balancer_service = sdk2.parameters.String(
                'Default balancer service',
                required=True,
                default='pdb_nodejs_feature_balancer',
                description='Конфигурация данного сервиса будет обновлена, а именно nginx.conf.',
            )

            pr_number = sdk2.parameters.Integer(
                'Pull request number',
                required=True,
                default=0,
                description='Номер пулл-реквеста.',
            )

    @singleton_property
    def gencfg_client(self):
        return GenCfgApi(get_vault('YASAP', 'gencfg_oauth_token'))

    @property
    def service_id(self):
        return '{prefix}{num}'.format(
            prefix=self.Parameters.service_prefix,
            num=self.Parameters.pr_number
        )

    @property
    def description(self):
        return 'Feature instance for pr {}'.format(self.Parameters.pr_number)

    @property
    def curr_dir(self):
        return os.path.dirname(os.path.realpath(__file__))

    def get_extended_gencfg_groups(self, service):
        # runtime_attrs.content.instances.extended_gencfg_groups.
        properties_chain = ['runtime_attrs', 'content', 'instances', 'extended_gencfg_groups', 'groups']

        return self.get_service_value(service, properties_chain, [])

    def get_template_set_files(self, service):
        # runtime_attrs.content.instances.extended_gencfg_groups.
        properties_chain = ['runtime_attrs', 'content', 'resources', 'template_set_files']

        return self.get_service_value(service, properties_chain, [])

    def get_extended_gencfg_group_name(self, service):
        extended_gencfg_groups = self.get_extended_gencfg_groups(service)

        return extended_gencfg_groups[0]['name'] if extended_gencfg_groups else None

    def get_used_gencfg_group_names(self):
        used_groups = []
        services = self.get_exist_services(self.Parameters.category)

        for service in services:
            group_name = self.get_extended_gencfg_group_name(service)

            if group_name:
                used_groups.append(group_name)

        return used_groups

    def get_gencfg_group_names(self):
        with open(os.path.join(self.curr_dir, 'gencfg_params.json')) as f:
            config = json.load(f)

            return config.get('groups', [])

    def list_not_contains(self, lst1, lst2):
        return [value for value in lst1 if value not in lst2]

    def get_gencfg_group(self):
        all_groups_names = self.get_gencfg_group_names()
        used_groups = self.get_used_gencfg_group_names()

        logging.debug('pre defined gencfg groups: {}'.format(all_groups_names))
        logging.debug('used gencfg groups: {}'.format(used_groups))

        not_used_group_names = self.list_not_contains(all_groups_names, used_groups)

        if not_used_group_names:
            group_name = not_used_group_names[0]
            logging.debug('choose gencfg group for deploy {}'.format(group_name))

            return not_used_group_names[0]

    def get_engine(self, service):
        # runtime_attrs.content.instances.engines.
        properties_chain = ['runtime_attrs', 'content', 'engines']

        return self.get_service_value(service, properties_chain, {})

    def get_extended_gencfg_groups_main(self, instances):
        return self.deep_get(instances, ['content', 'extended_gencfg_groups'], {})

    def get_gencfg_host_info_by_group_name(self, group_name):
        host = {}

        logging.debug('trying to get info for gencfg group :{}'.format(group_name))

        # запрос для получения ифнормации о хосте
        host_info = self.gencfg_client.fetch_host(group_name)
        instances = host_info['instances'][0] if host_info['instances'] else None

        if not instances:
            raise Exception('Can not find host for group {}'.format(group_name))

        host['name'] = self.deep_get(instances, ['hbf', 'interfaces', 'backbone', 'hostname'], None)

        # запрос для получения инфомации порта для хоста (за один запрос получить нельзя)
        group_info = self.gencfg_client.fetch_group(group_name)
        host['port'] = self.deep_get(group_info, ['reqs', 'instances', 'port'], None)

        return host

    def format_server_config_for_pr(self, service, nginx_server_config):
        formatted_config = None
        service_id = service['_id']

        logging.debug('trying to format nginx servers config for service :{}'.format(service_id))

        group_name = self.get_extended_gencfg_group_name(service)

        if group_name:
            host = self.get_gencfg_host_info_by_group_name(group_name)
            pr_number = service_id.replace(self.Parameters.service_prefix, '')
            endpoint = 'http://{host_name}:{host_port}'.format(
                host_name=host['name'],
                host_port=host['port'],
            )
            hostname = self.Parameters.stand_url_template.format(pr_number)

            formatted_config = nginx_server_config.replace('{{endpoint}}', endpoint)
            formatted_config = formatted_config.replace('{{hostname}}', hostname)

            logging.debug('nginx server config for pr number {pr_number} is : {formatted_config}'.format(
                pr_number=pr_number,
                formatted_config=formatted_config,
            ))

        return formatted_config

    def format_servers_config(self, service_for_prs):
        server_configs = []

        with open(os.path.join(self.curr_dir, 'nginx.beta.conf')) as nginx_beta_config:
            nginx_server_config = nginx_beta_config.read()

        for service in service_for_prs:
            server_config = self.format_server_config_for_pr(service, nginx_server_config)

            if server_config:
                server_configs.append(server_config)

        return server_configs

    def get_nginx_conf(self):
        new_nginx_conf = None

        # получаем список всех активных сервисов
        services = self.get_exist_services()

        service_for_prs = []

        for service in services:
            if service['_id'] != self.Parameters.default_balancer_service:
                service_for_prs.append(service)

        if service_for_prs:
            server_configs = self.format_servers_config(service_for_prs)

            # формируем новый конфиг
            if server_configs:
                server_formatted_config = '\n'.join(server_configs)

                with open(os.path.join(self.curr_dir, 'nginx.conf')) as nginx_full_config:
                    new_nginx_conf = nginx_full_config.read().replace('{{config}}', server_formatted_config)

        return new_nginx_conf

    def update_balancer(self):
        should_patch = False
        # получаем содержимое nginx.conf для всех активных сервисов не "offline"
        new_nginx_conf = self.get_nginx_conf()

        if not new_nginx_conf:
            raise Exception('new nginx.conf is empty.')

        # получаем информацию для текущего nginx.conf у сервиса балансера
        balancer_service = self.get_service(self.Parameters.default_balancer_service)
        template_set_files = self.get_template_set_files(balancer_service)

        # проверяем что настройки nginx.conf для всех активных пулл-реквестов не изменились
        if template_set_files:
            for idx, set_file in enumerate(template_set_files):
                if set_file['local_path'] == 'nginx.conf' and set_file['layout'] != new_nginx_conf:
                    should_patch = True
                    template_set_files[idx]['layout'] = new_nginx_conf

        # если изменились, обновляем nginx.conf и активируем снэпшот
        if should_patch:
            comment = 'Update by Sandbox task {type} with id: {id}'.format(type=self.type, id=self.id)
            balancer_service['runtime_attrs']['content']['resources']['template_set_files'] = template_set_files

            snapshot_info = self.update_service_resources(self.Parameters.default_balancer_service, {
                'content':  balancer_service['runtime_attrs']['content']['resources'],
                'comment': comment,
                'snapshot_id': balancer_service['snapshot_id'],
            })

            logging.info('nanny service {} was updated successfully', self.Parameters.default_balancer_service)
            logging.debug('snapshot_info: {}', snapshot_info)

            logging.info('Setting new snapshot state to ACTIVE...')

            new_snapshot_id = snapshot_info['runtime_attrs']['_id']
            content = {
                'state': 'ACTIVE',
                'recipe': 'common',
                'comment': comment,
                'snapshot_id': new_snapshot_id
            }

            self.create_event(
                service_id=self.Parameters.default_balancer_service,
                type='SET_SNAPSHOT_STATE',
                content=content,
            )

    def create_new_service(self, service_id):
        # получаем незанятую gencfg группу (как правило, это одна машинка, на которую будут деплоится шаблоны)
        group = self.get_gencfg_group()
        # получаем последний trunk tag, для получения последних прав и настроек для gencfg группы
        tag = self.gencfg_client.get_first_tag()

        if not group or not tag:
            raise Exception('Can not create new nanny configuration.')

        # копируем сервис dev-а по умолчанию, так как там есть уже все нужные настройки и ресурсы
        service = self.copy_service(
            src_service_id=self.Parameters.default_service,
            new_service_id=service_id,
            category=self.Parameters.category,
            description=self.description,
            copy_auth_attrs=True
        )
        release = 'tags/{}'.format(tag)

        instances = self.get_service_instances(service_id)

        logging.debug('instances payload for {service_id} is {instances}'.format(
            service_id=service_id,
            instances=instances,
        ))

        # выбранный инстанс gen_cfg (выделенная машинка)
        instances['content']['extended_gencfg_groups']['groups'] = [{'release': release, 'name': group, }]
        # обязательные настройки для конфигурации (спросить у @ftdebugger)
        instances['content']['extended_gencfg_groups']['gencfg_volumes_settings'] = {'use_volumes': True, }
        instances['content']['extended_gencfg_groups']['network_settings'] = {'use_mtn': True, }

        data = {
            'content': instances['content'],
            'comment': 'Set new settings (prepare configuration)',
            'snapshot_id': instances['snapshot_id'],
        }

        logging.debug('trying to set gencfg group {group} for configuration {service_id}'.format(
            group=group,
            service_id=service_id,
        ))

        # обновляем выделенную gencfg группу для сервиса, так как скопировали с настройками dev-а
        self.update_service_instances(service_id, data)

        # настраиваем ISS engine, если в настройках стоит ISS (legacy), по умолчанию ISS_MULTI для всех локаций
        # возможно можно удалить, перенс "as is"
        engine = self.get_engine(service)
        engine_type = engine.get('engine_type', 'ISS')

        if engine_type == 'ISS':
            dc = re.match(r'^(MSK|MAN|VLA|SAS)', group)

            if dc:
                engine['engine_type'] = 'ISS_{}'.format(dc[0])
                self.update_service_engine(service_id, engine)

        # обновляем балансер (nginx.conf), где будет добавлен новый пр
        # доступный по адресу pull-{pr_number}.collections.test.yandex.ru
        self.update_balancer()

        return service

    def init_service(self, service_id):
        # проверяем, что сервис уже существует
        service = self.get_service(service_id)

        if not self.get_service(service_id):
            service = self.create_new_service(service_id)

        return service

    def get_files(self, service):
        # runtime_attrs.content.resources.sandbox_files.
        properties_chain = ['runtime_attrs', 'content', 'resources', 'sandbox_files']

        return self.get_service_value(service, properties_chain, [])

    def get_task(self):
        return sdk2.Task.find(id=self.Parameters.task_id, children=True).first()

    def deploy_resource(self, service_id):
        # получаем информаци о таске по id
        task = self.get_task()
        comment = 'Activate by Sandbox task {type} with id: {id}'.format(type=self.type, id=self.id)
        recipe = 'common'

        self.update_service_sandbox_file(
            service_id=service_id,
            task_type=str(task.type),
            task_id=str(task.id),
            comment=comment,
            skip_not_existing_resources=True,
            recipe=recipe,
        )

        # получаем свежую информацию о ресурсах сервиса перед деплоем (активации)
        snapshot_id = self.get_snapshot_id(service_id)

        content = {
            'is_enabled': True,
            'snapshot_id': snapshot_id,
            'comment': comment,
            'recipe': recipe,
        }

        # активируем снэпшот с ресурсом из переданной таски
        self.create_event(service_id, 'SET_TARGET_STATE', content)
        # ждем активации
        self.__wait_services_for_status([service_id], status='ACTIVE')

        return

    def on_execute(self):
        # создаем новую конфигурацию self.service_id либо ничего не делаем (нужно только для пулл-реквестов)
        self.init_service(self.service_id)
        # обновляем Sandbox ресурс для текущей конфигурации и активируем снэпшот
        self.deploy_resource(self.service_id)
