# coding: utf-8
import requests
import logging
import json
import traceback
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk import environments


def iterate_column(sheet_values, start_end, column):
    for i in xrange(*start_end):
        yield i, sheet_values[i][column]


def format_names(names):
    return ', '.join(name + '@' for name in names)


class Calendar(object):
    # Copypaste from https://a.yandex-team.ru/arc/trunk/arcadia/search/tools/devops/libs/calendar_api.py
    MODES = {  # todo: add development and prestable modes
        "production": "http://calendar-api.tools.yandex.net/internal/",
        "testing": "http://calendar-api.calcorp-test-back.cmail.yandex.net/internal/",
    }
    DEFAULT_HEADERS = {'Content-Type': 'text/json'}

    def __init__(self, mode="testing", uid="1120000000046069"):
        self._mode = mode
        self._base_url = self.MODES[mode]
        self._uid = uid
        self._session = requests.Session()
        requests.packages.urllib3.disable_warnings()

    def _post(self, url, data, headers=None):
        r = self._session.post(url, json=data, headers=headers or self.DEFAULT_HEADERS)
        try:
            r.raise_for_status()
        except Exception:
            logging.error('POST request %s to %s failed', data, url)
            logging.debug("Traceback:\n%s", traceback.format_exc())
        return r.json()

    @staticmethod
    def _process_response_status(response):
        is_ok = response.get('status', '') == 'ok'
        if not is_ok:
            logging.error("Fail:\n%s", json.dumps(response.get("error", {}), indent=2))
        return is_ok, response

    def create_event(self, request_data):
        path = '{}create-event?uid={}&tz=Europe/Moscow'.format(self._base_url, self._uid)
        data = {
            'type': request_data.get('type', 'user'),
            'startTs': request_data['startTs'],
            'endTs': request_data['endTs'],
            'name': request_data.get('name', 'No name'),
            'location': request_data.get('location', 'Everywhere'),
            'description': request_data.get('description', ''),
            'isAllDay': request_data.get('isAllDay', False),
            'availability': request_data.get('availability', 'available'),
            'attendees': request_data.get('attendees', []),
            'organizer': request_data['organizer'],
            'participantsCanInvite': request_data.get('participantsCanInvite', False),
            'participantsCanEdit': request_data.get('participantsCanEdit', False),
            'othersCanView': request_data.get('otherCanView', True)
        }
        if 'repetition' in request_data:
            data['repetition'] = request_data['repetition']
        if 'layerId' in request_data:
            data['layerId'] = request_data['layerId']

        response = self._post(path, data)
        return self._process_response_status(response)


class CollectionsDutyCalendar(sdk2.Task):
    """
        Collections duty task
    """
    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('gspread', '2.0.0'),
            environments.PipEnvironment('oauth2client', '4.1.2'),
        )

    class Parameters(sdk2.Parameters):
        environment = sdk2.parameters.String(
            'testing|production',
            required=True,
            default='testing',
        )

        week = sdk2.parameters.String(
            'week date DD.MM.YYYY',
            required=False
        )

        duty_switch_time = sdk2.parameters.String(
            'Duty switch time HH:MM',
            required=False
        )

        standup_time = sdk2.parameters.String(
            'Standup meeting time HH:MM',
            required=False
        )

        standup_duration = sdk2.parameters.Integer(
            'Standup meeting duration (mins)',
            required=False,
            default=15
        )

        constant_standup_members = sdk2.parameters.List(
            'Constant standup members',
            sdk2.parameters.String
        )

        debug = sdk2.parameters.Bool(
            'Do not create events',
            required=False,
            default=False
        )

    def _get_event_data(self, start, end, duty_first, duty_second):
        return {
            'startTs': start.isoformat(),
            'endTs': end.isoformat(),
            'organizer': 'robot-collections@yandex-team.ru',
            'isAllDay': False,
            'participantsCanEdit': True,
            'attendees': [name + '@yandex-team.ru' for name in duty_first] + [name + '@yandex-team.ru' for name in duty_second],
            'name': "Дежурство Коллекций 1st – {}, 2nd – {}".format(format_names(duty_first), format_names(duty_second))
        }

    def _get_standup_data(self, start, end, duty_first, duty_second):
        return {
            'startTs': start.isoformat(),
            'endTs': end.isoformat(),
            'organizer': 'robot-collections@yandex-team.ru',
            'isAllDay': False,
            'participantsCanEdit': True,
            'attendees': [name + '@yandex-team.ru' for name in self.Parameters.constant_standup_members] + [name + '@yandex-team.ru' for name in duty_first],
            'name': "Дежурный стендап {}".format(format_names(duty_first))
        }

    def _process_event(self, start, end, duty_first, duty_second, get_data):
        if self.Parameters.debug:
            message = 'Create event parameters:\n{}'.format(json.dumps(get_data(start, end, duty_first, duty_second), indent=4))
            self.set_info(message)
        else:
            cal = Calendar(self.Parameters.environment, 1120000000074789)
            is_ok, _ = cal.create_event(get_data(start, end, duty_first, duty_second))
            if not is_ok:
                raise Exception()

    def _get_data(self, week):
        week = week.strftime("%d.%m.%Y")
        import gspread
        from oauth2client.service_account import ServiceAccountCredentials
        scope = ['https://spreadsheets.google.com/feeds']
        secret = sdk2.Vault.data('YASAP', 'SPREADSHEET_TOKEN')
        creds = ServiceAccountCredentials.from_json_keyfile_dict(json.loads(secret), scope)
        client = gspread.authorize(creds)
        document = client.open_by_url("https://docs.google.com/spreadsheets/d/1THzXcw-JuRpv1iQ3C6WCWphxW5hV3hwtCgFe-iWyMtY/edit#gid=859499061")
        sheet = document.sheet1
        sheet_values = sheet.get_all_values()
        start_week_index = None
        first = []
        second = []
        for i, row in enumerate(sheet_values):
            code = row[0]
            if code == 'START_WEEK':
                start_week_index = i
            elif code == 'FIRST':
                first.append(i + 1)
            elif code == 'SECOND':
                first.append(i)
                second.append(i + 1)
            elif code == 'END':
                second.append(i)
        assert start_week_index is not None
        assert len(first) == 2
        assert len(second) == 2
        first_names = {
            i: name.split('@')
            for i, name in iterate_column(sheet_values, first, 2)
            if name
        }
        second_names = {
            i: name.split('@')
            for i, name in iterate_column(sheet_values, second, 2)
            if name
        }
        column = None
        for i, start in enumerate(sheet_values[start_week_index]):
            if start == week:
                column = i
                break
        assert column is not None
        duty_first = [
            first_names[i][1]
            for i, value in iterate_column(sheet_values, first, column)
            if value == 'x'
        ]

        duty_second = [
            second_names[i][1]
            for i, value in iterate_column(sheet_values, second, column)
            if value == 'x'
        ]
        return duty_first, duty_second

    def _week_dates(self):
        if self.Parameters.week:
            td = datetime.strptime(self.Parameters.week, "%d.%m.%Y")
            next_week_start = (td - timedelta(days=td.weekday()))
        else:
            td = datetime.today().replace(second=0, microsecond=0, hour=0, minute=0)
            next_week_start = (td - timedelta(days=td.weekday()) + timedelta(days=7))

        if self.Parameters.duty_switch_time:
            t = datetime.strptime(self.Parameters.duty_switch_time, "%H:%M")
            next_week_start = next_week_start.replace(hour=t.hour, minute=t.minute)
            next_week_end = next_week_start + timedelta(days=7)
        else:
            next_week_end = next_week_start + timedelta(days=8)
        return next_week_start, next_week_end

    def on_execute(self):
        next_week_start, next_week_end = self._week_dates()
        first, second = self._get_data(next_week_start)
        self._process_event(next_week_start, next_week_end, first, second, self._get_event_data)
        if self.Parameters.standup_time:
            s = datetime.strptime(self.Parameters.standup_time, "%H:%M")
            for i in xrange(1, 5):
                standup_start = next_week_start.replace(hour=s.hour, minute=s.minute) + timedelta(days=i)
                self._process_event(
                    standup_start,
                    standup_start + timedelta(minutes=self.Parameters.standup_duration),
                    first,
                    second,
                    self._get_standup_data
                )
