# -*- coding: utf-8 -*-

import re
import logging
import os
import urllib
import requests
import json
from shutil import copytree, copyfile, move

from sandbox import sdk2
from sandbox.common.types import task
from sandbox.projects.collections.CollectionsDeployNannyService import CollectionsDeployNannyService
from sandbox.projects.collections.CollectionsFrontendKarma import CollectionsFrontendKarma
from sandbox.projects.collections.CollectionsFrontendGemini import CollectionsFrontendGemini
from sandbox.projects.collections.CollectionsFrontendE2E import CollectionsFrontendE2E
from sandbox.projects.collections.CollectionsFrontendMocha import CollectionsFrontendMocha
from sandbox.projects.collections.CollectionsFrontendBase import resources
from sandbox.projects.collections.CollectionsFrontendEslint import CollectionsFrontendEslint
from sandbox.projects.collections.CollectionsFrontendBase.base import CollectionsFrontendBase
from sandbox.projects.collections.CollectionsFrontend.detector import detect_collisions
from sandbox.projects.collections.CollectionsFrontendS3 import CollectionsFrontendS3
from sandbox.projects.collections.CollectionsFrontendTslint import CollectionsFrontendTslint
from sandbox.projects.collections.CollectionsFrontendStylelint import CollectionsFrontendStylelint
from sandbox.projects.collections.CollectionsFrontendTypescript import CollectionsFrontendTypescript
from sandbox.projects.collections.CollectionsFrontendJest import CollectionsFrontendJest
from sandbox.projects.collections.CollectionsFrontendPalmsync import CollectionsFrontendPalmsync
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.nanny.client import NannyApiException
from sandbox.projects.common.gnupg import GpgKey
from sandbox.projects.common.debpkg import DebRelease
from sandbox.sandboxsdk.ssh import Key
from sandbox.sandboxsdk.process import run_process, CustomOsEnviron
from sandbox.common.errors import TaskError

DEB_REPO = 'verstka'

DUPLOAD_CONF = {
    'verstka': {
        'fqdn': 'verstka.dupload.dist.yandex.ru',
        'method': 'scpb',
        'login': 'robot-pdb-builder',
        'incoming': '/repo/verstka/mini-dinstall/incoming/',
        'dinstall_runs': 0,
    }
}

YASTATIC_COLLECTIONS_URL = {
    'production': 'https://yastatic.net/collections/_/',
    'testing': 'https://betastatic.yastatic.net/collections/_/',
}


class CollectionsFrontend(nanny.ReleaseToNannyTask2, CollectionsFrontendBase):
    """
        Collections frontend
    """
    class Parameters(CollectionsFrontendBase.Parameters):
        with CollectionsFrontendBase.Parameters.build() as build:
            production = sdk2.parameters.Bool('Production', default=True)
            testing = sdk2.parameters.Bool('Testing', default=True)
            storybook = sdk2.parameters.Bool('Storybook', default=False)
            yastatic = sdk2.parameters.Bool('yastatic.net', default=False)
            betastatic = sdk2.parameters.Bool('betastatic.yastatic.net', default=False)
            yastatic_s3 = sdk2.parameters.Bool('yastatic s3', default=False)
            build_env = sdk2.parameters.Dict('Build environment variables', default={})

        with sdk2.parameters.Group('Features') as features:
            extensions = sdk2.parameters.Bool('Build extension', default=True)
            size = sdk2.parameters.Bool('Check bundle size', default=False)
            eslint = sdk2.parameters.Bool('Run eslint', default=False)
            tslint = sdk2.parameters.Bool('Run tslint', default=False)
            stylelint = sdk2.parameters.Bool('Run stylelint', default=False)
            mocha = sdk2.parameters.Bool('Mocha', default=False)
            karma = sdk2.parameters.Bool('Karma', default=False)
            gemini = sdk2.parameters.Bool('Gemini', default=False)
            e2e = sdk2.parameters.Bool('E2E Components', default=False)
            e2e_features = sdk2.parameters.Bool('E2E Features', default=False)
            tsc = sdk2.parameters.Bool('Check TS types', default=False)
            jest = sdk2.parameters.Bool('Jest', default=False)
            palmsync = sdk2.parameters.Bool('Palmsync', default=False)

        with CollectionsFrontendBase.Parameters.github() as github:
            git_lfs = sdk2.parameters.Bool('Git Lfs', default=True)

        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Build events') as build_events:
                is_subtasks_ready = sdk2.parameters.Bool(
                    'Subtasks ready',
                    description='Parameter will be True, if task created all its subtask, else it will be False.',
                )

    def on_execute(self):
        self.build()
        self.install_modules(production=True)
        self.create_subtasks()

        environment = self.environment

        if self.Parameters.production:
            self._build(
                environment=environment,
                project_env='production',
                resource=self.production_resource,
                yastatic=bool(self.Parameters.yastatic),
            )

            self._run_static_upload_script(
                project_env='production',
            )


            if self.is_script_exist('ci:deploy:release'):
                run_process(
                    [self.node_bin, self.npm_bin, 'run', 'ci:deploy:release'],
                    work_dir=self.src_root,
                    log_prefix='deploy',
                    environment=environment
                )

        if self.Parameters.testing:
            pr_number = self.pr_number

            if pr_number:
                environment['PR'] = pr_number

            self._build(
                environment=environment,
                project_env='testing',
                build='release:testing',
                resource=self.testing_resource,
                yastatic=bool(self.Parameters.yastatic and self.Parameters.betastatic),
            )

            self._run_static_upload_script(
                project_env='testing',
            )

            if self.is_script_exist('ci:deploy'):
                run_process(
                    [self.node_bin, self.npm_bin, 'run', 'ci:deploy'],
                    work_dir=self.src_root,
                    log_prefix='deploy',
                    environment=environment
                )

            if self.Parameters.extensions:
                self._build_chrome()

        if self.Parameters.storybook:
            self._build(
                build='release:gemini',
                check_size=False,
                environment=environment,
                project_env='gemini',
                resource=self.testing_resource,
            )

    def build(self):
        self.checkout()
        self.prepare_node()

    def create_subtasks(self):
        if self.Parameters.eslint:
            self._build_eslint()

        if self.Parameters.tslint:
            self._build_tslint()

        if self.Parameters.stylelint:
            self._build_stylelint()

        if self.Parameters.mocha:
            self._build_mocha()

        if self.Parameters.karma:
            self._build_karma()

        if self.Parameters.gemini:
            self._build_gemini()

        if self.Parameters.e2e:
            self._build_e2e_components()

        if self.Parameters.e2e_features:
            self._build_e2e_features()

        if self.Parameters.tsc:
            self._build_tsc()

        if self.Parameters.jest:
            self._build_jest()

        if self.Parameters.palmsync:
            self._build_palmsync()

        self.Parameters.is_subtasks_ready = True

    def on_release(self, additional_parameters):
        try:
            sdk2.Task.on_release(self, additional_parameters)
            nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)

            release_status = additional_parameters['release_status']

            if self.Parameters.yastatic_s3:
                self._release_yastatic_s3(release_status)
            elif self.Parameters.yastatic:
                self._release_yastatic(release_status)

        except NannyApiException:
            logging.info('Error during release')

    def on_before_end(self, status):
        if self.Parameters.is_subtasks_ready is None:
            self.Parameters.is_subtasks_ready = False

    def on_finish(self, prev_status, status):
        self.on_before_end(status)

        super(CollectionsFrontend, self).on_finish(prev_status, status)

    def on_break(self, prev_status, status):
        self.on_before_end(status)

        super(CollectionsFrontend, self).on_break(prev_status, status)

    @property
    def production_resource(self):
        return resources.CollectionsFrontendResourcePackage

    @property
    def testing_resource(self):
        return resources.CollectionsFrontendResourceTestPackage

    @property
    def size_resource(self):
        return resources.CollectionsFrontendResourceSize

    def _build(self, project_env, environment, resource, build='release', yastatic=False, check_size=True):
        yastatic_s3 = self.Parameters.yastatic_s3

        env = self.Parameters.build_env or {}
        env.update(environment)
        env['YENV'] = project_env

        if yastatic_s3:
            env['YASTATIC_S3'] = 'yes'
        elif yastatic:
            env['YASTATIC'] = 'yes'

        if self.Parameters.use_ram_drive:
            files = [self.node_root, self.src_root]
        else:
            files = [
                os.path.basename(self.node_root),
                os.path.basename(self.src_root)
            ]

        if self.Parameters.suspend:
            self.suspend()

        run_process(
            [self.node_bin, self.npm_bin, 'run', build],
            log_prefix='release',
            work_dir=str(self.src_root),
            environment=env
        )

        frontend_package = sdk2.ResourceData(resource(
            self,
            '[{}] Frontend package'.format(project_env),
            'frontend-{}.tar.gz'.format(project_env)
        ))

        self._create_tar(files, str(frontend_package.path))

        frontend_package.ready()

        if yastatic_s3:
            self._build_yastatic_s3(
                project_env=project_env
            )
        elif yastatic:
            # Build deb package
            self._build_yastatic(
                project_env=project_env
            )

        # Calculate size
        if check_size:
            self._calc_size(
                project_env=project_env,
                environment=env
            )

        # Save trace as resource
        self._save_trace(
            project_env=project_env
        )

    def _build_deploy(self):
        self.create_task(CollectionsDeployNannyService).enqueue()

    def _build_eslint(self):
        self.create_task(CollectionsFrontendEslint).enqueue()

    def _build_tslint(self):
        self.create_task(CollectionsFrontendTslint).enqueue()

    def _build_stylelint(self):
        self.create_task(CollectionsFrontendStylelint).enqueue()

    def _build_mocha(self):
        self.create_task(CollectionsFrontendMocha, platform='mobile').enqueue()
        self.create_task(CollectionsFrontendMocha, platform='desktop').enqueue()

    def _build_karma(self):
        self.create_task(CollectionsFrontendKarma, platform='mobile').enqueue()
        self.create_task(CollectionsFrontendKarma, platform='desktop').enqueue()

    def _build_gemini(self):
        self.create_task(CollectionsFrontendGemini, platform='mobile').enqueue()
        self.create_task(CollectionsFrontendGemini, platform='desktop').enqueue()

    def _build_e2e_components(self):
        # component e2e tests
        self.create_task(
            CollectionsFrontendE2E,
            preset='mobile',
            semaphore_name='collections_e2e'
        ).enqueue()

        self.create_task(
            CollectionsFrontendE2E,
            preset='desktop',
            semaphore_name='collections_e2e'
        ).enqueue()

    def _build_e2e_features(self):
        # features e2e tests
        self.create_task(
            CollectionsFrontendE2E,
            preset='features:desktop',
            semaphore_name='collections_features_e2e'
        ).enqueue()

        self.create_task(
            CollectionsFrontendE2E,
            preset='features:mobile',
            semaphore_name='collections_features_e2e'
        ).enqueue()

        self.create_task(
            CollectionsFrontendE2E,
            preset='features:app',
            semaphore_name='collections_features_e2e'
        ).enqueue()

    def _build_tsc(self):
        self.create_task(CollectionsFrontendTypescript).enqueue()

    def _build_jest(self):
        self.create_task(CollectionsFrontendJest, platform='mobile').enqueue()
        self.create_task(CollectionsFrontendJest, platform='desktop').enqueue()

    def _build_palmsync(self):
        self.create_task(CollectionsFrontendPalmsync, app_ver=self.Context.app_ver).enqueue()

    def _build_chrome(self):
        """
            Build chrome extension
        """
        if self.Parameters.use_ram_drive:
            dist = str(self.working_path(self.project_dir, 'dist'))
        else:
            dist = os.path.join(self.src_root, 'dist')

        zip_file = ''

        for filename in os.listdir(dist):
            root, ext = os.path.splitext(filename)

            if root.startswith('chrome') and ext == '.zip':
                zip_file = filename

        if zip_file:
            if self.Parameters.use_ram_drive:
                zip_path = str(self.working_path(self.project_dir, 'dist', zip_file))
            else:
                zip_path = os.path.join(dist, zip_file)

            chrome = sdk2.ResourceData(resources.CollectionsFrontendResourceChrome(
                self, 'Chrome extension', zip_file
            ))

            copyfile(zip_path, str(chrome.path))

            chrome.ready()

    def _calc_size(self, environment, project_env):
        if self.Parameters.size:
            run_process(
                [self.node_bin, self.gulp_bin, 'size:calc', '--' + project_env],
                log_prefix='gulp_size',
                work_dir=str(self.src_root),
                environment=environment
            )

            target = self.pr_number

            if not target:
                target = 'master'

            size = sdk2.ResourceData(self.size_resource(
                self,
                '[{}] Frontend size'.format(project_env),
                'size-{}.json'.format(project_env),
                env=project_env,
                target=target
            ))

            if self.Parameters.use_ram_drive:
                source_path = str(self.working_path(self.project_dir, 'dist/web/size.json'))
                report_path = str(self.working_path(self.project_dir, 'dist/web/size.html'))
            else:
                source_path = os.path.join(self.src_root, 'dist/web/size.json')
                report_path = os.path.join(self.src_root, 'dist/web/size.html')

            copyfile(
                str(source_path),
                str(size.path)
            )

            size.ready()

            report = open(report_path, 'r')
            self.Context.footer += report.read()

    def _save_trace(self, project_env):
        target = self.pr_number

        if self.Parameters.use_ram_drive:
            path = str(self.working_path(self.project_dir, 'dist', 'trace'))
        else:
            path = os.path.join(self.src_root, 'dist', 'trace')

        if not os.path.isdir(path):
            return

        if not target:
            target = 'master'

        trace = sdk2.ResourceData(resources.CollectionsFrontendResourceTrace(
            self,
            '[{}] Frontend trace'.format(project_env),
            'trace-{}'.format(project_env),
            env=project_env,
            target=target
        ))

        move(
            path,
            str(trace.path)
        )

        trace.ready()

    def get_vault_data(self, vault_item_owner, vault_item_name):
        # Added for compatibility with sandbox.projects.common.gnupg, sandboxsdk.ssh
        return sdk2.Vault.data(vault_item_owner, vault_item_name)

    @property
    def yastatic_environment(self):
        env = {}

        user = DUPLOAD_CONF[DEB_REPO]['login']
        user_email = '{}@yandex-team.ru'.format(user)

        env.update(self.environment)
        env.update({
            'DEBFULLNAME': user,
            'DEBEMAIL': user_email,
            'EMAIL': user_email,
        })

        return env

    def yastatic_meta(self, project_env, prj='pdb'):
        base_dir = os.path.join('~/', project_env)

        version = self.Context.app_ver

        if project_env != 'production':
            version = version + '-' + project_env

        return {
            'package_name': 'yandex-static-pdb',
            'current_version': version,
            'build_dir': os.path.join(base_dir, 'build/collections'),

            'src_root_debian_dir': os.path.join(self.src_root, 'debian'),
            'src_root_static_dir': os.path.join(self.src_root, 'dist/web/client/collections'),

            'yastatic_url': YASTATIC_COLLECTIONS_URL[project_env]
        }

    def _is_deb_package_exist(self, package_name, version):
        # example: http://dist.yandex.ru/find?pkg=yandex-static-pdb&ver=1.59.1&repo=verstka&strict=True
        response = requests.get(
            'http://dist.yandex.ru/find?pkg={package_name}&ver={version}&repo={repo}&strict=True'.format(
                package_name=package_name,
                version=version,
                repo=DEB_REPO
            )
        )

        return response.status_code == 200

    def _get_deb_package_url(self, package_name, version):
        # example: http://dist.yandex.ru/find?pkg=yandex-static-pdb&ver=1.66.0&repo=verstka&strict=True&withurls=True
        response = requests.get(
            'http://dist.yandex.ru/find?pkg={package_name}&ver={version}&repo={repo}&strict=True&withurls=True'.format(
                package_name=package_name,
                version=version,
                repo=DEB_REPO
            )
        )

        if response.status_code == 200:
            links = re.findall(r'http://dist\.yandex\.ru/[^"]+\.deb', response.text)
            return links[0] if len(links) else ''

        return ''

    def _create_conductor_ticket(self, package_name, version, branch='testing'):
        CONDUCTOR_URL = 'http://c.yandex-team.ru'

        comment = 'Automatic release from sandbox https://sandbox.yandex-team.ru/task/{}'.format(self.id)
        conductor_oauth_token = sdk2.Vault.data('YASAP', 'yastatic_conductor_oauth_token')

        query_args = [
            ('ticket[branch]', branch),
            ('ticket[comment]', comment),
            ('package[0]', package_name),
            ('version[0]', version),
        ]

        query_url = '{}/auth_update/ticket_add?{}'.format(CONDUCTOR_URL, urllib.urlencode(query_args, True))
        headers = {'Authorization': 'OAuth {}'.format(conductor_oauth_token)}

        requests.get(query_url, headers=headers).raise_for_status()

    def _yastatic_detect_collisions(self, build_pdb_dir, dst_url):
        if not os.path.isdir(build_pdb_dir):
            logging.warn('yastatic build dir not exists: %s', build_pdb_dir)
            return

        if self.is_static_upload_script_exist:
            logging.warn('is_static_upload_script_exist exists')
            return

        stats = detect_collisions(
            build_pdb_dir,
            dst_url,
            exclude_list=[
                # favicons
                'android-chrome-192x192.png',
                'apple-touch-icon-114x114.png',
                'apple-touch-icon-120x120.png',
                'apple-touch-icon-144x144.png',
                'apple-touch-icon-152x152.png',
                'apple-touch-icon-180x180.png',
                'apple-touch-icon-57x57.png',
                'apple-touch-icon-60x60.png',
                'apple-touch-icon-72x72.png',
                'apple-touch-icon-76x76.png',
                'favicon-160x160.png',
                'favicon-16x16.png',
                'favicon-194x194.png',
                'favicon-32x32.png',
                'favicon-96x96.png',
                'favicon-images.ico',
                'favicon.ico',
                'icon-150.png',
                'icon-20.png',
                'icon-25.png',
                'icon-30.png',
                'icon-40.png',
                'icon-44.png',
                'icon-50.png',
                'icon128-dark.png',
                'icon128.png',
                'icon16-dark.png',
                'icon16.png',
                'icon32-dark.png',
                'icon32.png',
                'icon48-dark.png',
                'icon48.png',
                'mstile-144x144.png',
                'favicon-16x16.svg',
                'safari-pinned-tab.svg',
                'safari-plugin.png',
                'collections_searchapp.png',
                # tap icons,
                'tap-icon-28.png',
                'tap-icon-48.png',
                'tap-icon-144.png',
                'tap-icon-512.png',
                'tap-icon-576.png',
                # other
                'manifest.json',
                'sw-mobile.js',
                'sw-desktop.js',
                'sw-mobile-testing.js',
                'sw-desktop-testing.js',
                'collections.safariextz',
                'collections-yabro.crx',
                'updates.xml'
            ],
            exclude_re=re.compile('^.*\.gz$', re.IGNORECASE),
        )

        stats_created = stats['created']
        logging.info('yastatic create: %s', len(stats_created))
        for src in stats_created:
            logging.info('[create] ' + src)

        stats_updated = stats['updated']
        logging.info('yastatic update: %s', len(stats_updated))
        for src in stats_updated:
            logging.info('[update] ' + src)

        stats_collisions = stats['collisions']
        if len(stats_collisions):
            messages_list = []
            logging.error('yastatic collisions: %s', len(stats_collisions))

            for info in stats_collisions:
                (src, src_md5, dst, dst_md5) = info
                message = '[md5:{src_md5}] {src} -> [md5:{dst_md5}] {dst}'.format(
                    src_md5=src_md5,
                    src=src,
                    dst_md5=dst_md5,
                    dst=dst
                )
                messages_list.append(message)
                logging.error(message)

            raise TaskError('yastatic collisions detected:\n\n' + '\n'.join(messages_list))

        http_errors = stats['http_errors']
        if len(http_errors):
            logging.error('yastatic http errors: %s', len(http_errors))

            for info in http_errors:
                (status_code, url) = info
                logging.error('[code:{code}] {url}'.format(
                    code=status_code,
                    url=url,
                ))

            raise TaskError('yastatic http errors detected')

        req_errors = stats['errors']
        if len(req_errors):
            logging.error('yastatic errors: %s', len(req_errors))

            for info in req_errors:
                (url, message) = info
                logging.error('[{url}] {message}'.format(
                    url=url,
                    message=message,
                ))

            raise TaskError('yastatic errors detected: ' + str(len(req_errors)))

    def _get_yastatic_s3_meta(self, project_env):
        s3_bucket_name = 'fiji-static'
        key_prefix = 'podb'
        prj = 'collections'

        if project_env != 'production':
            s3_bucket_name = 'serp-static-testing'

        return s3_bucket_name, key_prefix, prj

    def _build_yastatic_s3(self, project_env):
        s3_bucket_name, key_prefix, prj = self._get_yastatic_s3_meta(project_env)

        yastatic_url = 'https://yastatic.net/s3/' + s3_bucket_name + '/' + key_prefix + '/'
        src_path = 'dist/web/client/' + prj
        static_files_src = os.path.join(self.src_root, src_path)

        if project_env != 'production':
            yastatic_url = 'https://serp-static-testing.s3.yandex.net/' + key_prefix + '/'

        yastatic_artifact_id = self._build_yastatuc_s3_resource(
            project_env=project_env,
            version=self.Context.app_ver,
            src_path=src_path,
        )

        # check files collisions (performance issue: check only for production for now)
        if project_env == 'production':
            self._yastatic_detect_collisions(static_files_src, yastatic_url)

        if project_env == 'production':
            self.Context.yastatic_artifact_id_prod = yastatic_artifact_id
        else:
            self.Context.yastatic_artifact_id_test = yastatic_artifact_id

    def _build_yastatuc_s3_resource(self, project_env, version, src_path):
        static_files_src = os.path.join(self.src_root, src_path)
        package_name = 'collections-static-s3'

        if project_env != 'production':
            version = version + '-' + project_env

        # create artifact with static files
        res = resources.CollectionsFrontendResourceStatic(
            self,
            '[{}] Frontend yastatic.net package {}'.format(project_env, package_name),
            '{}_{}.tar.gz'.format(package_name, version),
            env=project_env,
            version=version
        )
        artifact = sdk2.ResourceData(res)

        # create artifact
        self._create_tar(
            [static_files_src],
            str(artifact.path),
            relative=True
        )

        artifact.ready()

        return res.id

    def _build_yastatic(self, project_env):
        self._build_yastatic_project(
            project_env=project_env,
            prj='pdb'
        )

    def _build_yastatic_project(self, project_env, prj):
        yastatic_environment = self.yastatic_environment
        meta = self.yastatic_meta(project_env, prj)

        package_name = meta['package_name']
        current_version = meta['current_version']
        build_dir = meta['build_dir']

        src_root_debian_dir = meta['src_root_debian_dir']
        src_root_static_dir = meta['src_root_static_dir']

        build_debian_dir = os.path.join(build_dir, 'debian')
        # prj equal to pdb or answers
        build_dst_dir = os.path.join(build_dir, prj)

        if not os.path.isdir(src_root_static_dir):
            logging.warn('yastatic src static dir not exists: %s', src_root_static_dir)
            return

        copytree(
            src_root_debian_dir,
            build_debian_dir
        )

        copytree(
            src_root_static_dir,
            build_dst_dir,
            ignore=lambda _, names: [name for name in names if name.endswith('.gz') or name.endswith('.br')]
        )

        # create artifact with debian package files
        artifact = sdk2.ResourceData(resources.CollectionsFrontendResourceStatic(
            self,
            '[{}] Frontend yastatic.net package {}'.format(project_env, package_name),
            '{}_{}.tar.gz'.format(package_name, current_version),
            env=project_env,
            version=current_version
        ))

        # create artifact
        self._create_tar(
            [build_dst_dir],
            str(artifact.path),
            relative=True
        )

        artifact.ready()

        # check files collisions (performance issue: check only for production for now)
        if project_env == 'production':
            self._yastatic_detect_collisions(build_dst_dir, meta['yastatic_url'])

        # check version on dist
        if self._is_deb_package_exist(package_name, current_version):
            logging.warn('%s with version %s already exists in deb dist', package_name, current_version)
            return False

        # start debian package build process
        with CustomOsEnviron(yastatic_environment):
            # update changelog
            run_process(
                [
                    'dch',
                    '--newversion', current_version,
                    '--force-bad-version',
                    '--distribution', 'unstable',
                    '--force-distribution',
                    'Release of {package_name}@{version}'.format(
                        package_name=package_name,
                        version=current_version
                    )
                ],
                work_dir=build_dir,
                shell=True,
                log_prefix='yastatic_changelog'
            )

            with GpgKey(self, 'YASAP', 'yastatic_gpg_private_key', 'yastatic_gpg_public_key'), Key(self, 'YASAP', 'yastatic_ssh_key'), DebRelease(DUPLOAD_CONF):
                # build deb package
                run_process(
                    [
                        'debuild',
                        '--no-lintian',
                        '-b',
                        '-e{}'.format(yastatic_environment['DEBEMAIL'])
                    ],
                    work_dir=build_dir,
                    shell=True,
                    log_prefix='yastatic_debuild'
                )

                # deploy deb package to the dist
                run_process(
                    [
                        'debrelease',
                        '--to', DEB_REPO,
                        '--nomail',
                    ],
                    work_dir=build_dir,
                    shell=True,
                    log_prefix='yastatic_debrelease'
                )

                # add link to debian package
                dist_link = self._get_deb_package_url(package_name, current_version)
                logging.info('dist link: %s', dist_link)

                if dist_link:
                    self.set_info(
                        'Frontend yastatic.net debian package <a href="{dist_link}">{package_name}@{version}</a>'.format(
                            dist_link=dist_link,
                            package_name=package_name,
                            version=current_version
                        ),
                        do_escape=False
                    )

        return True

    def _release_yastatic(self, release_status):
        if release_status != 'stable':
            if self.Parameters.betastatic:
                # testing
                self._deploy_yastatic('testing', branch='testing')

            # prestable: deploy production package to betastatic.yastatic.net
            self._deploy_yastatic('production', branch='testing')
        else:
            # production
            self._deploy_yastatic('production', branch='stable')

    def _deploy_yastatic(self, project_env, branch):
        meta = self.yastatic_meta(project_env)

        package_name = meta['package_name']
        current_version = meta['current_version']

        package_title = '{}@{}'.format(package_name, current_version)

        # make conductor ticker
        self._create_conductor_ticket(
            package_name=package_name,
            version=current_version,
            branch=branch
        )

        self.set_info(
            'Conductor ticket for <a href="https://c.yandex-team.ru/packages/{}/tickets">{}</a> was created.'.format(
                package_name,
                package_title
            ),
            do_escape=False
        )

    def _release_yastatic_s3(self, release_status):
        if release_status != 'stable':
            # testing
            self._deploy_yastatic_s3('testing', self.Context.yastatic_artifact_id_test)

            # prestable: deploy production package to test env
            self._deploy_yastatic_s3('testing', self.Context.yastatic_artifact_id_prod)
        else:
            # production
            self._deploy_yastatic_s3('production', self.Context.yastatic_artifact_id_prod)

    def _deploy_yastatic_s3(self, project_env, resource_id):
        if resource_id:
            s3_bucket_name, key_prefix, prj = self._get_yastatic_s3_meta(project_env)

            CollectionsFrontendS3(
                self,
                priority=task.Priority(task.Priority.Class.SERVICE, task.Priority.Subclass.HIGH),
                description='Subtask of {}'.format(self.id),
                static_resource=resource_id,
                bucket_name=s3_bucket_name,
                freeze_path=prj,
                key_prefix=key_prefix,
                compress_needed=bool(project_env == 'production'),
            ).enqueue()

    @property
    def static_upload_script_name(self):
        return 'ci:collections:deploy:static'

    @property
    def is_static_upload_script_exist(self):
        with open(self.package_path) as package_file:
            data = json.load(package_file)
            scripts = data['scripts'] if 'scripts' in data else None
            return scripts is not None and self.static_upload_script_name in scripts
        return False

    def is_script_exist(self, script_name):
        with open(self.package_path) as package_file:
            data = json.load(package_file)
            scripts = data['scripts'] if 'scripts' in data else None
            return scripts is not None and script_name in scripts
        return False

    def _run_static_upload_script(self, project_env):
        yastatic_s3 = self.Parameters.yastatic_s3
        build_env = self.Parameters.build_env

        if self.is_static_upload_script_exist and yastatic_s3:
            env = build_env or {}
            env.update(self.environment)

            env['YENV'] = project_env

            run_process(
                [self.node_bin, self.npm_bin, 'run', self.static_upload_script_name],
                log_prefix='static',
                work_dir=str(self.src_root),
                environment=env
            )
