# -*- coding: utf-8 -*-

from urlparse import urlparse

import re
import requests
import logging
import os
import tarfile
import json
from contextlib import nested
from shutil import move, rmtree

from sandbox import sdk2
from sandbox.common.types import misc as ctm
from sandbox.common.types import task as ctt
from sandbox.common.types import resource as ctr
from sandbox.common.utils import singleton_property, get_resource_link
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.collections.CollectionsFrontendBase import resources
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.projects.sandbox_ci.utils.git import git_cache
from sandbox.projects.sandbox_ci.utils.context import GitWithoutLfsProcessing, Debug
from sandbox.projects.sandbox_ci import SANDBOX_CI_LXC_IMAGE, parameters, \
    SANDBOX_CI_NODE_MODULES_BUNDLE, SandboxCiGitCache
from sandbox.projects.sandbox_ci.parameters import ScriptsSourcesParameters
from sandbox.projects.sandbox_ci.managers import ArtifactsManager, ScriptsManager
from sandbox.projects.sandbox_ci.managers.dependencies import NPM_DEPS_FILES, files_sha1
from sandbox.projects.collections.mixins import get_vault

GIT_URL = 'https://github.yandex-team.ru/mm-interfaces/collections.git'

class CollectionsFrontendBase(sdk2.Task):
    class Requirements(sdk2.Requirements):
        disk_space = 4096
        dns = ctm.DnsType.DNS64
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 4096, None)
        ram = 16384
        cores = 8

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Source control') as source_block:
            project_git_base_ref = parameters.project_git_base_ref(
                default='dev',
            )
            project_git_base_commit = parameters.project_git_base_commit()
            project_git_merge_ref = parameters.project_git_merge_ref()
            project_git_merge_commit = parameters.project_git_merge_commit()

            use_cache_in_checkout = sdk2.parameters.Bool('Use cache in checkout', default=False)

        with sdk2.parameters.Group('Sandbox CI params') as sandbox_ci_params:
            node_js_version = sdk2.parameters.String(
                'SandboxCI node js version',
                description='Позволяет переопределить версию, которая выставлена в LXC-контейнере по умолчанию для sandbox_ci зависимостей',
            )

        with sdk2.parameters.Group('GitHub') as github:
            repository = sdk2.parameters.String('Repository', default=GIT_URL)
            pr_number = sdk2.parameters.String('Pull request number', default='')
            hash = sdk2.parameters.String('Commit', default='master')
            rebase = sdk2.parameters.Bool('Rebase', default=False)
            git_lfs = sdk2.parameters.Bool('Git Lfs', default=False)
            rebase_branch = sdk2.parameters.String('Rebase branch', default='master')
            rebase_ignore = sdk2.parameters.Bool('Ignore rebase errors', default=False)

        with sdk2.parameters.Group('Build') as build:
            dir = sdk2.parameters.String('Build subdirectory', default='')
            reuse_node_modules_cache = sdk2.parameters.Bool('Reuse node_modules cache', default=True)
            reuse_workcopy_cache = sdk2.parameters.Bool('Reuse workcopy cache resource', default=True)
            draft_mode = sdk2.parameters.Bool('Create subtasks in draft mode, do not run build', default=False)

        with sdk2.parameters.Group('Dependencies') as dependencies:
            node = sdk2.parameters.LastReleasedResource(
                'Node.js',
                resource_type=resources.CollectionsNodeBinary,
                required=True
            )

        with sdk2.parameters.Group('Debug') as debug:
            suspend = sdk2.parameters.Bool('Suspend execution', default=False)
            use_ram_drive = sdk2.parameters.Bool('Use ram drive', default=False)

        _container = sdk2.parameters.Container(
            'Build environment',
            resource_type=SANDBOX_CI_LXC_IMAGE,
            platform='linux_ubuntu_16.04_xenial',  # necessary for correct default resource searching
            required=True,
        )

        kill_timeout = 1800
        dump_disk_usage = False
        scripts_sources = ScriptsSourcesParameters

    class Context(sdk2.Context):
        footer = ''
        tag = ''
        app_ver = ''
        feature_branch = ''
        yastatic_artifact_id_prod = None
        yastatic_artifact_id_test = None

    project_name = 'collections'

    @singleton_property
    def artifacts(self):
        return ArtifactsManager(self)

    @singleton_property
    def scripts(self):
        return ScriptsManager(self, 'scripts')

    @property
    def environment(self):
        environment = os.environ.copy()
        environment['PATH'] = self.node_bin_root + ':' + os.environ['PATH']
        environment['GITHUB_OAUTH_TOKEN'] = self.github_oauth
        environment['STAT_OAUTH_TOKEN'] = self.stat_oauth
        environment['GITHUB_HASH'] = self.Parameters.hash
        environment['GITHUB_SHORT_HASH'] = self.Parameters.hash[0:7]
        environment['GITHUB_PR_NUMBER'] = self.pr_number
        environment['SANDBOX_ID'] = str(self.id)
        environment['IS_SANDBOX'] = 'True'
        environment['CI'] = 'true'
        environment['TRENDBOX_CI'] = 'true'
        environment['TRENDBOX_PULL_REQUEST_NUMBER'] = self.pr_number
        environment['FRONTEND_S3_ACCESS_KEY_ID'] = self.s3_access_key_id
        environment['FRONTEND_S3_SECRET_ACCESS_KEY'] = self.s3_secret_access_key

        environment['NANNY_OAUTH_TOKEN'] = get_vault('YASAP', 'nanny_oauth_token')
        environment['SANDBOX_OAUTH_TOKEN'] = get_vault('YASAP', 'sandbox_oauth_token')
        environment['SANDBOX_SYNCHROPHAZOTRON_PATH'] = str(self.synchrophazotron)

        return environment

    @property
    def project_dir(self):
        return 'pdb-frontend'

    @property
    def src_root(self):
        if self.Parameters.use_ram_drive:
            return self.working_path(self.project_dir)

        return os.path.abspath(self.project_dir)

    @property
    def node_root(self):
        if self.Parameters.use_ram_drive:
            return str(self.working_path('node'))

        return os.path.abspath('node')

    @property
    def package_path(self):
        if self.Parameters.use_ram_drive:
            return str(self.working_path(self.project_dir, 'package.json'))

        return os.path.join(self.src_root, 'package.json')

    @property
    def node_bin_root(self):
        return os.path.join(self.node_root, 'bin')

    @property
    def node_bin(self):
        if self.Parameters.use_ram_drive:
            return str(self.working_path('node', 'bin', 'node'))

        return os.path.join(self.node_bin_root, 'node')

    @property
    def npm_bin(self):
        if self.Parameters.use_ram_drive:
            return str(self.working_path('node', 'bin', 'npm'))

        return os.path.join(self.node_bin_root, 'npm')

    @property
    def gulp_bin(self):
        if self.Parameters.use_ram_drive:
            return str(self.working_path(self.project_dir, 'node_modules', '.bin', 'gulp'))

        return os.path.join(self.src_root, 'node_modules/.bin/gulp')

    @property
    def issue_keys_bin(self):
        if self.Parameters.use_ram_drive:
            return str(self.working_path(self.project_dir, 'node_modules', '.bin', 'issue-keys'))

        return os.path.join(self.src_root, 'node_modules/.bin/issue-keys')

    @property
    def nm_path(self):
        if self.Parameters.use_ram_drive:
            return str(self.working_path(self.project_dir, 'node_modules'))

        return os.path.join(self.src_root, 'node_modules')

    @property
    def pr_number(self):
        pr_number = self.Parameters.pr_number

        if pr_number and pr_number != 'test':
            return pr_number

        return ''

    @property
    def github_oauth(self):
        return sdk2.Vault.data('YASAP', 'GITHUB_OAUTH_TOKEN')

    @property
    def stat_oauth(self):
        return sdk2.Vault.data('YASAP', 'STAT_OAUTH_TOKEN')

    @property
    def s3_access_key_id(self):
        return sdk2.Vault.data('YASAP', 'collections_s3_key_id')

    @property
    def s3_secret_access_key(self):
        return sdk2.Vault.data('YASAP', 'collections_s3_secret_key')

    @property
    def footer(self):
        if self.Context.footer:
            return self.Context.footer

        return ''

    def working_path(self, *args):
        if self.Parameters.use_ram_drive and self.ramdrive and self.ramdrive.path:
            return self.ramdrive_path(*args)

        return self.path(*args)

    def ramdrive_path(self, *args):
        return self.ramdrive.path.joinpath(*args) if args else self.ramdrive.path

    def on_prepare(self):
        self.scripts.sync_resource()

    def check_build_cache(self):
        if not self.Parameters.reuse_workcopy_cache:
            return None

        return self.artifacts.get_last_project_artifact_resource(
            resource_type=resources.CollectionsFrontendCiArtifact,
            type=self.project_name,
            status=ctt.Status.SUCCESS,
            project_tree_hash=self.Parameters.hash
        )

    def prepare_working_copy(self):
        workcopy = self.check_build_cache()

        if workcopy:
            logging.debug('Reuse project workcopy from cache resource')
            self.artifacts.unpack_build_artifacts([workcopy], self.working_path(self.src_root), 1)
        else:
            logging.debug('Project workcopy resource not found. Performing checkout from scratch…')
            self.checkout()

        # self.checkout_git_meta()

        self.prepare_node()

    def checkout(self):
        if self.Parameters.use_cache_in_checkout:
            self.checkout_with_cache()
        else:
            self.checkout_without_cache()

        if self.Parameters.reuse_workcopy_cache and not self.check_build_cache():
            workcopy = self.artifacts.create_project_artifact_resource(
                relative_path='collections.tar.gz',
                resource_type=resources.CollectionsFrontendCiArtifact,
                type=self.project_name,
                status=ctt.Status.SUCCESS,
                project_tree_hash=self.Parameters.hash,
                public=True,
            )
            collection_package = sdk2.ResourceData(workcopy)

            if self.Parameters.use_ram_drive:
                self._create_tar([str(self.src_root)], 'collections.tar.gz')
            else:
                self._create_tar([os.path.basename(self.src_root)], 'collections.tar.gz')

            collection_package.ready()

    def checkout_with_cache(self):
        git_url = self.Parameters.repository

        git_cache_dir = self.fetch_repository_cache()
        if git_cache_dir is None:
            git_cache_dir = git_cache(git_url)

        git_base_ref = self.Parameters.project_git_base_ref
        git_base_commit = self.Parameters.project_git_base_commit
        git_merge_ref = self.Parameters.project_git_merge_ref
        git_merge_commit = self.Parameters.project_git_merge_commit

        params = {
            'cache-dir': git_cache_dir,

            'ref': git_base_ref,
            'commit': git_base_commit if git_base_commit else '',

            'merge-ref': git_merge_ref if git_merge_ref else '',
            'merge-commit': git_merge_commit if git_merge_commit else '',

            'author-name': 'robot-pdb-builder',
            'author-email': 'robot-pdb-builder@yandex-team.ru',
        }

        contexts = (Debug('*'),)
        if self.Parameters.git_lfs:
            contexts += (GitWithoutLfsProcessing(git_url),)
            params.update(lfs=True)

        logging.debug('Prepare working copy with params: {}'.format(params))

        with nested(*contexts):
            self.scripts.run_js(
                'script/prepare-working-copy.js',
                git_url,
                self.working_path('collections'),
                params,
            )

        # Creating symlink: pdb-frontend -> collections
        working_path_str = str(self.working_path('collections'))
        src_path_str = str(self.src_root)

        run_process(
            ['cp', '--verbose', '--recursive', '--link', working_path_str, src_path_str],
            work_dir=str(self.working_path()),
            log_prefix='make-link',
        )

        os.symlink(
            str(self.working_path('collections', '.git')),
            str(self.working_path(self.project_dir, '.git')),
        )

        run_process(
            ['git', 'fetch', 'origin', '"+refs/tags/*:refs/tags/*"'],
            work_dir=str(self.working_path('collections')),
            log_prefix='git-fetch-origin',
        )

        self.Context.tag = run_process(
            ['git', 'describe', '--tags', '--always'],
            work_dir=str(self.working_path('collections')),
            outs_to_pipe=True,
        ).stdout.read().strip().replace('collections/', '')

        logging.debug('Tag: "{}"'.format(self.Context.tag))

        match = re.findall(r'^v([0-9].+?)$', self.Context.tag)
        if len(match):
            self.Context.app_ver = match[0]
        else:
            hash = run_process(
                ['git', 'rev-parse', 'HEAD'],
                work_dir=str(self.working_path('collections')),
                outs_to_pipe=True,
            ).stdout.read().strip()

            logging.debug('Hash: "{}"'.format(hash))

            with open(self.package_path) as package_file:
                data = json.load(package_file)
                self.Context.app_ver = data['version'] + '-' + hash[:8]

    def fetch_repository_cache(self):
        git_cache_dir = None
        cache_resource_type = SandboxCiGitCache
        cache_resource_attrs = {
            'git_ref': 'dev',
            'git_url': 'git@github.yandex-team.ru:mm-interfaces/collections.git',
        }

        logging.debug('Looking for resource "{resource_type}" with attrs: {attrs}'.format(
            resource_type=cache_resource_type,
            attrs=cache_resource_attrs,
        ))

        cache_resource = sdk2.Resource.find(
            resource_type=cache_resource_type,
            attrs=cache_resource_attrs,
            state=ctr.State.READY,
        ).first()

        if cache_resource:
            logging.debug('Cache resource: {}'.format(cache_resource))

            resource_path = sdk2.ResourceData(cache_resource).path
            archive_path = str(self.working_path(resource_path.name))

            logging.debug('Syncing {resource} from {from_path} to {dest_path}'.format(
                resource=cache_resource,
                from_path=resource_path,
                dest_path=archive_path,
            ))

            with tarfile.open(str(resource_path), mode='r:*') as tar:
                git_cache_dir = str(self.working_path('fiji_cache'))
                logging.debug('Extracting {archive} to {destination}'.format(
                    archive=archive_path,
                    destination=git_cache_dir,
                ))
                tar.extractall(git_cache_dir)

        return git_cache_dir

    def checkout_without_cache(self):
        """
        Clone git repository
        """
        logging.info('Checking out %s tag of repository',
                     self.Parameters.hash)

        src_root = str(self.src_root)

        run_process(
            ['git', 'clone', self.Parameters.repository, src_root],
            log_prefix='git_clone'
        )

        run_process(
            ['git', 'checkout', self.Parameters.hash],
            work_dir=src_root,
            log_prefix='git_checkout'
        )

        try:
            if self.Parameters.rebase:
                run_process(
                    ['git', 'config', '--global', 'user.email', 'polubob@yandex.ru'],
                    work_dir=src_root,
                    log_prefix='git_config'
                )
                run_process(
                    ['git', 'config', '--global', 'user.name', 'polubob'],
                    work_dir=src_root,
                    log_prefix='git_config'
                )
                run_process(
                    ['git', 'rebase', str(self.Parameters.rebase_branch)],
                    work_dir=src_root,
                    log_prefix='git_rebase'
                )

            if self.Parameters.git_lfs:
                run_process(
                    ['git', 'lfs', 'fetch', '--exclude="video,images"'],
                    work_dir=src_root,
                    log_prefix='git_lfs_fetch'
                )

                run_process(
                    ['git', 'lfs', 'checkout'],
                    work_dir=src_root,
                    log_prefix='git_lfs_checkout'
                )
        except SandboxSubprocessError:
            if self.Parameters.rebase_ignore:
                run_process(
                    ['git', 'rebase', '--abort'],
                    work_dir=src_root,
                    log_prefix='git_rebase_abort'
                )
            else:
                raise

        self.Context.tag = run_process(
            ['git', 'describe', '--tags', '--always'],
            outs_to_pipe=True,
            work_dir=src_root
        ).stdout.read().strip()

        match = re.findall(r'^v([0-9].+?)$', self.Context.tag)
        if len(match):
            self.Context.app_ver = match[0]
        else:
            hash = run_process(
                ['git', 'rev-parse', 'HEAD'],
                outs_to_pipe=True,
                work_dir=src_root
            ).stdout.read().strip()

            with open(self.package_path) as package_file:
                data = json.load(package_file)
                self.Context.app_ver = data['version'] + '-' + hash[:8]

        return src_root

    def checkout_git_meta(self):
        src_root = str(self.src_root)

        self.Context.feature_branch = run_process(
            ['git', 'rev-parse', '--abbrev-ref', 'HEAD'],
            outs_to_pipe=True,
            work_dir=src_root
        ).stdout.read().strip()

    def prepare_node(self):
        """
            Unpack node.js
        """
        node = sdk2.ResourceData(self.Parameters.node)
        node_path = self.node_root

        self._extract(str(node.path), str(node_path))

        return node_path

    def get_node_version(self):
        return run_process(
            [self.node_bin, '-p', 'process.versions.modules'],
            outs_to_pipe=True,
            work_dir=str(self.src_root)
        ).stdout.read().strip()

    def get_npm_deps_cache_checksum(self):
        deps_paths = [self.working_path(self.src_root, deps_file) for deps_file in NPM_DEPS_FILES]

        logging.debug('Dependencies: {}'.format(', '.join(map(str, deps_paths))))

        return files_sha1(*deps_paths)

    def get_npm_deps_cache(self, resource_type, checksum, **kwargs):
        attrs = dict(
            project=self.project_name,
            checksum=checksum,
            **kwargs
        )

        logging.info('Trying to find resource "{name}" with attributes: {attrs}'.format(
            name=resource_type.name,
            attrs=attrs,
        ))

        return self.artifacts.get_last_artifact_resource(resource_type, **attrs)

    def install_modules(self, production=False):
        """
            Install node modules
        """
        install_mode = '--production' if production else '--development'

        node_version = self.get_node_version()
        checksum = self.get_npm_deps_cache_checksum()

        logging.debug('Dependency files checksum: {}'.format(checksum))
        if self.Parameters.reuse_node_modules_cache:
            cache = self.get_npm_deps_cache(
                resource_type=SANDBOX_CI_NODE_MODULES_BUNDLE,
                checksum=checksum,
                install_mode=install_mode,
                node_version=node_version,
            )
            if cache:
                self.sync_npm_deps_cache(cache)
                return self.set_cache_reuse_info(cache)

        logging.debug('Installing dependencies in {} mode'.format(install_mode))

        if production:
            command = 'ci'
            log_prefix = 'npm_ci'
        else:
            command = 'install'
            log_prefix = 'npm_install'

        run_process(
            [self.node_bin, self.npm_bin, '--registry=https://npm.yandex-team.ru', command, install_mode],
            log_prefix=log_prefix,
            work_dir=str(self.src_root),
            environment=self.environment,
        )

        return self.create_npm_deps_cache(
            resource_type=SANDBOX_CI_NODE_MODULES_BUNDLE,
            checksum=checksum,
            install_mode=install_mode,
            node_version=node_version,
        )

    def sync_npm_deps_cache(self, cache_resource):
        node_modules_path = self.working_path(self.src_root, 'node_modules')

        logging.debug('Unpacking resource contents from {} into {}'.format(cache_resource, node_modules_path))

        return self.artifacts.unpack_build_artifacts([cache_resource], node_modules_path)

    def create_npm_deps_cache(self, resource_type, checksum, **kwargs):
        node_modules_path = self.working_path(self.src_root, 'node_modules')
        archive_filename = self.working_path('node_modules.tar.gz')

        logging.debug('Packing contents of {dir} into {archive}'.format(
            dir=node_modules_path,
            archive=archive_filename,
        ))

        archive = self.artifacts.pack_into_archive(node_modules_path, archive_filename=archive_filename, gzip=True)
        attrs = dict(
            project=self.project_name,
            checksum=checksum,
            **kwargs
        )

        logging.debug('Creating resource "{name}" from file {archive} with attributes: {attrs}'.format(
            name=resource_type.name,
            archive=archive_filename,
            attrs=attrs,
        ))

        cache_resource = self.artifacts.create_artifact_resource(archive, resource_type, **attrs)
        sdk2.ResourceData(cache_resource).ready()

        return cache_resource

    def set_cache_reuse_info(self, cache):
        self.set_info('{title}: <a href="{url}">{type}#{id}</a>'.format(
            title='Reused node_modules cache',
            id=cache.id,
            url=get_resource_link(cache.id),
            type=cache.type,
        ), do_escape=False)

    def create_task(self, task, **kwargs):
        return task(
            self,
            priority=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.HIGH),
            description='Subtask of {}'.format(self.id),
            repository=self.Parameters.repository,
            pr_number=self.Parameters.pr_number,
            hash=self.Parameters.hash,
            rebase=self.Parameters.rebase,
            rebase_branch=self.Parameters.rebase_branch,
            rebase_ignore=self.Parameters.rebase_ignore,
            reuse_node_modules_cache=self.Parameters.reuse_node_modules_cache,
            node=self.Parameters.node,

            use_cache_in_checkout=self.Parameters.use_cache_in_checkout,

            project_git_base_ref=self.Parameters.project_git_base_ref,
            project_git_base_commit=self.Parameters.project_git_base_commit,

            project_git_merge_ref=self.Parameters.project_git_merge_ref,
            project_git_merge_commit=self.Parameters.project_git_merge_commit,

            **kwargs
        )

    def enqueue_task(self, task):
        if not self.Parameters.draft_mode:
            task.enqueue()

    def report_to_github(self, context, state, description=''):
        token = self.github_oauth
        path = urlparse(str(self.Parameters.repository)).path

        if path.endswith('.git'):
            path = path[1:-4]

        url = 'https://api.github.yandex-team.ru/repos/{}/statuses/{}' \
            .format(path, self.Parameters.hash)

        logging.info('Report status to github. Url: %s. Context: %s.', url, context)

        return requests.post(
            url,
            headers={
                'Authorization': 'token {}'.format(str(token))
            },
            json={
                'context': context,
                'target_url': 'https://sandbox.yandex-team.ru/task/{}/view'.format(str(self.id)),
                'description': description,
                'state': state
            }
        )

    def report_pending_to_github(self, context, description=''):
        return self.report_to_github(context, 'pending', description)

    def report_success_to_github(self, context, description=''):
        return self.report_to_github(context, 'success', description)

    def report_error_to_github(self, context, description=''):
        return self.report_to_github(context, 'error', description)

    def _create_tar(self, filenames, tarname, relative=False, tarmode='w:gz'):
        tar = tarfile.open(tarname, tarmode)
        # Dereferencing links
        # tar.dereference = True

        def excludes_fn(name):
            str_name = str(name)

            return ('.git' in str_name) or '{}/cache'.format(self.project_dir) in str_name

        if self.Parameters.suspend:
            self.suspend()

        for fname in filenames:
            logging.info('Add %s to archive', fname)
            arcname = os.path.basename(fname) if relative else None

            if self.Parameters.use_ram_drive:
                tar.add(str(fname), arcname=arcname, exclude=excludes_fn)
            else:
                tar.add(fname, arcname=arcname, exclude=excludes_fn)

        tar.close()

    def _extract(self, fname, newfname=None):
        if tarfile.is_tarfile(fname):
            if not newfname:
                idx = fname.find('.tar')
                newfname = fname[:idx]
            tar = tarfile.open(fname)
            tar.extractall(path=newfname)
            tar.close()

            return newfname
        return fname
