# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.sdk2.task import WaitTask
from sandbox.common.types import task as ctt
from sandbox.projects.collections.CollectionsFrontendE2E import CollectionsFrontendE2E
from sandbox.projects.sandbox_ci import SANDBOX_CI_LXC_IMAGE


class CollectionsFrontendExperimentsRunner(sdk2.Task):
    """Запуск тестов Я.Коллекций для с экспериментальными флагами"""

    name = 'COLLECTIONS_FRONTEND_EXPERIMENTS_RUNNER'
    project_task_name = 'COLLECTIONS_FRONTEND'

    class Requirements(sdk2.Requirements):
        cores = 1

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('abt'):
            test_id = sdk2.parameters.String(
                'AB test id',
                required=True,
            )

        with sdk2.parameters.Group('source'):
            release = sdk2.parameters.RadioGroup(
                'Release',
                choices=(
                    ('latest release', 'latest_release'),
                    ('trunk', 'trunk'),
                    ('custom', 'custom'),
                ),
                sub_fields={'custom': ['git_commit']},
                default='latest_release',
                required=True,
            )
            git_commit = sdk2.parameters.String(
                'Git commit',
                required=True,
            )

        with sdk2.parameters.Group('startrek'):
            st_issue = sdk2.parameters.String(
                'Startrek issue id',
            )

        _container = sdk2.parameters.Container(
            'Build environment',
            resource_type=SANDBOX_CI_LXC_IMAGE,
            platform='linux_ubuntu_16.04_xenial',  # necessary for correct default resource searching
            required=True,
        )

    def run_e2e_test_task(self, base_task, **kwargs):
        return CollectionsFrontendE2E(
            self,
            priority=self.Parameters.priority,
            description='running automatic ab deployment collections test [{}]'.format(self.id),

            rebase=False,
            reuse_node_modules_cache=True,
            use_cache_in_checkout=True,

            # 'https://github.yandex-team.ru/mm-interfaces/collections.git'
            repository=base_task.Parameters.repository,
            # '.'
            dir=base_task.Parameters.dir,

            node=base_task.Parameters.node,
            hash=base_task.Parameters.hash,
            project_git_base_ref=base_task.Parameters.project_git_base_ref,
            project_git_base_commit=base_task.Parameters.project_git_base_commit,

            **kwargs
        )

    def get_latest_release_task(self):
        """
        Получаем задачу для последнего релиза
        :rtype: sdk2.Task
        """
        task_type = self.project_task_name
        release_type = 'stable',

        task = sdk2.Task.find(
            type=task_type,
            status=ctt.Status.RELEASED,
            release=release_type,
            children=True,
            tags='HOTFIX',
            input_parameters={
                'production': True,
            },
            order='-id',
        ).first()

        if not task:
            self.set_info('Released task {} not found'.format(task_type))
            return

        logging.debug(
            'Last released in "{release_type}" task with type "{type}": {task}'.format(
                release_type=release_type,
                type=task_type,
                task=task,
            )
        )

        return task

    def get_trunk_task(self):
        """
        Получаем задачу последнего собранного dev
        :rtype: sdk2.Task
        """
        task_type = self.project_task_name

        task = sdk2.Task.find(
            type=task_type,
            status=ctt.Status.SUCCESS,
            children=True,
            tags='TRUNK',
            order='-id',
        ).first()

        if not task:
            self.set_info('Trunk task {} not found'.format(task_type))
            return

        logging.debug(
            'Last trunk task with type "{type}": {task}'.format(
                type=task_type,
                task=task,
            )
        )

        return task

    def run_all_tests(self):
        presets = [
            'mobile',
            'desktop',
            'features:mobile',
            'features:desktop',
        ]

        custom_subtask_params = {}
        if self.Parameters.release == 'latest_release':
            base_task = self.get_latest_release_task()
        elif self.Parameters.release == 'latest_release' == 'trunk':
            base_task = self.get_trunk_task()
        else:
            base_task = self.get_trunk_task()
            git_commit = self.Parameters.git_commit
            custom_subtask_params = {
                'hash': git_commit,
                'project_git_base_commit': git_commit,
                'project_git_base_ref': '',
            }

        subtasks = []
        for preset in presets:
            subtask = self.run_e2e_test_task(
                base_task,
                preset=preset,
                flags=['--testid=' + str(self.Parameters.test_id)],
                **custom_subtask_params
            ).enqueue()

            subtasks.append(subtask)

        return subtasks

    def get_result_report(self, tasks_report):
        success_badge = '<span class="status status_success">success</span>'
        fail_badge = '<span class="status status_failure">failure</span>'

        result_report = ''
        for (success, preset, report_url) in tasks_report:
            badge = success_badge if success else fail_badge
            desc = '<a href="{url}" target="_blank">{text}</a>'.format(url=report_url, text=preset)
            result_report += '<tr><td>{badge}</td><td>{desc}</td></tr>'.format(badge=badge, desc=desc)

        return '<h3>Report:</h3><table>{}</table>'.format(result_report)

    @sdk2.header()
    def header(self):
        return self.Context.report_data

    def on_execute(self):
        with self.memoize_stage.wait_subtasks:
            subtasks = self.run_all_tests()
            self.Context.subtasks_ids = [subtask.id for subtask in subtasks]

            raise WaitTask(
                self.Context.subtasks_ids,
                statuses=ctt.Status.Group.FINISH | ctt.Status.Group.BREAK
            )

        tasks_report = []
        invalid_tasks = []
        for task_id in self.Context.subtasks_ids:
            task = sdk2.Task[task_id]

            tasks_report.append((
                task.status == ctt.Status.SUCCESS,
                task.Parameters.preset,
                task.Context.report_url
            ))

            if task.status != ctt.Status.SUCCESS:
                invalid_tasks.append(task)

        self.Context.report_data = self.get_result_report(tasks_report)

        # TODO: send notifications into issue tracker: self.Parameters.st_issue

        if invalid_tasks:
            self.set_info('ERROR: Some subtasks failed: {!r}'.format(invalid_tasks))
            raise TaskFailure('Some subtasks failed')
