# -*- coding: utf-8 -*-

import os
from sandbox import sdk2
from sandbox import common
import sandbox.common.types.misc as ctm

from sandbox.projects.collections.CollectionsFrontendBase.base import CollectionsFrontendBase
from sandbox.projects.collections.CollectionsFrontendBase.resources import CollectionsFrontendResourceGeminiReport
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.process import run_process
from sandbox.common import config
from sandbox.projects.sandbox_ci import SANDBOX_CI_LXC_IMAGE
from sandbox.common.errors import TaskFailure

DEFAULT_TUNNELER_HOSTS = '["ci2-tunnelerapi.si.yandex-team.ru", "ci3-tunnelerapi.si.yandex-team.ru"]'


class CollectionsFrontendGemini(CollectionsFrontendBase):
    """
        Collections gemini tests
    """

    name = 'COLLECTIONS_FRONTEND_GEMINI'

    class Requirements(CollectionsFrontendBase.Requirements):
        dns = ctm.DnsType.LOCAL

    class Parameters(CollectionsFrontendBase.Parameters):
        kill_timeout = 3600
        _container = sdk2.parameters.Container(
            'Build environment',
            resource_type=SANDBOX_CI_LXC_IMAGE,
            platform='linux_ubuntu_16.04_xenial',  # necessary for correct default resource searching
            required=True,
        )

        with CollectionsFrontendBase.Parameters.github() as github:
            git_lfs = sdk2.parameters.Bool('Git Lfs', default=True)

        with sdk2.parameters.Group('Features') as features:
            platform = sdk2.parameters.String('Platform', default='mobile')
            flags = sdk2.parameters.List('Flags', default=[])

        with sdk2.parameters.Group('Tunneler settings') as tunneler_settings:
            tunneler_user = sdk2.parameters.String('Tunneller user', default='buildfarm')
            tunneler_hosts = sdk2.parameters.String('Tunneller hosts', default=DEFAULT_TUNNELER_HOSTS)

    def get_report_http_proxy_link(self, report):
        return common.rest.Client().resource[report.id].read()["http"]["proxy"]

    class Context(CollectionsFrontendBase.Context):
        report_url = ''

    def create_gemini_report_resource(self):
        try:
            if self.Parameters.use_ram_drive:
                report_path = self.src_root.joinpath('gemini-report')
            else:
                report_path = os.path.join(self.src_root, 'gemini-report')

            res = CollectionsFrontendResourceGeminiReport(
                self,
                'Gemini report: {}'.format(self.Parameters.platform),
                report_path,
                platform=self.Parameters.platform,
                commit=self.Parameters.hash,
            )

            report = sdk2.ResourceData(res)
            report.ready()

            report_url = '{}/index.html'.format(self.get_report_http_proxy_link(res)) if report else ''

            if report_url:
                self.Context.report_url = report_url
        except Exception as e:
            raise TaskFailure('Could not make resource {}'.format(e))

    @sdk2.header()
    def header(self):
        report_url = self.Context.report_url

        return '<h4><a href="{}" target="_blank">Gemini Report</a></h4>'.format(report_url) if report_url else ''

    def on_execute(self):
        platform = self.Parameters.platform
        context = '[Collections CI] Gemini: {}'.format(platform)

        self.report_pending_to_github(context)

        try:
            self.prepare_working_copy()
            self.install_modules()

            environment = self.environment
            environment['TUNNELER_USERNAME'] = self.Parameters.tunneler_user
            environment['TUNNELER_HOST_ID'] = config.Registry().this.id
            environment['tunneler_hosts'] = self.Parameters.tunneler_hosts

            with sdk2.ssh.Key(self, 'SANDBOX_CI_SEARCH_INTERFACES', 'teamcity.id_rsa'):
                run_params = [
                    self.node_bin,
                    self.gulp_bin,
                    'gemini',
                    '--set',
                    platform
                ]

                if self.Parameters.flags:
                    run_params = run_params + self.Parameters.flags

                run_process(
                    run_params,
                    work_dir=self.src_root,
                    log_prefix='gemini',
                    environment=environment
                )
                self.create_gemini_report_resource()

        except SandboxSubprocessError:
            self.report_error_to_github(context)
            self.create_gemini_report_resource()
            raise

        self.report_success_to_github(context)
