# -*- coding: utf-8 -*-

import logging
import tarfile
import os

from sandbox import sdk2
from sandbox.projects.collections.CollectionsFrontendBase.resources import CollectionsFrontendResourceStatic
from sandbox.projects.sandbox_ci import SANDBOX_CI_LXC_IMAGE

from sandbox.sandboxsdk.environments import PipEnvironment

from sandbox.projects.sandbox_ci.managers.s3mds import S3Manager
from sandbox.projects.sandbox_ci.utils.static import compress_static
from sandbox.projects.sandbox_ci.utils.process import run_process


class CollectionsFrontendS3(sdk2.Task):
    """Загрузка статики через S3 API в MDS"""

    name = 'COLLECTIONS_FRONTEND_S3'

    class Requirements(sdk2.Requirements):
        cores = 1

        environments = (
            PipEnvironment('boto3', '1.4.4', use_wheel=True),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Parameters):
        static_resource = sdk2.parameters.Resource(
            'Static resource',
            resource_type=CollectionsFrontendResourceStatic,
            required=True,
        )

        bucket_name = sdk2.parameters.String(
            'Bucket name (default: serp-static-testing)',
            default='serp-static-testing',
        )

        freeze_path = sdk2.parameters.String(
            'Path to dir with resource files (default: collections)',
            default='collections',
        )

        key_prefix = sdk2.parameters.String(
            'Key prefix (default: podb)',
            default='podb',
        )

        compress_needed = sdk2.parameters.Bool(
            'Create .gz and .br versions before upload (default: false)',
            default=False,
        )

        vault_key_id = sdk2.parameters.String(
            'Name of secret which is Key id for S3 (default: collections_s3_key_id)',
            default='collections_s3_key_id',
        )

        vault_secret_key = sdk2.parameters.String(
            'Name of secret which is Access key for S3 (default: collections_s3_secret_key)',
            default='collections_s3_secret_key',
        )

        vault_group = sdk2.parameters.String(
            'Name of group for Key id and Access key secrets (default: YASAP)',
            default='YASAP'
        )

        _container = sdk2.parameters.Container(
            'Build environment',
            resource_type=SANDBOX_CI_LXC_IMAGE,
            platform='linux_ubuntu_16.04_xenial',  # necessary for correct default resource searching
            required=True,
        )

    @property
    def key_prefix(self):
        return str(self.Parameters.key_prefix)

    @property
    def s3_bucket_name(self):
        if self.Parameters.bucket_name:
            return str(self.Parameters.bucket_name)

        return 'serp-static-testing'

    @property
    def resource(self):
        return self.Parameters.static_resource

    @property
    def compress_needed(self):
        return self.Parameters.compress_needed

    def static_files_src(self):
        return str(self.path('yastatic'))

    def deploy_static_files_src(self):
        return os.path.join(self.static_files_src(), str(self.Parameters.freeze_path))

    def vault_s3_secrets(self):
        s3_key_id = self.Parameters.vault_key_id
        s3_secret_key = self.Parameters.vault_secret_key
        vault_group = self.Parameters.vault_group

        key_id = sdk2.Vault.data(vault_group, s3_key_id)
        access_key = sdk2.Vault.data(vault_group, s3_secret_key)

        return key_id, access_key

    def extract_tar(self, src, dst=None):
        logging.info('Extracting tar files from "{src}" to "{dst}"'.format(src=src, dst=dst))

        if tarfile.is_tarfile(src):
            if not dst:
                idx = src.find('.tar')
                dst = src[:idx]

            tar = tarfile.open(src)
            tar.extractall(path=dst)
            tar.close()

            return dst
        return src

    def prepare(self):
        logging.info('Prepare static files')
        artifact = sdk2.ResourceData(self.resource)
        self.extract_tar(str(artifact.path), self.static_files_src())

        # remove all gzip files
        run_process([
            'find', self.static_files_src(),
            '-name', '"*.gz"',
            '-type', 'f',
            '-delete',
        ])

        # remove all brotli files
        run_process([
            'find', self.static_files_src(),
            '-name', '"*.br"',
            '-type', 'f',
            '-delete',
        ])

    def compress(self):
        logging.info('Creating .gz and .br versions')
        compress_static(self.deploy_static_files_src())

    def upload_to_mds(self):
        logging.info('Starting to upload files to mds')
        key_id, access_key = self.vault_s3_secrets()
        client = S3Manager(key_id, access_key, self.s3_bucket_name)
        client.upload_dir(self.deploy_static_files_src(), self.key_prefix)

    def on_execute(self):
        self.prepare()
        if self.compress_needed:
            self.compress()
        self.upload_to_mds()
