# -*- coding: utf-8 -*-

from urlparse import urlparse

from sandbox import sdk2

import datetime as dt
import requests
import logging

from sandbox.projects.collections.CollectionsFrontendBase.base import CollectionsFrontendBase
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.process import run_process

TIME_FORMAT = "%Y-%m-%d"

class CollectionsFrontendTankerDownload(CollectionsFrontendBase):
    """
        Collections translates update
    """

    class Parameters(CollectionsFrontendBase.Parameters):
        ticket = sdk2.parameters.String('Ticket', default=None)

    def on_execute(self):
        ticket = self.Parameters.ticket
        src_root = str(self.src_root)

        try:
            self.prepare_working_copy()
            self.install_modules()

            environment = self.environment
            environment['STOP_WATCH'] = 'true'

            if ticket:
                branch_name = 'collections/polubob.translates-update/' + ticket
                commit_message = ticket + ': tanker translates update'
            else:
                date = dt.datetime.now().strftime(TIME_FORMAT)
                branch_name = 'collections/polubob.translates-update-' + date + '/TRIVIAL'
                commit_message = 'TRIVIAL : tanker translates update'

            # git auth
            run_process(
                ['git', 'config', '--global', 'user.email', 'polubob@yandex.ru'],
                work_dir=src_root,
                log_prefix='git_config'
            )
            run_process(
                ['git', 'config', '--global', 'user.name', 'polubob'],
                work_dir=src_root,
                log_prefix='git_config'
            )

            # checkout new branch
            run_process(
                ['git', 'checkout', '-b', branch_name],
                work_dir=src_root,
                shell=True,
                timeout=10000,
                log_prefix='git_checkout_branch'
            )

            # npm run tanker:download
            run_process(
                [
                    self.node_bin,
                    self.npm_bin,
                    'run'
                    'tanker:download'
                ],
                work_dir=self.src_root,
                log_prefix='tanker_download',
                environment=environment,
                timeout=30000
            )

            run_process(
                ['git', 'add', '.'],
                work_dir=self.src_root,
                log_prefix='git_add'
            )

            run_process(
                ['git', 'commit', '-m', commit_message],
                work_dir=self.src_root,
                log_prefix='git_commit'
            )

            run_process(
                ['git', 'push', 'origin', branch_name],
                work_dir=self.src_root,
                shell=True,
                timeout=10000,
                log_prefix='git_push'
            )

            self.create_pr(context, branch_name)

        except SandboxSubprocessError:
            raise

    def create_pr(self, context, branch_name):
        token = self.github_oauth
        path = urlparse(str(self.Parameters.repository)).path

        if path.endswith('.git'):
            path = path[1:-4]

        url = 'https://api.github.yandex-team.ru/repos/{}/pulls' \
            .format(path, self.Parameters.hash)

        logging.info('Post pull request to github. Url: %s. Context: %s.', url, context)

        return requests.post(
            url,
            headers={
                'Authorization': 'token {}'.format(str(token))
            },
            json={
                'context': context,
                'target_url': 'https://sandbox.yandex-team.ru/task/{}/view'.format(str(self.id)),
                'title': 'Tanker translates update',
                'head': branch_name,
                'base': 'dev'
            }
        )


__TASK__ = CollectionsFrontendTankerDownload
