from datetime import datetime

import requests
from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import task as ctt
from sandbox.projects.yql.RunYQL2 import RunYQL2
from sandbox.sandboxsdk import environments
from urlparse import urljoin


class CollectionsImportMarketFavorites(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('yandex-yt'),
        )

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('API') as api_parameters:
            api_url = sdk2.parameters.String('URL', required=True)
            api_token = sdk2.parameters.YavSecret('OAuth token', required=True)

        with sdk2.parameters.Group('Tables') as table_parameters:
            cluster = sdk2.parameters.String('Cluster', required=True)
            import_table_dir = sdk2.parameters.String('Import table dir', required=True)
            board_table = sdk2.parameters.String('Board table', required=True)
            user_table = sdk2.parameters.String('User table', required=True)
            card_table = sdk2.parameters.String('Card table', required=True)
            market_users_table = sdk2.parameters.String('Market users table', required=True)
            market_items_table = sdk2.parameters.String('Market items table', required=True)

        with sdk2.parameters.Group('Market URL\'s') as market_urls_parameters:
            pokupki_product_url_prefix = sdk2.parameters.String('Pokupki product URL prefix', required=True)
            market_product_url_prefix = sdk2.parameters.String('Market product URL prefix', required=True)

    def on_execute(self):
        with self.memoize_stage.enqueue_create_table:
            self._enqueue_create_table()
        with self.memoize_stage.check_create_table:
            self._check_create_table()
        with self.memoize_stage.enqueue_import_table:
            self._enqueue_import_table()

    def _enqueue_create_table(self):
        from yt.wrapper import ypath_join

        self.Context.import_table = ypath_join(self.Parameters.import_table_dir, datetime.now().isoformat())
        create_table_task = RunYQL2(
            self,
            query=self._get_create_table_query(),
            custom_placeholders={
                '%CLUSTER%': self.Parameters.cluster,
                '%IMPORT_TABLE%': self.Context.import_table,
                '%BOARD_TABLE%': self.Parameters.board_table,
                '%USER_TABLE%': self.Parameters.user_table,
                '%CARD_TABLE%': self.Parameters.card_table,
                '%MARKET_USERS_TABLE%': self.Parameters.market_users_table,
                '%MARKET_ITEMS_TABLE%': self.Parameters.market_items_table,
                '%POKUPKI_PRODUCT_URL_PREFIX%': self.Parameters.pokupki_product_url_prefix,
                '%MARKET_PRODUCT_URL_PREFIX%': self.Parameters.market_product_url_prefix,
            },
            use_v1_syntax=True,
            trace_query=True,
            retry_period=60
        )
        create_table_task.enqueue()
        self.Context.create_table_task_id = create_table_task.id
        self.Context.save()
        raise sdk2.WaitTask([self.Context.create_table_task_id], ctt.Status.Group.FINISH + ctt.Status.Group.BREAK)

    def _check_create_table(self):
        create_table_task = sdk2.Task[self.Context.create_table_task_id]
        if create_table_task.status != ctt.Status.SUCCESS:
            raise errors.TaskFailure('Subtask failed')

    def _enqueue_import_table(self):
        api_token = self.Parameters.api_token.data()[self.Parameters.api_token.default_key]
        resp = requests.post(urljoin(self.Parameters.api_url, '/api/import_market_favorites'), json={
            'yt_cluster': self.Parameters.cluster,
            'yt_table': self.Context.import_table
        }, headers={
            'Authorization': 'OAuth {}'.format(api_token),
            'Content-Type': 'application/json'
        })
        resp.raise_for_status()

    def _get_create_table_query(self):
        return """
use %CLUSTER%;

$import_table = "%IMPORT_TABLE%";
$board_table = "%BOARD_TABLE%";
$user_table = "%USER_TABLE%";
$card_table = "%CARD_TABLE%";
$market_users_table = "%MARKET_USERS_TABLE%";
$market_items_table = "%MARKET_ITEMS_TABLE%";
$pokupki_product_url_prefix = "%POKUPKI_PRODUCT_URL_PREFIX%";
$market_product_url_prefix = "%MARKET_PRODUCT_URL_PREFIX%";

$boards = (
    select
        id as id
    from $board_table
    where Yson::YPathString(document, '/slug') = 'my_goods'
);
$users = (
    select
        id as id,
        cast(Yson::YPathString(document, '/uid/_$numberLong') as Int32?) as uid
    from $user_table
    where Yson::YPathString(document, '/uid/_$numberLong') is not null
);
$cards = (
    select
        id as id,
        owner as owner,
        board as board,
        Yson::YPathString(document, '/source_meta/page_url') as page_url
    from $card_table
);

$user_cards = (
    select
        c.owner as owner,
        c.id as id,
        c.page_url as page_url,
        u.uid as uid
    from $cards as c
    join $boards as b
        on c.board = b.id
    join $users as u
        on c.owner = u.id
);
$market_favorites = (
    select
        u.p_uid as p_uid,
        bi.id as id,
        bi.image as image,
        bi.title as title,
        bi.price_amount as price_amount,
        bi.price_currency as price_currency,
        if(bi.reference_type = 0, $pokupki_product_url_prefix || bi.reference_id, $market_product_url_prefix || bi.reference_id) as page_url
    from $market_users_table as u
    join $market_items_table as bi
        on u.id = bi.owner_id
    where
        u.p_uid is not null
        and bi.image != ""
        and bi.price_currency = 0
        and (bi.reference_type = 0 or bi.reference_type = 2 or bi.reference_type = 3)
);

insert into $import_table
select
    mf.p_uid as p_uid,
    mf.id as id,
    mf.image as image,
    mf.title as title,
    mf.price_amount as price_amount,
    mf.page_url as page_url,
    u.id as user_id
from $market_favorites as mf
left join $users as u
    on mf.p_uid = u.uid
left join $user_cards as uc
    on mf.p_uid = uc.uid and mf.page_url = uc.page_url
where uc.page_url is null
        """
