import datetime
import os
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment


YQL_QUERY = '''
DECLARE $ballot AS String;
DECLARE $vote AS String;
DECLARE $output AS String;

$user_ballot_table = (
SELECT
    Yson::ConvertToString(document['user']['_$oid']) AS user_id,
    Yson::ConvertToString(document['ballot']['_$oid']) AS ballot_id
FROM $vote
);

$ballot_competition_table = (
SELECT 
    id as ballot_id,
    Yson::ConvertToString(document['competition']['_$oid']) AS competition_id
FROM $ballot
);

INSERT INTO $output WITH TRUNCATE
SELECT DISTINCT
    user_id, competition_id
FROM (
    SELECT
        user_id,
        competition_id
    FROM $user_ballot_table AS v
    INNER JOIN $ballot_competition_table AS b
    ON v.ballot_id == b.ballot_id
);

'''


class CollectionsExtractCompetitionVoters(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yql'),
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    class Parameters(sdk2.Task.Parameters):
        yql_token_secret = sdk2.parameters.String('Yql token secret')
        yt_cluster = sdk2.parameters.String('Yt cluster (proxy)')
        backup_dir = sdk2.parameters.String('Dir with votes and ballot in dump')
        output_path = sdk2.parameters.String('Output path')

    def get_yql_client(self):
        from yql.api.v1.client import YqlClient
        from yql.config import config
        yql_token = sdk2.Vault.data(self.Parameters.yql_token_secret)
        yql_client = YqlClient(db=self.Parameters.yt_cluster, token=yql_token)
        config.no_color = True
        config.is_tty = True
        return yql_client

    def on_execute(self):
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder

        vote_path = os.path.join(self.Parameters.backup_dir, 'vote')
        ballot_path = os.path.join(self.Parameters.backup_dir, 'ballot')

        parameters = {
            '$vote': ValueBuilder.make_string(vote_path),
            '$ballot': ValueBuilder.make_string(ballot_path),
            '$output': ValueBuilder.make_string(self.Parameters.output_path),
        }

        yql_client = self.get_yql_client()
        request = yql_client.query(query=YQL_QUERY, syntax_version=1)
        request.run(parameters=ValueBuilder.build_json_map(parameters))
        _ = request.get_results()
