# -*- encoding: utf-8 -*-
import json
from datetime import datetime, timedelta

import requests
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.collections.GeneratePushNotifications import CollectionsGeneratePushNotifications, \
    GeneralParameters
from sandbox.sandboxsdk.environments import PipEnvironment
from six.moves.urllib import parse as urlparse


class CollectionsGenerateOnCompetitionStartPushNotifications(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yandex-yt'),
            PipEnvironment("yandex-yt-transfer-manager-client"),
        )
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(GeneralParameters):
        with sdk2.parameters.Group('API parameters') as api_parameters:
            api_url = sdk2.parameters.String(
                'API url',
                required=True
            )
            api_token_vault = sdk2.parameters.String(
                'API token vault',
                required=True
            )
            max_competition_age = sdk2.parameters.Float(
                'Max competition age in hours',
                required=True,
                default=24
            )
            ignore_can_push_flag = sdk2.parameters.Bool(
                'Ignore can_push_on_start flag of competition',
                default=False
            )

        with sdk2.parameters.Group('Dump parameters') as dump_parameters:
            dump_path = sdk2.parameters.String(
                'Dump path',
                required=True
            )
            dump_state = sdk2.parameters.String(
                'Dump state',
                required=True
            )

    def on_execute(self):
        api_url = self.Parameters.api_url
        headers = {}
        api_token = sdk2.Vault.data(self.owner, self.Parameters.api_token_vault)
        headers['AUTHORIZATION'] = 'OAuth {}'.format(api_token)

        if not self.Context.generate_task_id:
            resp = requests.get(urlparse.urljoin(api_url, '/api/competitions/last'), headers=headers)
            resp.raise_for_status()
            competition = resp.json()

            if (
                not self.Parameters.ignore_can_push_flag
                and not competition.get('competition_info', {}).get('can_push_on_start', False)
            ):
                return

            competition_start = datetime.strptime(competition['competition_info']['start'][:19], '%Y-%m-%dT%H:%M:%S')

            if competition_start < datetime.utcnow() - timedelta(hours=self.Parameters.max_competition_age):
                return

            competition_end = datetime.strptime(competition['competition_info']['end'][:10], '%Y-%m-%d')
            competition_end = competition_end.strftime('%Y-%m-%d')

            task = CollectionsGeneratePushNotifications(
                self,
                description=self.Parameters.description,
                push_type='competition-start',
                command_args=[
                    '--dump-path', self.Parameters.dump_path,
                    '--dump-state', self.Parameters.dump_state,
                    '--competition-name', competition['title'],
                    '--competition-evaluation', competition_end,
                    '--competition-link', competition['link']
                ],
                **{
                    param: getattr(self.Parameters, param)
                    for param in [
                        'yt_proxy', 'yt_token_vault', 'tables_to_save', 'sup_url',
                        'sup_token_vault', 'dry_run', 'sup_additional_parameters',
                        'push_table_dir'
                    ]
                }
            )
            task.enqueue()
            self.Context.generate_task_id = task.id
            self.Context.competition_id = competition['id']
            self.Context.save()

            raise sdk2.WaitTask([self.Context.generate_task_id], (ctt.Status.Group.FINISH + ctt.Status.Group.BREAK))

        task = sdk2.Task[self.Context.generate_task_id]

        if task.status != ctt.Status.SUCCESS:
            raise TaskFailure('Subtask failed')

        if not self.Parameters.dry_run:
            resp = requests.patch(urlparse.urljoin(api_url, '/api/boards/{}'.format(self.Context.competition_id)), json={
                'competition_info': {
                    'can_push_on_start': False
                }
            }, headers=headers)
            resp.raise_for_status()


__TASK__ = CollectionsGenerateOnCompetitionStartPushNotifications
