# -*- encoding: utf-8 -*-

import json
import logging
import os
import shlex
from datetime import datetime

import requests
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import utils
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.helpers import subprocess, ProcessRegistry, ProcessLog

from sandbox.projects.collections.mixins import YasmReportable
from sandbox.projects.collections.resources import CollectionsPushesBinary

SUP_PROXY = 'hahn'


class GeneralParameters(sdk2.Task.Parameters):
    dry_run = sdk2.parameters.Bool('Dry run', default=True)

    with sdk2.parameters.Group('YT parameters') as yt_parameters:
        yt_proxy = sdk2.parameters.String(
            'YT proxy',
            required=True
        )
        yt_token_vault = sdk2.parameters.String(
            'YT token vault',
            required=True,
            default='yt_token'
        )
        tables_to_save = sdk2.parameters.Integer(
            'How many tables to save',
            required=False,
            default=366
        )

    with sdk2.parameters.Group('SUP parameters') as sup_parameters:
        sup_url = sdk2.parameters.String(
            'SUP url',
            required=True
        )
        sup_token_vault = sdk2.parameters.String(
            'SUP token vault',
            required=True
        )
        sup_additional_parameters = sdk2.parameters.String(
            'SUP additional parameters',
            default=''
        )

    with sdk2.parameters.Group('Parameters for generating push tables') as push_parameters:
        push_table_dir = sdk2.parameters.String(
            'Push tables directory',
            required=True
        )


class CollectionsGeneratePushNotifications(YasmReportable, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yandex-yt'),
            PipEnvironment("yandex-yt-transfer-manager-client"),
        )
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(GeneralParameters):
        with sdk2.parameters.Group('Parameters for generating push tables') as push_parameters:
            push_type = sdk2.parameters.String(
                'Type of push notifications',
                required=True
            )
            command_line = sdk2.parameters.String(
                'Remaining compilation command to generate pushes',
                required=False
            )
            command_args = sdk2.parameters.List(
                'Remaining compilation command args to generate pushes',
                required=False
            )
            input_journal_table = sdk2.parameters.String(
                'Input journal table path',
                required=False
            )
        with sdk2.parameters.Group('Monitoring parameters') as monitoring_parameters:
            monitoring_server_host = sdk2.parameters.String(
                'Monitoring server',
                default='monit.n.yandex-team.ru',
            )

    def on_execute(self):
        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_vault)
        sup_token = sdk2.Vault.data(self.owner, self.Parameters.sup_token_vault)

        tool_id = utils.get_and_check_last_released_resource_id(CollectionsPushesBinary)
        tool_path = str(sdk2.ResourceData(sdk2.Resource[tool_id]).path)

        if self.Parameters.command_line and self.Parameters.command_args:
            raise TaskFailure('Must be passed only command_line or command_args.')

        args = tool_path + ' ' + self.Parameters.push_type
        args += ' --yt-proxy ' + self.Parameters.yt_proxy
        from yt.wrapper import ypath_join
        output_table_path = ypath_join(self.Parameters.push_table_dir, datetime.today().strftime("%Y-%m-%dT%H:%M:%S.%f"))
        args += ' --output-table-path ' + output_table_path

        from yt.wrapper import YtClient

        yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)
        if self.Parameters.yt_proxy != SUP_PROXY:
            sup_yt_client = YtClient(SUP_PROXY, token=yt_token)

        if self.Parameters.input_journal_table:
            output_journal_table = yt_client.create_temp_table(prefix='journal_table.')
            args += ' --input-journal-table ' + self.Parameters.input_journal_table
            args += ' --output-journal-table ' +  output_journal_table

        if self.Parameters.command_line:
            args += ' ' + self.Parameters.command_line

        args = shlex.split(args)
        if self.Parameters.command_args:
            args.extend(self.Parameters.command_args)
        logging.info(args)
        with ProcessLog(self, logger='push_generator') as log, ProcessRegistry:
            env = os.environ.copy()
            env['YT_TOKEN'] = yt_token
            subprocess.check_call(args, stdout=log.stdout, stderr=log.stderr, env=env)
        if self.Parameters.yt_proxy != SUP_PROXY:
            from yt.transfer_manager.client import TransferManager

            transfer_client = TransferManager(token=yt_token)
            transfer_client.add_task(source_cluster=self.Parameters.yt_proxy,
                                     source_table=output_table_path,
                                     destination_cluster=SUP_PROXY,
                                     destination_table=output_table_path,
                                     sync=True)

        sup_url = self.Parameters.sup_url + 'pushes/batch'
        if self.Parameters.sup_additional_parameters:
            sup_url += '?' + self.Parameters.sup_additional_parameters
        if self.Parameters.dry_run:
            self.set_info('Mock POST to {url} with {data}'.format(
                url=sup_url,
                data=json.dumps({'path': output_table_path})
            ))
            return

        response = requests.post(
            sup_url,
            data=json.dumps({'path': output_table_path}),
            headers={
                'Authorization': 'OAuth {}'.format(sup_token),
                'Content-Type': 'application/json;charset=UTF-8'
            }
        )
        if response.status_code < 500 or response.status_code > 600:
            response.raise_for_status()
        self.set_info(json.dumps(response.json()))

        if self.Parameters.input_journal_table:
            yt_client.move(output_journal_table, self.Parameters.input_journal_table, force=True)

        if getattr(self.Parameters, "tables_to_save"):
            self._clean_old_tables(yt_client)
            if self.Parameters.yt_proxy != SUP_PROXY:
                self._clean_old_tables(sup_yt_client)

        self._yasm_report(
            args=[
                "--id", "collections_{}_pushes_count".format(self.Parameters.push_type),
                "--value", str(yt_client.get_attribute(output_table_path, 'row_count')),
                "--policy", "abs"
            ]
        )

        self._report_lag("collections_{}_pushes_sent".format(self.Parameters.push_type))

    def _clean_old_tables(self, yt_client):
        lst = yt_client.list(self.Parameters.push_table_dir, absolute=True, sort=True)
        states = [table for table in lst if yt_client.get(table + "/@type") == "table"]
        states_to_remove = states[:-self.Parameters.tables_to_save]
        for state in states_to_remove:
            yt_client.remove(state)


__TASK__ = CollectionsGeneratePushNotifications
