from __future__ import absolute_import

import json
import logging
import os
import random
from sandbox import sdk2
import time
from Cookie import BaseCookie
from sandbox.common import errors
from sandbox.common.types.task import Status
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2 import parameters
from sandbox.projects.collections import resources
from sandbox.projects.tank.load_resources.resources import STRESS_SESSION_IDS

LABEL = 'Collections session ids ({ts})'
REDUCE_FACTOR = 10000
REQUESTS_TABLENAME_TMPL = 'collections_load_requests_{}'
RESULT_FILENAME = 'ammos.txt'


def make_ammo(method, url, headers, case, body):
    """Copied from yandex.tank docs"""
    # http request w/o entity body template
    req_template = (
          "%s %s HTTP/1.1\r\n"
          "%s\r\n"
          "\r\n"
    )

    # http request with entity body template
    req_template_w_entity_body = (
          "%s %s HTTP/1.1\r\n"
          "%s\r\n"
          "Content-Length: %d\r\n"
          "\r\n"
          "%s\r\n"
    )

    if not body:
        req = req_template % (method, url, headers)
    else:
        req = req_template_w_entity_body % (method, url, headers, len(body), body)

    # phantom ammo template
    ammo_template = (
        "%d %s\n"
        "%s"
    )

    return ammo_template % (len(req), case, req)


class CollectionsGenerateTankAmmos(sdk2.Task):

    class Parameters(sdk2.Parameters):
        yt_requests_path = parameters.String(label='Path to Requests on YT')
        yt_proxy = parameters.String(label='YT Proxy')
        yt_token = parameters.String(label='YT Token Secret')
        yt_directory = parameters.String(label='Working Directory on YT')

        tvm2_env_container = parameters.Integer(
            label='tvm2 environment container', default=513532842
        )
        blackbox_host = parameters.String(
            label='Blackbox to generate sessions for',
            default='blackbox-stress.yandex.net',
        )
        use_default_tvm = parameters.Bool(label='Use default TVM', default=True)
        with use_default_tvm.value[False]:
            tvm_secret_owner = parameters.String(label='TVM2 Secret owner (task owner by default)')
            tvm_client_id = parameters.String(label='Client ID for TVM2')
            tvm_secret = parameters.String(label='TVM2 Secret (Vault Key)')

    class Requirements(sdk2.Requirements):
        environments = (
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
            PipEnvironment('pymongo'),
        )

    def get_yt_client(self):
        from yt.wrapper.client import YtClient

        yt_token = sdk2.Vault.data(self.Parameters.yt_token)
        client = YtClient(
            proxy=self.Parameters.yt_proxy,
            token=yt_token,
        )
        return client

    def _check_task(self, task_id, valid_statuses):
        """
        :type task: int
        :type valid_statuses: list
        """
        task = sdk2.Task[task_id]
        if task.status not in valid_statuses:
            raise errors.TaskFailure(
                'Subtask is not OK: {task_name}::{id}::{status}'.format(
                    task_name=type(task).name,
                    id=task.id,
                    status=task.status,
                )
            )

    def _wait_task(self, executing_tasks):
        """
        :type task: int | list[int]
        """
        memoize_name = (
            ','.join([str(i) for i in executing_tasks])
            if isinstance(executing_tasks, list) else
            str(executing_tasks)
        )
        with self.memoize_stage[memoize_name]:
            raise sdk2.WaitTask(
                executing_tasks,
                list(Status.Group.FINISH + Status.Group.BREAK),
                wait_all=True,
            )

    def _get_yandex_uid_from_cookie(self, cookie):
        cookie_obj = BaseCookie()
        cookie_obj.load(cookie)
        yandex_uid = cookie_obj.get('yandexuid')
        if yandex_uid:
            yandex_uid = yandex_uid.value
        return yandex_uid

    def filter_queries(self):
        if self.Context.filtered_requests_path:
            return self.Context.filtered_requests_path

        logging.info('Filter requests for loadtest')
        client = self.get_yt_client()

        shift = random.randrange(0, REDUCE_FACTOR)
        rows = client.read_table(self.Parameters.yt_requests_path, raw=False)
        data_gen = (
            row
            for index, row in enumerate(rows)
            if index % REDUCE_FACTOR == shift and row['method'] == 'GET'
        )

        tablename = os.path.join(
            self.Parameters.yt_directory,
            REQUESTS_TABLENAME_TMPL.format(time.time()),
        )
        client.write_table(tablename, data_gen)

        self.Context.filtered_requests_path = tablename
        self.Context.save()
        logging.info('Requests for loadtest filtered')
        return tablename

    def generate_session_ids(self, requests_table):
        if self.Context.generate_sessions_task_id:
            return self.Context.generate_sessions_task_id

        logging.info('Generating sessions for stressing')

        client = self.get_yt_client()
        rows = client.read_table(requests_table)
        uids = []
        for row in rows:
            yandex_uid = self._get_yandex_uid_from_cookie(row['cookies'])
            if yandex_uid:
                uids.append(yandex_uid)

        task_params = {
            '_container': self.Parameters.tvm2_env_container,
            'blackbox_host': self.Parameters.blackbox_host,
            'uids': '\n'.join(uids),
            'result_label': 'sessions for collections loadtest {}'.format(time.time()),
        }

        if not self.Parameters.use_default_tvm:
            task_params.update({
                'tvm_client_id': self.Parameters.tvm_client_id,
                'tvm_secret_token': self.Parameters.tvm_secret,
                'vault_owner': self.Parameters.tvm_secret_owner,
            })
        else:
            task_params['vault_owner'] = 'LOAD'

        task_cls = sdk2.Task['GENERATE_STRESS_SESSION_IDS']
        task = task_cls(
            self,
            description='sessions for collections loadtest',
            **task_params
        )
        self.Context.generate_sessions_task_id = task.enqueue().id
        self.Context.save()
        logging.info('Started generation sessions')
        return self.Context.generate_sessions_task_id

    def create_ammos(self, task_id, requests_table):
        resource = sdk2.Resource.find(
            STRESS_SESSION_IDS,
            task_id=task_id,
        ).first()
        sessions_path = str(sdk2.ResourceData(resource).path)

        sessions = {}
        with open(sessions_path, 'r') as sessions_file:
            for line in sessions_file.readlines():
                session_json = json.loads(line)
                default_uid = session_json['default_uid']
                sessions[default_uid] = session_json['new-session']['value']

        client = self.get_yt_client()
        rows = client.read_table(requests_table)
        with open(RESULT_FILENAME, 'w') as result_f:
            for row in rows:
                headers = [
                    'Host: yandex.ru',
                    'Connection: close',
                    'Accept-Encoding: gzip, deflate',
                ]

                yandex_uid = self._get_yandex_uid_from_cookie(row['cookies'])
                if yandex_uid:
                    session = sessions[yandex_uid]
                    cookie = BaseCookie()
                    cookie['Session_id'] = session
                    headers.append(cookie.output()[4:])

                ammo = make_ammo(
                    row['method'], row['request'], '\r\n'.join(headers), '', ''
                )
                result_f.write(ammo)

        resource_metadata = resources.CollectionsBackendTankAmmos(
            self,
            path=RESULT_FILENAME,
            description='Collections Tank Ammos',
        )
        resource_data = sdk2.ResourceData(resource_metadata)
        resource_data.ready()

    def on_execute(self):
        requests_table = self.filter_queries()
        task_id = self.generate_session_ids(requests_table)
        self._wait_task(task_id)
        self._check_task(task_id, [Status.SUCCESS])

        self.create_ammos(task_id, requests_table)
        yt_client = self.get_yt_client()
        yt_client.remove(requests_table)
