import datetime
import logging
from sandbox import sdk2
from sandbox.projects.collections.ImportPictureOfTheDay.settings import (
    OAUTH_TOKEN_VAULT_KEY
)
from sandbox.projects.collections.ImportPictureOfTheDay.utils import (
    download_export,
    upload_images,
    get_importing_images,
)


class CollectionsImportPictureOfTheDayTask(sdk2.Task):
    """Importing pictures of the day to given collection.
    """

    class Parameters(sdk2.Task.Parameters):
        api_url = sdk2.parameters.String(
            'Collections API URL.',
            required=True,
        )
        board_id = sdk2.parameters.String(
            'Collections picture of the day board ID.',
            required=True,
        )
        import_today = sdk2.parameters.Bool(
            "Import picture of task launch day.",
            default=True,
            required=False,
        )
        with import_today.value[False]:
            import_from = sdk2.parameters.String(
                'First date for importing pictures. (format: YYYY-MM-DD)',
                default=datetime.date.today().isoformat(),
                required=False,
            )

    def on_execute(self):
        today = datetime.date.today()
        logging.info(
            'Started import picture of the day {}'.format(today.isoformat())
        )

        content = download_export()

        if self.Parameters.import_today:
            first_date = datetime.date.today()
        else:
            first_date = datetime.datetime.strptime(
                self.Parameters.import_from,
                '%Y-%m-%d',
            ).date()
        last_date = datetime.date.today()
        importing_images = get_importing_images(content, first_date, last_date)

        token = sdk2.Vault.data(OAUTH_TOKEN_VAULT_KEY)
        upload_images(
            content,
            importing_images,
            self.Parameters.api_url,
            self.Parameters.board_id,
            token,
        )

        logging.info(
            'Finished import picture of the day {}'.format(today.isoformat())
        )
