# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import json
import logging
import requests
import urllib
import urlparse
import uuid

from sandbox.projects.collections.ImportPictureOfTheDay.settings import (
    EXPORT_URL,
    RESTRICTED_PARTNERS,
    REQUIRED_SIZE,
    PARTNERS_MAPPING,
)


def download_export():
    """
    Download export json and parse it.
    """
    logging.info('Downloading export json')
    response = urllib.urlopen(EXPORT_URL)
    result = json.loads(response.read())
    return result


def get_importing_images(parsed_export, first_date, last_date):
    """
    Getting order of images to be uploaded for cold start.
    """
    logging.info('Getting order of pictures for uploading')
    fmt_first_date = first_date.isoformat()
    fmt_last_date = last_date.isoformat()

    pub_dates = parsed_export['regular']['ru']
    ordered_images = sorted(
        pub_dates,
        key=lambda item: item['publication_date'],
    )
    result = [
        (image['image_id'], image['publication_date'])
        for image in ordered_images
        if fmt_first_date <= image['publication_date'] <= fmt_last_date
    ]
    return result


def collect_image_info(export_image):
    """
    Upload single image to collection into settings.
    """
    logging.info('Collecting description of the picture')
    image_props = export_image['l10n'].get('ru', {})
    author = image_props.get('author', '')
    title = image_props.get('title', '')
    author_url = export_image.get('author_url', '')
    description = image_props.get('picture_description', '')

    partner_code = export_image.get('partner', '')
    partner_name = PARTNERS_MAPPING.get(partner_code)

    if not partner_name:
        logging.warning(
            'Cant find partner by its code: {}'.format(partner_code)
        )
        return {}

    link = export_image['urls']['show'][REQUIRED_SIZE]
    return {
        'author': author,
        'title': title,
        'description': description,
        'link': link,
        'author_url': author_url,
        'partner': partner_name,
    }


def _upload_image(session, api_url, board_id, token, image_info):
    """
    Upload image to the api using given data.
    """
    logging.info('Uploading image to the collection')
    uploading_url = urlparse.urljoin(api_url, 'content')

    link = urlparse.urljoin('http:', image_info['link'])

    headers = {
        'Authorization': 'OAuth {}'.format(token),
        'Host': 'collections.yandex.ru',
        'Content-Type': 'application/json',
        'X-CSRFToken': session.cookies.get('csrftoken'),
    }
    data = {
        'source_type': 'image',
        'source': {
            'title': image_info.get('title', ''),
            'url': link,
        },
    }

    response = session.post(
        uploading_url,
        json=data,
        headers=headers,
    )
    if not response.ok:
        logging.warning(
            'Picture has not been uploaded. Status code: {}. {} {}'.format(
                response.status_code,
                link,
                response.text,
            )
        )
        return False

    parsed_response = json.loads(response.text)
    picture_id = parsed_response['id']

    creating_url = urlparse.urljoin(api_url, 'cards')
    author_url = urlparse.urlparse(image_info['author_url'])
    extended_description = (
        '{title}, {author} для «{partner}».'
        '\n\n{description}'
    ).format(
        title=image_info['title'],
        author=image_info['author'],
        partner=image_info['partner'],
        description=image_info['description'],
    )
    data = {
        'board_id': board_id,
        'content': [
            {
                'id': picture_id,
            },
        ],
        'description': extended_description,
        'source_meta': {
            'page_domain': author_url.netloc,
            'page_url': author_url.geturl(),
        },
    }
    response = session.post(creating_url, json=data, headers=headers)
    if not response.ok:
        logging.warning(
            'Card has not been created {}. Status code {}. {} {}'.format(
                picture_id,
                response.status_code,
                link,
                response.text,
            )
        )
        return False
    return True


def upload_images(
        export_data,
        importing_images,
        api_url,
        board_id,
        token,
):
    """
    Preprocess and upload images in order given in the list of image_ids.
    """
    logging.info('Preprocessing and uploading images to collection')
    with requests.Session() as session:
        session.cookies.set('csrftoken', str(uuid.uuid4()))

        for image_id, publication_date in importing_images:
            image = export_data['images'][str(image_id)]
            if image['partner'] in RESTRICTED_PARTNERS:
                continue

            image_info = collect_image_info(image)
            if not image_info:
                raise RuntimeError(
                    'Import failed. Failed at: {}'.format(publication_date)
                )

            result = _upload_image(
                session,
                api_url,
                board_id,
                token,
                image_info,
            )
            if not result:
                raise RuntimeError(
                    'Import failed. Failed at: {}'.format(publication_date)
                )
