# -*- encoding: utf-8 -*-
from __future__ import absolute_import

import logging
import Queue
import requests
from sandbox import sdk2
import threading
from sandbox.projects.collections.mixins import YasmReportable


CHANNELS_LIST_URL = "https://{domain}/api/channels/?page_size=100&page={page}"
CHANNEL_INFO_URL = "https://{domain}/api/cards/channels/{slug}/?page_size=30"

# Metrics
TOO_SMALL_CARDS_COUNT_METRIC_NAME = "collections-too-small-cards-count"
CHANNEL_REQUEST_TIMEOUT_METRIC_NAME = "collections-channel-request-timeouts-count"
CHANNEL_REQUEST_HTTP_ERRORS_METRIC_NAME = "collections-channel-request-http-error-count"
CHANNEL_REQUEST_UNKNOWN_ERRORS_METRIC_NAME = "collections-channel-request-unknown-error-count"
TOTAL_BAD_CHANNELS_COUNT_METRIC_NAME = "collections-bad-channels-count"
TASK_LAUNCH_FREQUENCY_METRIC_NAME = "collections-monitoring-task-launch-timelag"


class CollectionsMonitorChannelsAvailability(sdk2.Task, YasmReportable):
    """
        Task for Yandex.Collections channels availability monitoring.
    """

    class Parameters(sdk2.Task.Parameters):
        channels_list_domain = sdk2.parameters.String(
            "Domain for channels list retrieving",
            default="pdb-admin.n.yandex-team.ru",
        )
        channel_info_domain = sdk2.parameters.String(
            "Domain for channel info requests",
            default="collections.yandex.ru",
        )
        channels_list_request_retry_count = sdk2.parameters.Integer(
            "Channels list request retry count", default=5,
        )
        channels_list_request_timeout = sdk2.parameters.Float(
            "Channels list request timeout (in seconds)", default=10,
        )
        channel_info_request_retry_count = sdk2.parameters.Integer(
            "Channel info request retry count", default=1,
        )
        channel_info_request_timeout = sdk2.parameters.Float(
            "Channel info request timeout (in seconds)", default=1,
        )
        requests_threads_count = sdk2.parameters.Integer(
            "Threads count for requests", default=5,
        )
        min_cards_in_channel_count = sdk2.parameters.Integer(
            "Minimum cards in channel count", default=15,
        )
        monitoring_server_host = sdk2.parameters.String(
            'Monitoring server',
            default='monit.n.yandex-team.ru',
        )

    class Context(sdk2.Task.Context):
        pass

    metrics = (
        TOO_SMALL_CARDS_COUNT_METRIC_NAME,
        CHANNEL_REQUEST_TIMEOUT_METRIC_NAME,
        CHANNEL_REQUEST_HTTP_ERRORS_METRIC_NAME,
        CHANNEL_REQUEST_UNKNOWN_ERRORS_METRIC_NAME,
        TOTAL_BAD_CHANNELS_COUNT_METRIC_NAME,
    )

    def on_execute(self):
        input_channels = Queue.Queue()
        for channel in self._get_channels():
            input_channels.put(channel)

        bad_channels = Queue.Queue()
        thread_lock = threading.Lock()

        for metric in CollectionsMonitorChannelsAvailability.metrics:
            setattr(self.Context, metric, 0)
            self.Context.save()

        retry_count = self.Parameters.channel_info_request_retry_count
        for i in range(self.Parameters.requests_threads_count):
            session = requests.Session()
            session.mount("http://", requests.adapters.HTTPAdapter(max_retries=retry_count))
            session.mount("https://", requests.adapters.HTTPAdapter(max_retries=retry_count))

            thread = threading.Thread(target=self._process_channel, args=(input_channels, bad_channels, session, thread_lock))
            thread.daemon = True
            thread.start()

        input_channels.join()

        setattr(self.Context, TOTAL_BAD_CHANNELS_COUNT_METRIC_NAME, bad_channels.qsize())
        self.Context.save()
        for metric in CollectionsMonitorChannelsAvailability.metrics:
            self._yasm_report(
                args=[
                    "--id", metric,
                    "--value", getattr(self.Context, metric),
                    "--policy", "abs"
                ]
            )

        # Monitor task launch frequency
        self._report_lag(TASK_LAUNCH_FREQUENCY_METRIC_NAME)

    def _process_channel(self, channels_queue, bad_channels, session, thread_lock):
        min_cards_count = self.Parameters.min_cards_in_channel_count
        while True:
            try:
                channel = channels_queue.get_nowait()

                cards_count = self._get_channel_cards_count(channel, bad_channels, session, thread_lock)
                if cards_count is not None and cards_count < min_cards_count:
                    logging.debug("Channel '%s', cards count: %d", channel, cards_count)
                    self._process_bad_channel(channel, bad_channels, thread_lock, TOO_SMALL_CARDS_COUNT_METRIC_NAME)

                channels_queue.task_done()
            except Queue.Empty:
                break

    def _get_channel_cards_count(self, channel, bad_channels, session, thread_lock):
        try:
            url = CHANNEL_INFO_URL.format(domain=self.Parameters.channel_info_domain, slug=channel)

            request = session.get(url, timeout=self.Parameters.channel_info_request_timeout)
            request.raise_for_status()

            return len(request.json()["results"])
        except requests.Timeout:
            self._process_bad_channel(channel, bad_channels, thread_lock, CHANNEL_REQUEST_TIMEOUT_METRIC_NAME)
        except requests.HTTPError:
            self._process_bad_channel(channel, bad_channels, thread_lock, CHANNEL_REQUEST_HTTP_ERRORS_METRIC_NAME)
        except Exception:
            self._process_bad_channel(channel, bad_channels, thread_lock, CHANNEL_REQUEST_UNKNOWN_ERRORS_METRIC_NAME)
        finally:
            return None

    def _process_bad_channel(self, channel, bad_channels, thread_lock, error_type):
        bad_channels.put(channel)

        with thread_lock:
            old_value = getattr(self.Context, error_type)
            setattr(self.Context, error_type, old_value + 1)
            self.Context.save()

    def _get_channels(self):
        retry_count = self.Parameters.channels_list_request_retry_count

        session = requests.Session()
        session.mount("http://", requests.adapters.HTTPAdapter(max_retries=retry_count))
        session.mount("https://", requests.adapters.HTTPAdapter(max_retries=retry_count))

        page_num = 1
        while True:
            url = CHANNELS_LIST_URL.format(domain=self.Parameters.channels_list_domain, page=page_num)
            page_num += 1

            request = session.get(url, timeout=self.Parameters.channels_list_request_timeout)
            request.raise_for_status()

            response = request.json()
            for channel in response["results"]:
                if not channel["informer_enabled"]:
                    continue

                yield channel["slug"]

            if response["next"] is None:
                break
