import datetime
import logging
import os
from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2 import parameters
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.common import utils
from sandbox.projects.common.apihelpers import get_last_resource_with_attrs
from sandbox.projects.collections import resources
from sandbox.projects.collections.LaunchComplaintsTasks import LaunchYasapPdbBackendComplaintsTasks
from sandbox.projects.collections.mixins import YasmReportable
from sandbox.projects.images.bans.resources import (
    IMAGES_MIDDLESEARCH_ANTIPIRATE_URLS_BAN_FILE,
)


class CollectionsPerformExternalBansTask(YasmReportable, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yandex-yt'),
        )
        cores = 1
        ram = 4096

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        use_latest = parameters.Bool('Use latest bans tries')
        with use_latest.value[False]:
            antipirate_trie = parameters.Resource(
                'Resource with antipirate tries',
                resource_type=IMAGES_MIDDLESEARCH_ANTIPIRATE_URLS_BAN_FILE,
            )

        ban_dir = parameters.String('YT directory with ban tables')
        unban_dir = parameters.String('YT directory with unban tables')

        ban_backup_table = parameters.String('YT directory with backup for bans')
        unban_backup_table = parameters.String('YT directory with backup for unbans')

        dump_dir = parameters.String('YT directory with PDB Dump')
        admin_dump_dir = parameters.String('YT directory with PDBAdmin Dump')

        yt_token_secret = parameters.String('YT token secret')
        yt_proxy = parameters.String('YT proxy (cluster)')
        url = parameters.String('Admin tasks endpoint')

        process_additional_parameters = parameters.List('Additional parameters for the executable file', default=[])

        with sdk2.parameters.Group('Monit parameters') as monit_parameters:
            monitoring_server_host = sdk2.parameters.String(
                'Monitoring server',
                default='monit.n.yandex-team.ru',
            )
            monitoring_signal_prefix = sdk2.parameters.String(
                'Monitoring signal prefix',
            )

    def load_resource(self, resource_id):
        """return path to file in resource
        """
        resource_data = sdk2.ResourceData(sdk2.Resource[resource_id])
        path = str(resource_data.path)
        return path

    def fetch_tries(self):
        if self.Context.tries:
            return self.Context.tries

        if self.Parameters.use_latest:
            antipirate_resource_id = get_last_resource_with_attrs(
                IMAGES_MIDDLESEARCH_ANTIPIRATE_URLS_BAN_FILE, None,
                params={
                    'owner': 'YASAP'
                },
            ).id
        else:
            antipirate_resource_id = self.Parameters.antipirate_trie

        antipirate_trie = self.load_resource(antipirate_resource_id)

        self.Context.tries = (antipirate_trie,)
        self.Context.save()

        return self.Context.tries

    def filter_items(self, antipirate_trie):
        if self.Context.result_tables:
            return self.Context.result_tables

        yt_token = sdk2.Vault.data(self.Parameters.yt_token_secret)
        executable_id = utils.get_and_check_last_released_resource_id(resources.CollectionsExternalBansUrlsFilter)
        executable = self.load_resource(executable_id)
        now_str = datetime.datetime.utcnow().isoformat()

        ban_table = os.path.join(self.Parameters.ban_dir, now_str)
        unban_table = os.path.join(self.Parameters.unban_dir, now_str)

        with sdk2.helpers.ProcessLog(self, logging.getLogger('map_reduce')) as pl:
            process = subprocess.Popen(
                [
                    executable,
                    '--yt-proxy', self.Parameters.yt_proxy,
                    '--pdb-dump-dir', self.Parameters.dump_dir,
                    '--admin-dump-dir', self.Parameters.admin_dump_dir,
                    '--ban-table', ban_table,
                    '--unban-table', unban_table,
                    '--tries', antipirate_trie,
                ] + self.Parameters.process_additional_parameters,
                env={
                    'YT_TOKEN': yt_token,
                },
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            process.wait()
            if process.returncode:
                raise Exception(
                    'MR filter returned non-zero code [{}]'.format(
                        process.returncode
                    )
                )

        if self.Parameters.monitoring_signal_prefix:
            from yt.wrapper import YtClient
            yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)
            for tag, table in [('ban', ban_table), ('unban', unban_table)]:
                self._yasm_report(
                    args=[
                        "--id", "{}-{}-count".format(self.Parameters.monitoring_signal_prefix, tag),
                        "--value", str(yt_client.get_attribute(table, 'row_count')),
                        "--policy", "abs"
                    ]
                )

        self.Context.result_tables = ban_table, unban_table
        self.Context.save()

        return self.Context.result_tables

    def run_task(self, table, backup_table, task_name):
        if not self.Context.subtasks:
            self.Context.subtasks = []

        subtask = (table, backup_table, task_name)
        subtasks = {tuple(item) for item in self.Context.subtasks}

        if subtask in subtasks:
            return

        task = LaunchYasapPdbBackendComplaintsTasks(
            task=task_name,
            base_url=self.Parameters.url,
            additional_params={
                'table': table,
                'backup_table': backup_table,
            },
            description='Launching table processing {}'.format(table),
            kill_timeout=30 * 60,
            max_restarts=0,
            parent=self,
        )
        task.save()
        task_id = task.enqueue().id

        self.Context.subtasks.append(subtask)
        self.Context.save()

        raise sdk2.WaitTask(
            [task_id],
            (ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
            wait_all=True
        )

    def on_execute(self):
        antipirate_trie, = self.fetch_tries()
        ban_table, unban_table = self.filter_items(antipirate_trie)
        self.run_task(ban_table, self.Parameters.ban_backup_table, 'perform_external_bans')
        self.run_task(unban_table, self.Parameters.unban_backup_table, 'perform_external_unbans')
        if self.Parameters.monitoring_signal_prefix:
            self._report_lag("{}-timelag".format(self.Parameters.monitoring_signal_prefix))
