import logging
import yaml
from datetime import datetime
from sandbox import sdk2
import sandbox.common.types.resource as ctr
from sandbox.projects.collections.mixins import YasmReportable, yt_path_to_yasm


LOGGER = logging.getLogger(__name__)
ATTRS = [
    'account',
    'recursive_resource_usage',
    'type',
    'path',
]


def tree_from_list(path_list):
    return '\t' + '\t'.join(path_list) + '\t'


def tree(path):
    return tree_from_list(path.strip('/').split('/'))


class StatRowMaker(object):
    def __init__(self, fielddate, cluster):
        self._fielddate = fielddate
        self._cluster = cluster

    def __call__(self, node):
        resources = node.attributes['recursive_resource_usage']
        return {
            'fielddate': self._fielddate,
            'cluster': self._cluster,
            'account': node.attributes['account'],
            'path': tree(str(node)),
            'disk_space': resources['disk_space'],
            'chunk_count': resources['chunk_count'],
            'node_count': resources['node_count'],
        }


class ReportYtDirStats(sdk2.Task, YasmReportable):
    """
    It makes reports about our YT quota stats
    """
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        monitoring_server_host = sdk2.parameters.String(
            'Monitoring server',
            default='monit.n.yandex-team.ru',
        )
        with sdk2.parameters.Group('YT parameters') as yt_parameters:
            yt_proxy = sdk2.parameters.String(
                'YT proxy',
                required=True,
            )
            yt_token_vault = sdk2.parameters.String(
                'YT token vault',
                required=True,
            )
            yt_path = sdk2.parameters.String(
                'Path in YT',
                required=True,
            )
        with sdk2.parameters.Group('Statface parameters') as statface_parameters:
            stat_report_path = sdk2.parameters.String(
                'Stat report path',
                required=True,
            )
            stat_report_title = sdk2.parameters.String(
                'Stat report title',
                required=True,
            )
            stat_token_vault = sdk2.parameters.String(
                'Stat token vault',
                required=True,
            )
        kill_timeout = 43200  # 12 hours
        UseLastBinary = sdk2.parameters.Bool('Use last binary archive', default=True)
        with UseLastBinary.value[True]:
            with sdk2.parameters.RadioGroup("Binary release type") as ReleaseType:
                ReleaseType.values.stable = ReleaseType.Value('stable', default=True)
                ReleaseType.values.test = ReleaseType.Value('test')
        with UseLastBinary.value[False]:
            custom_tasks_archive_resource = sdk2.parameters.Resource(
                'task archive resource',
                default=None, )

    def on_execute(self):
        from yt.wrapper import YtClient
        from statface_client import ProductionStatfaceClient
        stat_token = sdk2.Vault.data(self.owner, self.Parameters.stat_token_vault)
        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_vault)
        yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)
        sfclient = ProductionStatfaceClient(oauth_token=stat_token)

        nodes = yt_client.search(
            self.Parameters.yt_path,
            node_type='map_node',
            depth_bound=5,
            attributes=ATTRS,
        )
        row_maker = StatRowMaker(
            fielddate=datetime.now().replace(minute=0, second=0, microsecond=0).isoformat(),
            cluster=self.Parameters.yt_proxy,
        )
        data = (row_maker(node) for node in nodes)
        self._publish(sfclient, data, self.Parameters.stat_report_path, self.Parameters.stat_report_title)
        self._report_lag(
            self.make_signal_name(self.Parameters.yt_proxy, self.Parameters.yt_path)
        )

    def on_save(self):
        if self.Parameters.UseLastBinary:
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
                attrs={'target': 'arcadia/ReportYtDirStats/bin', 'released': self.Parameters.ReleaseType or 'stable'},
                state=ctr.State.READY,
                owner="YASAP"
            ).first().id
        else:
            self.Requirements.tasks_resource = self.Parameters.custom_tasks_archive_resource

    def _publish(self, sfclient, data, path, title):
        from statface_client import StatfaceReportConfig
        from library.python import resource
        report = sfclient.get_report(path)
        config = StatfaceReportConfig()
        config.from_dict(yaml.safe_load(resource.find('sandbox/projects/collections/ReportYtDirStats/publish.yaml')))
        config.title = title
        report.upload_config(config)
        report.upload_data(scale='hourly', data=data)

    @staticmethod
    def make_signal_name(cluster, path):
        path = yt_path_to_yasm(path)
        return 'collections_report_lag_{}_{}'.format(
            cluster, path,
        )


__TASK__ = ReportYtDirStats
