import logging
import subprocess
from datetime import datetime

from sandbox import sdk2
from sandbox.projects.common import utils
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2 import parameters

from sandbox.projects.collections.mixins import YasmReportable
from sandbox.projects.collections.resources import CollectionsRtmrViewsBinary


class CollectionsRtmrViews(YasmReportable, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yandex-yt'),
        )
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('YT parameters'):
            yt_proxy = parameters.String(
                'YT proxy',
                required=True
            )
            yt_token_vault = sdk2.parameters.String(
                'YT token vault',
                required=True,
                default='yt_token'
            )
            tables_to_save = sdk2.parameters.Integer(
                'How many tables to save',
                required=False,
                default=366
            )
        with sdk2.parameters.Group('Input tables data'):
            logs_dir = sdk2.parameters.String(
                'Logs directory',
                required=True
            )
            latest_processed_logs_path = sdk2.parameters.String(
                'Path of the file containing latest processed logs',
                required=True
            )
            states_dir = sdk2.parameters.String(
                'RTMR views states storage directory',
                required=True
            )
            soft_link_name = sdk2.parameters.String(
                'Name of the link containing latest state',
                required=True,
                default='latest'
            )
        with sdk2.parameters.Group('Monitoring'):
            monitoring_server_host = sdk2.parameters.String(
                'Monitoring server',
                default='monit.n.yandex-team.ru',
            )
            monitoring_timelag_signal_id = sdk2.parameters.String(
                'Monitoring timelag signal ID'
            )

    def on_save(self):
        self.Parameters.latest_processed_logs_path = self.Parameters.latest_processed_logs_path.rstrip('/')
        self.Parameters.logs_dir = self.Parameters.logs_dir.rstrip('/')
        self.Parameters.states_dir = self.Parameters.states_dir.rstrip('/')

    def on_execute(self):
        tool_id = utils.get_and_check_last_released_resource_id(CollectionsRtmrViewsBinary)
        resource = sdk2.Resource[tool_id]
        tool_path = str(sdk2.ResourceData(resource).path)

        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_vault)
        from yt.wrapper import YtClient, ypath_join

        yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)

        output_table_name = datetime.now().strftime('%Y-%m-%d')

        args = [
            tool_path,
            '--yt-proxy', self.Parameters.yt_proxy,
            '--input-table-path', self.Parameters.states_dir,
            '--input-table-name', self.Parameters.soft_link_name,
            '--output-table-path', self.Parameters.states_dir,
            '--output-table-name', output_table_name,
            '--logs-path', self.Parameters.logs_dir
        ]

        file = yt_client.read_file(self.Parameters.latest_processed_logs_path)
        latest_processed_logs = str(file.read())

        logs_tables = yt_client.list(self.Parameters.logs_dir, absolute=False, sort=True)

        needed_logs = []
        for logs_table in reversed(logs_tables):
            if str(logs_table) > latest_processed_logs:
                needed_logs.append(logs_table)
            else:
                break
        if len(needed_logs) == 0:
            raise Exception("No logs can be processed!")

        args += ["--logs-date"]
        args += needed_logs

        self._process(args, log_prefix='views_builder', env={'YT_TOKEN': yt_token})

        if getattr(self.Parameters, "tables_to_save"):
            self._clean_old_tables(yt_client)

        yt_client.link(ypath_join(self.Parameters.states_dir, output_table_name), ypath_join(self.Parameters.states_dir, self.Parameters.soft_link_name), force=True)
        yt_client.write_file(self.Parameters.latest_processed_logs_path, needed_logs[0])
        if self.Parameters.monitoring_timelag_signal_id:
            self._report_lag(self.Parameters.monitoring_timelag_signal_id)

    def _process(self, cmd_args, log_prefix, **kwargs):
        with sdk2.helpers.ProcessLog(self, logging.getLogger(log_prefix)) as pl:
            process = subprocess.Popen(
                cmd_args,
                stdout=pl.stdout,
                stderr=pl.stderr,
                **kwargs
            )
            process.wait()
            if process.returncode:
                raise Exception('Process was finished unsuccessfully')

    def _clean_old_tables(self, yt_client):
        from yt.wrapper import ypath_join
        lst = yt_client.list(self.Parameters.states_dir, absolute=True, sort=True)
        soft_link = ypath_join(self.Parameters.states_dir, self.Parameters.soft_link_name)
        states = [table for table in lst if table != soft_link and yt_client.get(table + "/@type") == "table"]
        states_to_remove = states[:-self.Parameters.tables_to_save]
        for state in states_to_remove:
            yt_client.remove(state)

