# -*- encoding: utf-8 -*-

import json
import logging
import subprocess
import sys
import urllib
from datetime import datetime
from tempfile import NamedTemporaryFile
from time import sleep

import requests
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import utils
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2 import parameters

from sandbox.projects.collections.mixins import YasmReportable
from sandbox.projects.collections.resources import CollectionsSaasSnippetsBinary


class _GeneralParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group('YT parameters'):
        yt_proxy = parameters.String(
            'YT proxy',
            required=True
        )
        yt_token_vault = sdk2.parameters.String(
            'YT token vault',
            required=True,
            default='yt_token'
        )

    with sdk2.parameters.Group('Mongo dump options'):
        dump_path = sdk2.parameters.String(
            'Mongo dump path',
            required=False
        )
        dump_state = sdk2.parameters.String(
            'Mongo dump state',
            required=False
        )
        boards_table = sdk2.parameters.String(
            'Boards table name',
            required=False
        )
        cards_table = sdk2.parameters.String(
            'Cards table name',
            required=False
        )
        users_table = sdk2.parameters.String(
            'Users table name',
            required=False
        )
    filter_bad_content = sdk2.parameters.Bool(
        'Filter bad content',
        default=False
    )


class _GeneralRequirements(sdk2.Task.Requirements):
    environments = (
        PipEnvironment('yandex-yt'),
        PipEnvironment('yandex-yt-yson-bindings-skynet'),
    )
    cores = 1
    ram = 2048

    class Caches(sdk2.Requirements.Caches):
        pass


class CollectionsSaasSnippetsChild(sdk2.Task):
    class Requirements(_GeneralRequirements):
        pass

    class Parameters(_GeneralParameters):
        mode = sdk2.parameters.String(
            'Type of snippets: "cards" or "boards"',
            required=True
        )

        output_table = parameters.String(
            'Output table',
            required=True
        )
        additional_args = sdk2.parameters.List(
            'Additional command line arguments',
            default=[]
        )


    def on_execute(self):
        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_vault)
        tool_id = utils.get_and_check_last_released_resource_id(CollectionsSaasSnippetsBinary)
        tool_path = str(sdk2.ResourceData(sdk2.Resource[tool_id]).path)

        args = [
            tool_path,
            self.Parameters.mode,
            '--yt-proxy', self.Parameters.yt_proxy,
            '--output-table', self.Parameters.output_table,
        ]

        if self.Parameters.filter_bad_content:
            args.append('--filter-bad-content')

        def add_optional_parameter(args_, key, param_name):
            if getattr(self.Parameters, param_name):
                args_ += [key, str(getattr(self.Parameters, param_name))]

        add_optional_parameter(args, '--dump-path', 'dump_path')
        add_optional_parameter(args, '--dump-state', 'dump_state')
        add_optional_parameter(args, '--board-table', 'boards_table')
        add_optional_parameter(args, '--card-table', 'cards_table')
        add_optional_parameter(args, '--user-table', 'users_table')
        args.extend(self.Parameters.additional_args)

        self._process(args, log_prefix='snippet_builder', env={'YT_TOKEN': yt_token})

    def _process(self, cmd_args, log_prefix, **kwargs):
        with sdk2.helpers.ProcessLog(self, logging.getLogger(log_prefix)) as pl:
            process = subprocess.Popen(
                cmd_args,
                stdout=pl.stdout,
                stderr=pl.stderr,
                **kwargs
            )
            process.wait()
            if process.returncode:
                raise Exception('Process was finished unsuccessfully')


class CollectionsSaasSnippets(sdk2.Task, YasmReportable):
    class Requirements(_GeneralRequirements):
        pass

    class Parameters(_GeneralParameters):
        with sdk2.parameters.CheckGroup('Type of snippets: "cards" or "boards"', required=True) as mode:
            mode.values.cards = "Cards"
            mode.values.boards = "Boards"

        output_dir = parameters.String(
            'Output directory',
            required=True
        )

        tables_to_save = sdk2.parameters.Integer(
            'How many tables to save',
            required=True
        )

        with sdk2.parameters.Group('Saas'):
            saas_url = sdk2.parameters.String(
                'SaaS uploading url',
                required=True,
                default='http://fastsnips.ferryman.n.yandex-team.ru'
            )
            saas_namespace = sdk2.parameters.String(
                'SaaS namespace',
                required=True,
                default='collections:docid_setprops'
            )

        additional_args_per_mode = sdk2.parameters.JSON(
            'Additional command line arguments for each of selected mode like "mode": ["arg1", "arg2", ...]',
            default={}
        )
        monitoring_server_host = sdk2.parameters.String(
            'Monitoring server',
            default='monit.n.yandex-team.ru',
        )
        monitoring_signal_id = sdk2.parameters.String(
            'Monitoring signal ID',
            default='collections-saas-snippets-count-rows',
        )
        monitoring_timelag_signal_id = sdk2.parameters.String(
            'Monitoring timelag signal ID',
            default='collections-saas-snippets-released',
        )

    def on_execute(self):
        from yt.wrapper import YtClient, ypath_join
        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_vault)
        yt_client = YtClient(
            proxy=self.Parameters.yt_proxy,
            token=yt_token,
            config={
                'pickling': {
                    'python_binary': '/skynet/python/bin/python'
                }
            })

        if len(self.Parameters.mode) == 0:
            raise TaskFailure('Mode is required')

        if not self.Context.tasks:
            general_parameters = {
                param: getattr(self.Parameters, param)
                for param in [
                    'yt_proxy', 'dump_path', 'dump_state', 'boards_table',
                    'cards_table', 'users_table', 'yt_token_vault', 'filter_bad_content'
                ]
            }

            self.Context.tasks = {}

            for mode in ['boards', 'cards']:
                if mode in self.Parameters.mode:
                    output_table = self._get_temp_path(yt_client, 'sass_snippets.{}.'.format(mode))

                    task = CollectionsSaasSnippetsChild(
                        self,
                        description="Child of {} for {}".format(self.id, mode),
                        owner=self.owner,
                        mode=mode,
                        output_table=output_table,
                        additional_args=self.Parameters.additional_args_per_mode.get(mode, []),
                        **general_parameters
                    )

                    task.enqueue()
                    self.Context.tasks[mode] = task.id

            self.Context.save()

            raise sdk2.WaitTask(self.Context.tasks.values(), (ctt.Status.Group.FINISH + ctt.Status.Group.BREAK))

        tasks = [sdk2.Task[task_id] for mode, task_id in self.Context.tasks.items()]

        for task in tasks:
            if task.status != ctt.Status.SUCCESS:
                raise TaskFailure('Some subtasks failed')

        output_table = ypath_join(self.Parameters.output_dir, datetime.now().strftime('%Y-%m-%dT%H:%M:%S'))

        self._merge_tables(yt_token, [task.Parameters.output_table for task in tasks], output_table)
        yt_client.run_sort(
            output_table,
            output_table,
            sort_by='Subkey_Url'
        )
        count_rows = int(yt_client.get(output_table + "/@row_count"))
        self._yasm_report(
            args=[
                "--id", self.Parameters.monitoring_signal_id,
                "--value", str(count_rows),
                "--policy", "abs"
            ]
        )
        self._report_lag(self.Parameters.monitoring_timelag_signal_id)

        self._upload_to_saas(
            self.Parameters.yt_proxy,
            output_table,
            wait=True
        )
        self._clean_old_tables(yt_client)

    def _get_temp_path(self, yt_client, prefix):
        path = yt_client.config["remote_temp_tables_directory"]
        yt_client.mkdir(path, recursive=True)

        if not path.endswith("/"):
            path = path + "/"

        name = yt_client.find_free_subpath(path + prefix)

        return name

    def _merge_tables(self, yt_token, input_tables, output_table):
        with NamedTemporaryFile(delete=False) as token_file:
            token_file.write(yt_token)
            token_file_name = token_file.name
        import sandbox.projects.collections.SaasSnippets.merge as merge

        if not isinstance(input_tables, (list, tuple)):
            input_tables = [input_tables]

        cmd = [sys.executable, merge.__file__,
               '--token-file', token_file_name,
               '--proxy', self.Parameters.yt_proxy,
               '--output-path', output_table,
               '--input-path'
        ]
        cmd.extend(input_tables)

        process.run_process(cmd=cmd, log_prefix='merge.py', wait=True, check=True)

    def _upload_to_saas(self, proxy, table, wait=False):
        api_data = {'tables': json.dumps([{
            'Cluster': proxy,
            'Path': table,
            'Namespace': self.Parameters.saas_namespace,
            'Timestamp': int((datetime.utcnow() - datetime(1970, 1, 1)).total_seconds() * 1000000)  # microseconds
        }])}
        response = requests.get(
            self.Parameters.saas_url + '/add-full-tables?{api_data}'.format(api_data=urllib.urlencode(api_data)))
        response.raise_for_status()

        if not wait:
            return

        batch = response.json()['batch']
        status = None
        while status != 'final':
            try:
                response = requests.get(
                    self.Parameters.saas_url + '/get-batch-status?batch={batch}'.format(batch=batch))
                response.raise_for_status()
                status = response.json()['status']
                logging.info('Batch {} status {}'.format(batch, status))
            except requests.exceptions.ConnectionError:
                pass
            if status == 'error':
                msg = "Couldn't export table to SaaSKV. Batch: {}. Details:\n{}".format(batch, response.json()['error'])
                raise Exception(msg)
            sleep(60)

    def _clean_old_tables(self, yt_client):
        from yt.wrapper import ypath_join
        states = yt_client.list(self.Parameters.output_dir, absolute=False)
        states_to_remove = sorted(states, reverse=True)[int(self.Parameters.tables_to_save):]
        for state in states_to_remove:
            yt_client.remove(ypath_join(self.Parameters.output_dir, state), recursive=True)


__TASK__ = CollectionsSaasSnippets
