# -*- coding: utf-8 -*-

import logging
import requests
from sandbox import sdk2
from sandbox.sandboxsdk.process import run_process
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.vcs.git import Git

from sandbox.projects.sandbox_ci import parameters
from sandbox.projects.sandbox_ci.utils.github import GitHubStatus

logger = logging.getLogger('pdb-test')


class SandboxCiCollectionsBackendTest(sdk2.Task):
    privileged = True

    class Parameters(sdk2.Parameters):
        _container = sdk2.parameters.Container("container", default_value=535662790, required=True)

        github_repo_parameters = parameters.ProjectGitHubRepositoryParameters

        with sdk2.parameters.Group('Source control') as source_block:
            project_git_base_ref = parameters.project_git_base_ref()
            project_git_base_commit = parameters.project_git_base_commit()
            project_git_merge_ref = parameters.project_git_merge_ref()
            project_git_merge_commit = parameters.project_git_merge_commit()

    @property
    def ref(self):
        if self.Parameters.project_git_merge_commit:
            return self.Parameters.project_git_merge_ref

        if self.Parameters.project_git_base_commit:
            return self.Parameters.project_git_base_ref

        return None

    @property
    def hash(self):
        if self.Parameters.project_git_merge_commit:
            return self.Parameters.project_git_merge_commit

        if self.Parameters.project_git_base_commit:
            return self.Parameters.project_git_base_commit

        return None

    def report_status(self, context, state, description):
        headers = {
            'Authorization': 'token {}'.format(sdk2.Vault.data('YASAP', 'github-pdb-backend-token'))
        }

        logging.info('Making github status {}'.format(state))

        response = requests.post(
            'https://api.github.yandex-team.ru/repos/{owner}/{repo}/statuses/{ref}'.format(
                owner=self.Parameters.project_github_owner,
                repo=self.Parameters.project_github_repo,
                ref=self.hash
            ),
            headers=headers,
            json={
                'context': context,
                'target_url': 'https://sandbox.yandex-team.ru/task/{}/view'.format(str(self.id)),
                'description': description,
                'state': state
            }
        )

        logging.info(response.text)

    def start_daemon(self):
        with ProcessLog(self, logger) as pl:
            run_process(
                'systemctl status docker.socket'.split(),
                wait=True,
                stdout=pl.stdout,
                stderr=pl.stderr
            )
            logging.info('Start daemon')
            run_process(
                'systemctl docker start'.split(),
                wait=True,
                stdout=pl.stdout,
                stderr=pl.stderr
            )

    def stop_daemon(self):
        with ProcessLog(self, logger) as pl:
            logging.info('Stop daemon')
            run_process(
                'systemctl docker stop'.split(),
                wait=True,
                stdout=pl.stdout,
                stderr=pl.stderr
            )

    def register_user(self):
        with ProcessLog(self, logger=logger) as pl:
            logging.info('Registering docker user')
            run_process(
                'docker login -u {login} -p {token_body} registry.yandex.net'.format(login='asntr', token_body=sdk2.Vault.data('YASAP', 'pdb-docker-token')).split(),
                wait=True,
                stderr=pl.stderr,
                stdout=pl.stdout
            )

    def prepare_docker(self):
        with ProcessLog(self, logger=logger) as pl:
            logging.info('Getting containers')
            run_process(
                'docker build -f Dockerfile -t pdb_backend:ci_testing .'.split(),
                wait=True,
                stderr=pl.stderr,
                stdout=pl.stdout
            )
            logging.info('pdb_backend:ci_testing is fetched')
            run_process(
                'docker build -f Dockerfile.test -t pdb_test:latest .'.split(),
                wait=True,
                stderr=pl.stderr,
                stdout=pl.stdout
            )
            logging.info('pdb_test:latest is fetched')
            run_process(
                'mkdir podb_settings && cd podb_settings && ln -s ../tests/data/local_settings.py local_settings.py && touch __init__.py && cd -'.split(),
                wait=True,
                stderr=pl.stderr,
                stdout=pl.stdout
            )
            logging.info('podb_settings are created')

    def on_failure(self, prev_status):
        self.report_status(state=GitHubStatus.FAILURE, description='Task failed', context='Sandbox CI')
        super(SandboxCiCollectionsBackendTest, self).on_failure(prev_status)

    def on_finish(self, prev_status, status):
        if status == 'EXCEPTION':
            self.report_status(state=GitHubStatus.ERROR, description='Task finished with exception', context='Sandbox CI')
        super(SandboxCiCollectionsBackendTest, self).on_finish(prev_status, status)

    def on_execute(self):
        self.report_status(state=GitHubStatus.PENDING, description='Task is running', context='Sandbox CI')
        project_git_url = 'https://github.yandex-team.ru/{owner}/{repo}.git'.format(
            owner=self.Parameters.project_github_owner,
            repo=self.Parameters.project_github_repo,
        )
        git = Git(project_git_url)
        git.clone('.', self.ref or self.hash)
        self.start_daemon()
        self.register_user()
        self.prepare_docker()
        build_pdb_backend = run_process('docker-compose run --rm instant_tests'.split())
        tests_result = build_pdb_backend.wait()
        logging.info('Result: ', tests_result)
        if tests_result == 0:
            self.report_status(state=GitHubStatus.SUCCESS, description='OK!', context='Sandbox CI')
        else:
            self.report_status(state=GitHubStatus.FAILURE, description='Something went wrong, go and see testing output', context='Sandbox CI')
        self.stop_daemon()
