from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment


YQL_TEMPLATE = """
PRAGMA yt.InferSchema;
PRAGMA yt.Pool = "collections_production";

DECLARE $full_table AS String;
DECLARE $additional_table AS String;
DECLARE $output AS String;
DECLARE $table_size AS Int64;

$current_table = (
    SELECT full_table.entity AS entity,
           full_table.complaint_reason AS complaint_reason,
           full_table.url AS url,
           full_table.id AS id,
           full_table.model as model,
           full_table.puid as puid
    FROM $full_table AS full_table
    LEFT ONLY JOIN $additional_table AS additional
    ON full_table.id == additional.id AND full_table.url == additional.url
    LIMIT $table_size
);

INSERT INTO $output WITH TRUNCATE
SELECT *
FROM $current_table;

INSERT INTO $additional_table
SELECT *
FROM $current_table
"""


class CollectionsSendCardsToCleanWeb(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            PipEnvironment('yql'),
            PipEnvironment('yandex-yt'),
            PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    class Parameters(sdk2.Task.Parameters):
        input_table = sdk2.parameters.String('Table with full data')
        additional_table = sdk2.parameters.String('Table with processed data')
        result_dir = sdk2.parameters.String('YT directory for result table')

        yql_token_secret = sdk2.parameters.String('YQL token secret')
        yt_token_secret = sdk2.parameters.String('YT token secret')
        yt_proxy = sdk2.parameters.String('YT proxy (cluster)')

        table_size = sdk2.parameters.Integer('Batch size', default=3375)

    def send_table_to_clean_web(self):
        if self.Context.table_for_clean_web:
            return self.Context.table_for_clean_web

        from yt.wrapper import YtClient, ypath_join
        yt_token = sdk2.Vault.data(self.owner, self.Parameters.yt_token_secret)
        yt_client = YtClient(proxy=self.Parameters.yt_proxy, token=yt_token)

        now = datetime.now()
        output_table = ypath_join(self.Parameters.result_dir, now.isoformat())
        yt_client.create(
            type='table',
            path=output_table,
            attributes={
                'expiration_time': (now + timedelta(days=1)).isoformat(),
            }
        )

        from yql.api.v1.client import YqlClient
        from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder
        yql_client = YqlClient(
            db=self.Parameters.yt_proxy,
            token=sdk2.Vault.data(self.owner, self.Parameters.yql_token_secret)
        )

        parameters = {
            '$full_table': ValueBuilder.make_string(self.Parameters.input_table),
            '$additional_table': ValueBuilder.make_string(self.Parameters.additional_table),
            '$table_size': ValueBuilder.make_int64(self.Parameters.table_size),
            '$output': ValueBuilder.make_string(output_table),
        }
        request = yql_client.query(YQL_TEMPLATE, syntax_version=1).run(parameters=ValueBuilder.build_json_map(parameters))
        errors = request.get_results(wait=True).errors
        if errors:
            raise RuntimeError([error.message for error in errors])

        self.Context.table_for_clean_web = output_table
        self.Context.save()
        return self.Context.table_for_clean_web

    def on_execute(self):
        self.send_table_to_clean_web()
